#ifndef UTIL___UTIL_MISC__HPP
#define UTIL___UTIL_MISC__HPP

/*  $Id: util_misc.hpp 578926 2019-01-28 18:17:50Z ivanov $
 * ===========================================================================
 *
 *                            PUBLIC DOMAIN NOTICE
 *               National Center for Biotechnology Information
 *
 *  This software/database is a "United States Government Work" under the
 *  terms of the United States Copyright Act.  It was written as part of
 *  the author's official duties as a United States Government employee and
 *  thus cannot be copyrighted.  This software/database is freely available
 *  to the public for use. The National Library of Medicine and the U.S.
 *  Government have not placed any restriction on its use or reproduction.
 *
 *  Although all reasonable efforts have been taken to ensure the accuracy
 *  and reliability of the software and data, the NLM and the U.S.
 *  Government do not and cannot warrant the performance or results that
 *  may be obtained by using this software or data. The NLM and the U.S.
 *  Government disclaim all warranties, express or implied, including
 *  warranties of performance, merchantability or fitness for any particular
 *  purpose.
 *
 *  Please cite the author in any work or product based on this material.
 *
 * ===========================================================================
 *
 * Author:  Sergey Satskiy
 *
 */

#include <corelib/ncbifile.hpp>


BEGIN_NCBI_SCOPE


/// Get a password without echoing the user input
///
/// @param
///   prompt Printed before the user input
/// @return
///   The user input till the first carriage return
/// @warning
///   UNIX and Windows implementations differ at processing Ctrl+C:
///   - UNIX (except FreeBSD & IntelMAC): The usual SIGINT is
///          generated by the system
///   - UNIX (FreeBSD & IntelMAC): Ignored
///   - Windows: an exception is thrown
/// @exception
///   The function throws CGetPasswordFromConsoleException exception if
///   anything goes wrong
NCBI_XUTIL_EXPORT
extern string g_GetPasswordFromConsole(const string& prompt);



/// Exception class for g_GetPasswordFromConsole
class NCBI_XUTIL_EXPORT  CGetPasswordFromConsoleException : public CException
{
public:
    enum EErrCode {
        eGetPassError,          ///< UNIX specific: error getting password
        eKeyboardInterrupt      ///< WIN specific:  Ctrl+C has been sent
    };

    /// Translate from the error code value to its string representation
    virtual const char* GetErrCodeString(void) const override;

    NCBI_EXCEPTION_DEFAULT(CGetPasswordFromConsoleException, CException);
};


/// Look for an NCBI application data file or directory of the given name and
/// type; in general, searches directories listed in the registry entry
/// [NCBI] DataPath (which the environment variable NCBI_DATA_PATH may
/// override) followed by [NCBI] Data (or NCBI_DATA).  (Passing an absolute
/// path bypasses this step.)  NCBI_DATA_PATH and [NCBI] DataPath may both
/// list multiple directories, delimited as per PATH (by ';' on Windows, ':'
/// on Unix).
/// @param name The name of the file or directory to look for.
/// @param type The type to look for (following links unless explicitly
///   looking for one); CDirEntry::eUnknown matches anything.
/// @return
///   A full path if found, the empty string otherwise.
NCBI_XUTIL_EXPORT
extern string g_FindDataFile(const CTempString& name,
                             CDirEntry::EType type = CDirEntry::eFile);

/// Ignore (or stop ignoring, depending on do_ignore) NCBI application
/// data files matching the given pattern to force the use of built-in
/// fallbacks; mainly of interest to unit tests.
NCBI_XUTIL_EXPORT
extern void g_IgnoreDataFile(const string& pattern, bool do_ignore = true);

/// Check whether the given file (a full path, as returned by
/// g_FindDataFile) is older than a built-in version containing
/// the specified "$Id: ..." line.
NCBI_XUTIL_EXPORT
extern bool g_IsDataFileOld(const CTempString& path,
                            const CTempString& id_line);

END_NCBI_SCOPE


#endif  /* UTIL___UTIL_MISC__HPP */
