/* This file is part of the 'atomes' software

'atomes' is free software: you can redistribute it and/or modify it under the terms
of the GNU Affero General Public License as published by the Free Software Foundation,
either version 3 of the License, or (at your option) any later version.

'atomes' is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU Affero General Public License along with 'atomes'.
If not, see <https://www.gnu.org/licenses/>

Copyright (C) 2022-2025 by CNRS and University of Strasbourg */

/*!
* @file compass.c
* @short Compass force field, from 'Compass.frc'
* @author Sébastien Le Roux <sebastien.leroux@ipcms.unistra.fr>
*/

/*
* This file: 'compass.c'
*
* Contains:
*

  Compass force field, data from taken from:

       - the file: 'Compass.frc'

  This file contains several tables:

    Atoms           : Compass_atoms[45][5]
    Equivalence     : Compass_equivalence[46][7]
    Quartic bonds   : Compass_bonds[54][7]
    Quartic angles  : Compass_angles[94][8]
    Torsions        : Compass_torsions[95][11]
    Inversions      : Compass_inversions[22][7]
    Non bonded      : Compass_vdw[46][4]
    Bond increments : Compass_bond_increments[55][5]

*/

#include "global.h"

int Compass_objects[14] = {45, 0, 46, 0, 54, 0, 0, 94, 0, 95, 22, 0, 46, 55};
int Compass_dim[14] = {5, 0, 7, 0, 7, 0, 0, 8, 0, 11, 7, 0, 4, 5};

/*
 Atoms
  0= Element, 1= Mass, 2= Key, 3= Description, 4= Compass info
*/
char * Compass_atoms[45][5]= {{"Ar", "39.94400", "ar", " ", "Ver. 1.0, Ref. 5"},
                            {"C", "12.01115", "c3a", "carbon ", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "c1o", "in CO", "Ver. 1.0, Ref. 5"},
                            {"C", "12.01115", "c2=", "in CO2 and CS2", "Ver. 1.0, Ref. 5"},
                            {"C", "12.01115", "c3'", "carbon [one polar substituent]", "Ver. 1.0, Ref. 7"},
                            {"C", "12.01115", "c4", "sp3 carbon", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "c41o", "sp3, in methanol", "Ver. 1.0, Ref. 8"},
                            {"C", "12.01115", "c43o", "sp3 in secondary alcohols", "Ver. 1.0, Ref. 8"},
                            {"C", "12.01115", "c43", "carbon with three heavy atoms attached", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "c44", "carbon with four heavy atoms attached", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "c4o", "carbon ", "Ver. 1.0, Ref. 3"},
                            {"C", "12.01115", "c4z", "sp3, bonded to -N3", "Ver. 1.0, Ref. 9"},
                            {"H", "1.00797", "h1", "hydrogen ", "Ver. 1.0, Ref. 1"},
                            {"H", "1.00797", "h1h", "in H2", "Ver. 1.0, Ref. 5"},
                            {"H", "1.00797", "h1o", "polar hydrogen, bonded to O,F", "Ver. 1.0, Ref. 3"},
                            {"He", "4.00300", "he", " ", "Ver. 1.0, Ref. 5"},
                            {"Kr", "83.80000", "kr", " ", "Ver. 1.0, Ref. 5"},
                            {"N", "14.00670", "n1n", "in N2", "Ver. 1.0, Ref. 5"},
                            {"N", "14.00670", "n1o", "in NO", "Ver. 1.0, Ref. 5"},
                            {"N", "14.00670", "n1z", "terminal atom in -N3", "Ver. 1.0, Ref. 5"},
                            {"N", "14.00670", "n2=", " ", "Ver. 1.0, Ref. 4"},
                            {"N", "14.00670", "n2o", "in NO2", "Ver. 1.0, Ref. 5"},
                            {"N", "14.00670", "n2t", "central atom in -N3", "Ver. 1.0, Ref. 9"},
                            {"N", "14.00670", "n2z", "first atom in -N3", "Ver. 1.0, Ref. 9"},
                            {"N", "14.00670", "n3m", "nitrogen in amides without hydrogen", "Ver. 1.0, Ref. 7"},
                            {"N", "14.00670", "n3o", "in nitro group", "Ver. 1.0, Ref. 6"},
                            {"Ne", "20.18300", "ne", " ", "Ver. 1.0, Ref. 5"},
                            {"O", "15.99940", "o1=", "in NO2 and SO2 [and carbonyl]", "Ver. 1.0, Ref. 5"},
                            {"O", "15.99940", "o1=*", "in CO2", "Ver. 1.0, Ref. 5"},
                            {"O", "15.99940", "o12", "in nitro group (-NO2)", "Ver. 1.0, Ref. 6"},
                            {"O", "15.99940", "o1c", "in CO", "Ver. 1.0, Ref. 5"},
                            {"O", "15.99940", "o1n", "in NO", "Ver. 1.0, Ref. 5"},
                            {"O", "15.99940", "o1o", "in O2", "Ver. 1.0, Ref. 5"},
                            {"O", "15.99940", "o2", "oxygen with two bonds attached", "Ver. 1.0, Ref. 2"},
                            {"O", "15.99940", "o2e", "oxygen ", "Ver. 1.0, Ref. 3"},
                            {"O", "15.99940", "o2h", "oxygen ", "Ver. 1.0, Ref. 3"},
                            {"O", "15.99940", "o2n", "in nitrates", "Ver. 1.0, Ref. 6"},
                            {"O", "15.99940", "o2s", "oxygen ", "Ver. 1.0, Ref. 7"},
                            {"O", "15.99940", "o2z", "in siloxanes and zeolites", "Ver. 1.0, Ref. 2"},
                            {"P", "30.97380", "p4=", " ", "Ver. 1.0, Ref. 4"},
                            {"S", "32.06400", "s1=", "in CS2", "Ver. 1.0, Ref. 5"},
                            {"S", "32.06400", "s2=", "in SO2", "Ver. 1.0, Ref. 5"},
                            {"Si", "28.08600", "si4", "silicon with four bonds attached", "Ver. 1.0, Ref. 2"},
                            {"Si", "28.08600", "si4c", "subset of si4, non-hydrogen atom attached [siloxanes]", "Ver. 1.0, Ref. 2"},
                            {"Xe", "131.30000", "xe", " ", "Ver. 1.0, Ref. 5"}};

/*
 Equivalence table for key symbols Compass with:
   0 = Key in atom list
   1 = Non-bonded
   2 = Bonds
   3 = Angles
   4 = Torsion
   5 = Inversion
   6 = Compass info
*/
char * Compass_equivalence[46][7]= {{"ar", "ar", "ar", "ar", "ar", "ar", "Ver. 1.0, Ref. 5"},
                                  {"c1o", "c1o", "c1o", "c1o", "c1o", "c1o", "Ver. 1.0, Ref. 5"},
                                  {"c2=", "c2=", "c2=", "c2=", "c2=", "c2=", "Ver. 1.0, Ref. 5"},
                                  {"c3a", "c3a", "c3a", "c3a", "c3a", "c3a", "Ver. 1.0, Ref. 1"},
                                  {"c3'", "c3'", "c3'", "c3'", "c3'", "c3'", "Ver. 1.0, Ref. 7"},
                                  {"c4", "c4", "c4", "c4", "c4", "c4", "Ver. 1.0, Ref. 1"},
                                  {"c41o", "c41o", "c4", "c4", "c4", "c4", "Ver. 1.0, Ref. 8"},
                                  {"c43o", "c43o", "c4", "c4", "c4", "c4", "Ver. 1.0, Ref. 8"},
                                  {"c43", "c43", "c4", "c4", "c4", "c4", "Ver. 1.0, Ref. 1"},
                                  {"c44", "c44", "c4", "c4", "c4", "c4", "Ver. 1.0, Ref. 1"},
                                  {"c4o", "c4o", "c4", "c4", "c4", "c4", "Ver. 1.0, Ref. 3"},
                                  {"c4z", "c4z", "c4", "c4", "c4", "c4", "Ver. 1.0, Ref. 9"},
                                  {"h1", "h1", "h1", "h1", "h1", "h1", "Ver. 1.0, Ref. 1"},
                                  {"h1h", "h1h", "h1h", "h1", "h1", "h1", "Ver. 1.0, Ref. 5"},
                                  {"h1o", "h1o", "h1", "h1", "h1", "h1", "Ver. 1.0, Ref. 3"},
                                  {"he", "he", "he", "he", "he", "he", "Ver. 1.0, Ref. 5"},
                                  {"kr", "kr", "kr", "kr", "kr", "kr", "Ver. 1.0, Ref. 5"},
                                  {"n1n", "n1n", "n1n", "n1n", "n1n", "n1n", "Ver. 1.0, Ref. 5"},
                                  {"n1o", "n1o", "n1o", "n1o", "n1o", "n1o", "Ver. 1.0, Ref. 5"},
                                  {"n1t", "n1t", "n1t", "n1t", "n1t", "n1t", "Ver. 1.2, Ref. 1"},
                                  {"n1z", "n1z", "n1t", "n1t", "n1t", "n1t", "Ver. 1.0, Ref. 9"},
                                  {"n2=", "n2=", "n2=", "n2=", "n2=", "n2=", "Ver. 1.0, Ref. 4"},
                                  {"n2o", "n2o", "n2o", "n2o", "n2o", "n2o", "Ver. 1.0, Ref. 5"},
                                  {"n2t", "n2t", "n2t", "n2t", "n2t", "n2t", "Ver. 1.0, Ref. 9"},
                                  {"n2z", "n2z", "n2z", "n2z", "n2z", "n2z", "Ver. 1.0, Ref. 9"},
                                  {"n3m", "n3m", "n3m", "n3m", "n3m", "n3m", "Ver. 1.0, Ref. 7"},
                                  {"n3o", "n3o", "n3o", "n3o", "n3o", "n3o", "Ver. 1.0, Ref. 6"},
                                  {"ne", "ne", "ne", "ne", "ne", "ne", "Ver. 1.0, Ref. 5"},
                                  {"o1=", "o1=", "o1=", "o1=", "o1=", "o1=", "Ver. 1.0, Ref. 5"},
                                  {"o1=*", "o1=*", "o1=", "o1=", "o1=", "o1=", "Ver. 1.0, Ref. 5"},
                                  {"o12", "o12", "o1=", "o1=", "o1=", "o1=", "Ver. 1.0, Ref. 6"},
                                  {"o1c", "o1c", "o1c", "o1c", "o1c", "o1c", "Ver. 1.0, Ref. 5"},
                                  {"o1n", "o1n", "o1n", "o1n", "o1n", "o1n", "Ver. 1.0, Ref. 5"},
                                  {"o1o", "o1o", "o1o", "o1o", "o1o", "o1o", "Ver. 1.0, Ref. 5"},
                                  {"o2", "o2", "o2", "o2", "o2", "o2", "Ver. 1.0, Ref. 2"},
                                  {"o2h", "o2h", "o2h", "o2", "o2", "o2", "Ver. 1.0, Ref. 3"},
                                  {"o2e", "o2e", "o2e", "o2", "o2", "o2", "Ver. 1.0, Ref. 3"},
                                  {"o2n", "o2n", "o2n", "o2n", "o2", "o2", "Ver. 1.0, Ref. 6"},
                                  {"o2s", "o2s", "o2e", "o2", "o2", "o2", "Ver. 1.0, Ref. 7"},
                                  {"o2z", "o2z", "o2z", "o2z", "o2z", "o2z", "Ver. 1.0, Ref. 2"},
                                  {"p4=", "p4=", "p4=", "p4=", "p4=", "p4=", "Ver. 1.0, Ref. 4"},
                                  {"s1=", "s1=", "s1=", "s1=", "s1=", "s1=", "Ver. 1.0, Ref. 5"},
                                  {"s2=", "s2=", "s2=", "s2=", "s2=", "s2=", "Ver. 1.0, Ref. 5"},
                                  {"si4", "si4", "si4", "si4", "si4", "si4", "Ver. 1.0, Ref. 2"},
                                  {"si4c", "si4c", "si4", "si4", "si4", "si4", "Ver. 1.0, Ref. 2"},
                                  {"xe", "xe", "xe", "xe", "xe", "xe", "Ver. 1.0, Ref. 5"}};

/*
 Quartic bonds
  0= Key_a, 1= Key_b, 2= R0 (A), 3= K2 (kcal mol-1), 4= K3 (kcal mol-1), 5= K4 (kcal mol-1), 6= Compass info

  E = K2 * (R - R0)^2  +  K3 * (R - R0)^3  +  K4 * (R - R0)^4
*/
char * Compass_bonds[54][7]= {{"c3a", "c3a", "1.4170", "470.8361", "-627.6179", "1327.6345", "Ver. 1.0, Ref. 1"},
                            {"c3a", "c4", "1.5010", "321.9021", "-521.8208", "572.1628", "Ver. 1.0, Ref. 1"},
                            {"c3a", "h1", "1.0982", "372.8251", "-803.4526", "894.3173", "Ver. 1.0, Ref. 1"},
                            {"c4", "c4", "1.5300", "299.6700", "-501.7700", "679.8100", "Ver. 1.0, Ref. 1"},
                            {"c4", "h1", "1.1010", "345.0000", "-691.8900", "844.6000", "Ver. 1.0, Ref. 1"},
                            {"o2z", "si4", "1.6400", "350.1232", "-517.3424", "673.7067", "Ver. 1.0, Ref. 2"},
                            {"c3a", "o2", "1.3768", "428.8798", "-738.2350", "1114.9655", "Ver. 1.0, Ref. 3"},
                            {"c3a", "o2e", "1.3768", "428.8798", "-738.2351", "1114.9655", "Ver. 1.0, Ref. 3"},
                            {"c3a", "o2h", "1.3768", "428.8798", "-738.2351", "1114.9655", "Ver. 1.0, Ref. 3"},
                            {"c4", "o2e", "1.4200", "400.3954", "-835.1951", "1313.0142", "Ver. 1.0, Ref. 3"},
                            {"c4", "o2h", "1.4200", "400.3954", "-835.1951", "1313.0142", "Ver. 1.0, Ref. 3"},
                            {"h1", "o2h", "0.9494", "540.3633", "-1311.8663", "2132.4446", "Ver. 1.0, Ref. 3"},
                            {"c3a", "n2=", "1.4000", "350.0000", "0.0000", "0.0000", "Ver. 1.0, Ref. 4"},
                            {"c3a", "p4=", "1.7890", "197.7020", "-332.2510", "325.7160", "Ver. 1.0, Ref. 4"},
                            {"c4", "n2=", "1.4740", "337.0600", "-147.3700", "213.6330", "Ver. 1.0, Ref. 4"},
                            {"c4", "p4=", "1.8000", "218.1400", "-329.5110", "290.3490", "Ver. 1.0, Ref. 4"},
                            {"cl1p", "p4=", "2.0000", "158.7770", "-239.1290", "210.0840", "Ver. 1.0, Ref. 4"},
                            {"f1p", "p4=", "1.5650", "340.0000", "-882.3840", "1197.9190", "Ver. 1.0, Ref. 4"},
                            {"h1", "n2=", "1.0310", "540.1120", "-1500.2952", "2431.0080", "Ver. 1.0, Ref. 4"},
                            {"h1", "p4=", "1.4300", "285.2040", "-575.6850", "677.8460", "Ver. 1.0, Ref. 4"},
                            {"n2=", "p4=", "1.5980", "393.0060", "-751.4050", "767.4310", "Ver. 1.0, Ref. 4"},
                            {"n3m", "p4=", "1.6780", "329.0000", "-713.7950", "902.9190", "Ver. 1.0, Ref. 4"},
                            {"o2", "p4=", "1.6000", "333.0980", "-726.6230", "924.6200", "Ver. 1.0, Ref. 4"},
                            {"h1h", "h1h", "0.7412", "414.2185", "-805.6549", "914.1296", "Ver. 1.0, Ref. 5"},
                            {"n1n", "n1n", "1.0977", "1651.3730", "-4069.3178", "5984.9629", "Ver. 1.0, Ref. 5"},
                            {"o1o", "o1o", "1.2074", "846.7150", "-2247.1760", "3478.9900", "Ver. 1.0, Ref. 5"},
                            {"c1o", "o1c", "1.1283", "1368.7676", "-3157.0007", "4247.5298", "Ver. 1.0, Ref. 5"},
                            {"n1o", "o1n", "1.1506", "1147.8362", "-3167.7349", "5099.5811", "Ver. 1.0, Ref. 5"},
                            {"o1=", "s2=", "1.4308", "730.8387", "-1531.7910", "1859.7753", "Ver. 1.0, Ref. 5"},
                            {"c2=", "o1=", "1.1600", "1161.3421", "-2564.5706", "3932.8735", "Ver. 1.0, Ref. 5"},
                            {"n2o", "o1=", "1.1930", "620.0000", "-1808.6018", "3077.5918", "Ver. 1.0, Ref. 5"},
                            {"c2=", "s1=", "1.5540", "559.0065", "-1348.6633", "1248.8604", "Ver. 1.0, Ref. 5"},
                            {"c3a", "n3o", "1.4300", "313.8329", "-568.6087", "600.9597", "Ver. 1.0, Ref. 6"},
                            {"c4", "n3o", "1.4740", "301.6051", "-535.7028", "555.0420", "Ver. 1.0, Ref. 6"},
                            {"c4", "o2n", "1.4350", "400.3954", "-835.1951", "1313.0142", "Ver. 1.0, Ref. 6"},
                            {"h1", "n3o", "1.0400", "439.9346", "-943.7307", "1180.9318", "Ver. 1.0, Ref. 6"},
                            {"n3o", "o1=", "1.2100", "765.0664", "-2070.2830", "2793.3218", "Ver. 1.0, Ref. 6"},
                            {"n3o", "o2n", "1.4020", "300.0000", "-1000.0000", "2000.0000", "Ver. 1.0, Ref. 6"},
                            {"c3'", "o2e", "1.3750", "368.7309", "-832.4784", "1274.0231", "Ver. 1.0, Ref. 7"},
                            {"c3'", "c4", "1.5140", "312.3719", "-465.8290", "473.8300", "Ver. 1.0, Ref. 7"},
                            {"c3'", "o1=", "1.2160", "823.7948", "-1878.7939", "2303.5310", "Ver. 1.0, Ref. 7"},
                            {"c3'", "c3a", "1.4890", "339.3574", "-655.7236", "670.2362", "Ver. 1.0, Ref. 7"},
                            {"c3'", "n3m", "1.3850", "359.1591", "-558.4730", "1146.3810", "Ver. 1.0, Ref. 7"},
                            {"c3a", "n3m", "1.3950", "344.0452", "-652.1208", "1022.2242", "Ver. 1.0, Ref. 7"},
                            {"n1z", "n1z", "1.1354", "1337.7450", "-2675.4900", "3121.4049", "Ver. 1.0, Ref. 9"},
                            {"n2z", "n2t", "1.2343", "720.3345", "-1542.6689", "1799.7804", "Ver. 1.0, Ref. 9"},
                            {"n2t", "n1z", "1.1354", "1198.7450", "-2675.4900", "3121.4049", "Ver. 1.0, Ref. 9"},
                            {"n2z", "c4", "1.4814", "324.4578", "-648.9156", "757.0681", "Ver. 1.0, Ref. 9"},
                            {"n2z", "h1", "1.0221", "440.1623", "-960.3246", "1120.3787", "Ver. 1.0, Ref. 9"},
                            {"c3a", "si4", "1.8634", "233.2433", "-276.8692", "161.6659", "Ver. 1.0, Ref. 10"},
                            {"c4", "si4", "1.8995", "189.6536", "-279.4210", "307.5135", "Ver. 1.0, Ref. 10"},
                            {"h1", "si4", "1.4783", "202.7798", "-305.3603", "280.2685", "Ver. 1.0, Ref. 10"},
                            {"si4", "si4", "2.3384", "114.2164", "-140.4212", "80.7084", "Ver. 1.0, Ref. 10"},
                            {"c4", "n3m", "1.4000", "350.0000", "0.0000", "0.0000", "Ver. 1.0, Ref. 10"}};

/*
 Quartic angles
  0= Key_a, 1= Key_b, 2= Key_c, 3= Theta0, 4= K2, 5= K3, 6= K4, 7= Compass info

  Delta = Theta - Theta0
  E = K2 * Delta^2  +  K3 * Delta^3  +  K4 * Delta^4
  With:
   - Theta = Angle
   - Theta0 = Equilibrium angle
*/
char * Compass_angles[94][8]= {{"c3a", "c3a", "c3a", "118.9000", "61.0226", "-34.9931", "0.0000", "Ver. 1.0, Ref. 1"},
                             {"c3a", "c3a", "c4", "120.0500", "44.7148", "-22.7352", "0.0000", "Ver. 1.0, Ref. 1"},
                             {"c3a", "c3a", "h1", "117.9400", "35.1558", "-12.4682", "0.0000", "Ver. 1.0, Ref. 1"},
                             {"c3a", "c4", "c3a", "111.0000", "44.3234", "-9.4454", "0.0000", "Ver. 1.0, Ref. 1"},
                             {"c3a", "c4", "c4", "108.4000", "43.9594", "-8.3924", "-9.3379", "Ver. 1.0, Ref. 1"},
                             {"c3a", "c4", "h1", "111.0000", "44.3234", "-9.4454", "0.0000", "Ver. 1.0, Ref. 1"},
                             {"c4", "c4", "c4", "112.6700", "39.5160", "-7.4430", "-9.5583", "Ver. 1.0, Ref. 1"},
                             {"c4", "c4", "h1", "110.7700", "41.4530", "-10.6040", "5.1290", "Ver. 1.0, Ref. 1"},
                             {"h1", "c4", "h1", "107.6600", "39.6410", "-12.9210", "-2.4318", "Ver. 1.0, Ref. 1"},
                             {"h1", "o2z", "si4", "122.8000", "23.7764", "-19.8152", "9.6331", "Ver. 1.0, Ref. 2"},
                             {"si4", "o2z", "si4", "159.0000", "8.5000", "-13.4188", "-4.1785", "Ver. 1.0, Ref. 2"},
                             {"c3a", "si4", "o2z", "114.9060", "23.0218", "-31.3993", "24.9814", "Ver. 1.0, Ref. 2"},
                             {"c4", "si4", "o2z", "114.9060", "23.0218", "-31.3993", "24.9814", "Ver. 1.0, Ref. 2"},
                             {"h1", "si4", "o2z", "107.4000", "57.6643", "-10.6506", "4.6274", "Ver. 1.0, Ref. 2"},
                             {"o2z", "si4", "o2z", "110.7000", "70.3069", "-6.9375", "0.0000", "Ver. 1.0, Ref. 2"},
                             {"c3a", "c3a", "o2", "123.4200", "73.6781", "-21.6787", "0.0000", "Ver. 1.0, Ref. 3"},
                             {"c4", "c4", "o2", "111.2700", "54.5381", "-8.3642", "-13.0838", "Ver. 1.0, Ref. 3"},
                             {"h1", "c4", "o2", "108.7280", "58.5446", "-10.8088", "-12.4006", "Ver. 1.0, Ref. 3"},
                             {"c3a", "o2", "c4", "102.9695", "38.9739", "-6.2595", "-8.1710", "Ver. 1.0, Ref. 3"},
                             {"c3a", "o2", "h1", "108.1900", "53.1250", "-8.5016", "0.0000", "Ver. 1.0, Ref. 3"},
                             {"c4", "o2", "c4", "104.5000", "35.7454", "-10.0067", "-6.2729", "Ver. 1.0, Ref. 3"},
                             {"c4", "o2", "h1", "105.8000", "52.7061", "-12.1090", "-9.8681", "Ver. 1.0, Ref. 3"},
                             {"c3a", "c3a", "n2=", "120.0000", "60.0000", "0.0000", "0.0000", "Ver. 1.0, Ref. 4"},
                             {"c3a", "c3a", "p4=", "120.0010", "47.8410", "-15.2290", "-10.9070", "Ver. 1.0, Ref. 4"},
                             {"c4", "c4", "n2=", "117.3170", "55.2420", "0.0000", "0.0000", "Ver. 1.0, Ref. 4"},
                             {"h1", "c4", "n2=", "107.4990", "62.7310", "0.0000", "0.0000", "Ver. 1.0, Ref. 4"},
                             {"h1", "c4", "p4=", "110.8860", "33.8300", "-7.0430", "-7.2460", "Ver. 1.0, Ref. 4"},
                             {"c4", "n2=", "h1", "117.2000", "37.2620", "0.0000", "0.0000", "Ver. 1.0, Ref. 4"},
                             {"h1", "n2=", "h1", "110.9100", "31.0910", "0.0000", "0.0000", "Ver. 1.0, Ref. 4"},
                             {"h1", "n2=", "p4=", "120.0000", "26.0680", "-8.2980", "-5.9430", "Ver. 1.0, Ref. 4"},
                             {"p4=", "n2=", "p4=", "135.0000", "23.8680", "-8.7360", "0.0000", "Ver. 1.0, Ref. 4"},
                             {"c4", "n3m", "p4=", "120.0830", "25.0010", "-6.1170", "-5.4570", "Ver. 1.0, Ref. 4"},
                             {"h1", "n3m", "p4=", "120.0830", "25.0010", "-6.1170", "-5.4570", "Ver. 1.0, Ref. 4"},
                             {"c4", "o2", "p4=", "118.2830", "35.0010", "-10.3600", "-7.8700", "Ver. 1.0, Ref. 4"},
                             {"h1", "o2", "p4=", "117.0000", "26.0310", "-5.8280", "-5.6200", "Ver. 1.0, Ref. 4"},
                             {"c3a", "p4=", "c3a", "110.2310", "56.1850", "-17.3160", "-12.7280", "Ver. 1.0, Ref. 4"},
                             {"c3a", "p4=", "h1", "108.2310", "36.1850", "-6.4880", "-7.6460", "Ver. 1.0, Ref. 4"},
                             {"c3a", "p4=", "n2=", "109.6000", "63.0620", "-19.7400", "-14.3290", "Ver. 1.0, Ref. 4"},
                             {"c3a", "p4=", "n3m", "108.1650", "70.9770", "-11.5480", "-15.1090", "Ver. 1.0, Ref. 4"},
                             {"c3a", "p4=", "o2", "107.3650", "71.9770", "-10.9430", "-15.2900", "Ver. 1.0, Ref. 4"},
                             {"c4", "p4=", "c4", "102.5000", "48.2320", "-5.7980", "-9.9660", "Ver. 1.0, Ref. 4"},
                             {"c4", "p4=", "h1", "102.9000", "52.0710", "-6.4680", "-10.7730", "Ver. 1.0, Ref. 4"},
                             {"c4", "p4=", "n2=", "119.3000", "47.3660", "-14.6410", "-10.7360", "Ver. 1.0, Ref. 4"},
                             {"h1", "p4=", "h1", "101.4080", "39.6950", "-5.1340", "-8.2270", "Ver. 1.0, Ref. 4"},
                             {"h1", "p4=", "n2=", "110.0330", "45.9780", "-14.0520", "-10.3990", "Ver. 1.0, Ref. 4"},
                             {"h1", "p4=", "n3m", "103.9780", "68.2570", "-9.2210", "-14.1740", "Ver. 1.0, Ref. 4"},
                             {"h1", "p4=", "o2", "103.9780", "73.2570", "-9.8970", "-15.2120", "Ver. 1.0, Ref. 4"},
                             {"n2=", "p4=", "n2=", "125.0000", "90.5230", "-20.8010", "-19.6020", "Ver. 1.0, Ref. 4"},
                             {"n2=", "p4=", "n3m", "123.2150", "89.9230", "-32.6120", "-21.0960", "Ver. 1.0, Ref. 4"},
                             {"n2=", "p4=", "o2", "112.2150", "99.9230", "-32.0930", "-22.8210", "Ver. 1.0, Ref. 4"},
                             {"n3m", "p4=", "n3m", "107.1000", "85.7690", "-5.7790", "-17.4890", "Ver. 1.0, Ref. 4"},
                             {"n3m", "p4=", "o2", "108.3000", "86.7690", "-5.1750", "-17.6710", "Ver. 1.0, Ref. 4"},
                             {"o2", "p4=", "o2", "107.5000", "86.7690", "-4.5700", "-17.8520", "Ver. 1.0, Ref. 4"},
                             {"o1=", "c2=", "o1=", "180.0000", "57.1000", "0.0000", "0.0000", "Ver. 1.0, Ref. 5"},
                             {"s1=", "c2=", "s1=", "180.0000", "48.0000", "0.0000", "0.0000", "Ver. 1.0, Ref. 5"},
                             {"o1=", "n2o", "o1=", "134.1000", "150.0000", "-82.1013", "-40.0005", "Ver. 1.0, Ref. 5"},
                             {"o1=", "s2=", "o1=", "119.3000", "115.2627", "-35.6278", "-26.1261", "Ver. 1.0, Ref. 5"},
                             {"c3a", "c3a", "n3o", "118.8000", "29.2436", "-8.8495", "-6.6020", "Ver. 1.0, Ref. 6"},
                             {"h1", "c4", "n3o", "107.0000", "54.9318", "-9.1333", "-11.5434", "Ver. 1.0, Ref. 6"},
                             {"h1", "c4", "o2n", "108.7280", "58.5446", "-10.8088", "-12.4006", "Ver. 1.0, Ref. 6"},
                             {"c3a", "n3o", "o1=", "117.7000", "63.9404", "-18.4524", "-14.3129", "Ver. 1.0, Ref. 6"},
                             {"c4", "n3o", "o1=", "117.5000", "64.5228", "-18.4582", "-14.4215", "Ver. 1.0, Ref. 6"},
                             {"h1", "n3o", "o1=", "115.7000", "53.8034", "-14.1991", "-11.8708", "Ver. 1.0, Ref. 6"},
                             {"o1=", "n3o", "o1=", "128.0000", "95.1035", "-47.4240", "-27.9164", "Ver. 1.0, Ref. 6"},
                             {"c4", "o2n", "n3o", "108.5000", "55.7454", "-10.0067", "-6.2729", "Ver. 1.0, Ref. 6"},
                             {"c4", "c4", "o2n", "105.0000", "54.5381", "-8.3642", "-13.0838", "Ver. 1.0, Ref. 6"},
                             {"o2n", "n3o", "o1=", "112.8000", "85.5228", "-18.4582", "-14.4215", "Ver. 1.0, Ref. 6"},
                             {"c3'", "o2", "c4", "109.0000", "38.9739", "-6.2595", "-8.1710", "Ver. 1.0, Ref. 7"},
                             {"c3'", "c4", "h1", "107.8594", "38.0833", "-17.5074", "0.0000", "Ver. 1.0, Ref. 7"},
                             {"c3'", "n3m", "c3'", "121.9556", "76.3105", "-26.3166", "-17.6944", "Ver. 1.0, Ref. 7"},
                             {"c3a", "c3a", "c3'", "116.0640", "71.2598", "-15.8273", "2.0506", "Ver. 1.0, Ref. 7"},
                             {"c3a", "c3'", "n3m", "108.4400", "84.8377", "-19.9640", "2.7405", "Ver. 1.0, Ref. 7"},
                             {"c3a", "c3'", "o1=", "125.5320", "72.3167", "-16.0650", "2.0818", "Ver. 1.0, Ref. 7"},
                             {"c3a", "c3a", "n3m", "120.7640", "73.2738", "-27.4033", "13.3920", "Ver. 1.0, Ref. 7"},
                             {"c3a", "n3m", "c3'", "120.0700", "47.1131", "-32.5592", "13.1257", "Ver. 1.0, Ref. 7"},
                             {"o1=", "c3'", "o2", "118.9855", "98.6813", "-22.2485", "10.3673", "Ver. 1.0, Ref. 7"},
                             {"o1=", "c3'", "c4", "119.3000", "65.1016", "-17.9766", "0.0000", "Ver. 1.0, Ref. 7"},
                             {"o2", "c3'", "c4", "100.3182", "88.8631", "-3.8323", "-7.9802", "Ver. 1.0, Ref. 7"},
                             {"n3m", "c3'", "o1=", "121.5420", "92.5720", "-34.4800", "-11.1871", "Ver. 1.0, Ref. 7"},
                             {"n2z", "n2t", "n1z", "171.6211", "47.7899", "0.0000", "0.0000", "Ver. 1.0, Ref. 9"},
                             {"n2t", "n2z", "h1", "110.0345", "55.7635", "0.6618", "0.0022", "Ver. 1.0, Ref. 9"},
                             {"n2t", "n2z", "c4", "113.5017", "82.6294", "0.9845", "0.0033", "Ver. 1.0, Ref. 9"},
                             {"n2z", "c4", "h1", "107.9744", "52.7803", "0.6615", "0.0023", "Ver. 1.0, Ref. 9"},
                             {"n2z", "c4", "c4", "110.9900", "77.9387", "0.9499", "0.0033", "Ver. 1.0, Ref. 9"},
                             {"c3a", "c3a", "si4", "120.0000", "30.4689", "-23.5439", "0.0000", "Ver. 1.0, Ref. 10"},
                             {"c4", "c4", "si4", "112.6700", "39.5160", "-7.4430", "0.0000", "Ver. 1.0, Ref. 10"},
                             {"h1", "c4", "si4", "112.0355", "28.7721", "-13.9523", "0.0000", "Ver. 1.0, Ref. 10"},
                             {"c3a", "si4", "h1", "109.5932", "41.9497", "-42.3639", "48.1442", "Ver. 1.0, Ref. 10"},
                             {"c4", "si4", "c4", "113.1855", "36.2069", "-20.3939", "20.0172", "Ver. 1.0, Ref. 10"},
                             {"c4", "si4", "h1", "112.0977", "36.4832", "-12.8094", "0.0000", "Ver. 1.0, Ref. 10"},
                             {"h1", "si4", "h1", "108.6051", "32.5415", "-8.3164", "0.0000", "Ver. 1.0, Ref. 10"},
                             {"c4", "si4", "si4", "113.0000", "19.4692", "-34.3471", "0.0000", "Ver. 1.0, Ref. 10"},
                             {"h1", "si4", "si4", "112.0893", "22.5062", "-11.5926", "0.0000", "Ver. 1.0, Ref. 10"},
                             {"si4", "si4", "si4", "114.2676", "24.9501", "-19.5949", "0.0000", "Ver. 1.0, Ref. 10"}};

/*
 Dihedrals / torsions
 0, 1, 2, 3 = Keys, 4= V1, 5= Phi0, 6= V2, 7= Phi0, 8= V3, 9= Phi0, 10= Compass info
 E = SUM(n=1,3) { V(n) * [ 1 - cos(n*Phi - Phi0(n)) ] }
 With:
  - Phi0 = Equilibrium angle
*/
char * Compass_torsions[95][11]= {{"c3a", "c3a", "c3a", "c3a", "8.3667", "0.0", "1.2000", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                {"c3a", "c3a", "c3a", "c4", "0.0000", "0.0", "4.4072", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                {"c3a", "c3a", "c3a", "h1", "0.0000", "0.0", "3.9661", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                {"c4", "c3a", "c3a", "h1", "0.0000", "0.0", "1.5590", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                {"h1", "c3a", "c3a", "h1", "0.0000", "0.0", "2.3500", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                {"c3a", "c3a", "c4", "c3a", "-0.2802", "0.0", "-0.0678", "0.0", "-0.0122", "0.0", "Ver. 1.0, Ref. 1"},
                                {"c3a", "c3a", "c4", "c4", "-0.2802", "0.0", "-0.0678", "0.0", "-0.0122", "0.0", "Ver. 1.0, Ref. 1"},
                                {"c3a", "c3a", "c4", "h1", "-0.2802", "0.0", "-0.0678", "0.0", "-0.0122", "0.0", "Ver. 1.0, Ref. 1"},
                                {"c3a", "c4", "c4", "h1", "-0.0228", "0.0", "0.0280", "0.0", "-0.1863", "0.0", "Ver. 1.0, Ref. 1"},
                                {"c4", "c4", "c4", "c4", "0.0000", "0.0", "0.0514", "0.0", "-0.1430", "0.0", "Ver. 1.0, Ref. 1"},
                                {"c4", "c4", "c4", "h1", "0.0000", "0.0", "0.0316", "0.0", "-0.1681", "0.0", "Ver. 1.0, Ref. 1"},
                                {"h1", "c4", "c4", "h1", "-0.1432", "0.0", "0.0617", "0.0", "-0.1530", "0.0", "Ver. 1.0, Ref. 1"},
                                {"*", "c3a", "c3a", "*", "0.0000", "0.0", "4.5000", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                {"*", "c4", "c4", "*", "0.0000", "0.0", "0.0000", "0.0", "-0.1530", "0.0", "Ver. 1.0, Ref. 1"},
                                {"h1", "o2z", "si4", "c4", "0.0000", "0.0", "0.0000", "0.0", "-0.0500", "0.0", "Ver. 1.0, Ref. 2"},
                                {"h1", "o2z", "si4", "h1", "0.0000", "0.0", "0.0000", "0.0", "-0.0500", "0.0", "Ver. 1.0, Ref. 2"},
                                {"h1", "o2z", "si4", "o2z", "0.0000", "0.0", "0.0000", "0.0", "-0.0500", "0.0", "Ver. 1.0, Ref. 2"},
                                {"si4", "o2z", "si4", "c4", "0.0000", "0.0", "0.0000", "0.0", "-0.0100", "0.0", "Ver. 1.0, Ref. 2"},
                                {"si4", "o2z", "si4", "h1", "0.0000", "0.0", "0.0000", "0.0", "-0.0100", "0.0", "Ver. 1.0, Ref. 2"},
                                {"si4", "o2z", "si4", "o2z", "-0.2250", "0.0", "0.0000", "0.0", "-0.0100", "0.0", "Ver. 1.0, Ref. 2"},
                                {"*", "o2z", "si4", "*", "0.0000", "0.0", "0.0000", "0.0", "-0.0100", "0.0", "Ver. 1.0, Ref. 2"},
                                {"c3a", "c3a", "c3a", "o2", "0.0000", "0.0", "4.8498", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 3"},
                                {"h1", "c3a", "c3a", "o2", "0.0000", "0.0", "1.7234", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 3"},
                                {"c3a", "c3a", "o2", "c4", "0.0000", "0.0", "1.5000", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 3"},
                                {"c3a", "c3a", "o2", "h1", "-0.6900", "0.0", "0.5097", "0.0", "0.0095", "0.0", "Ver. 1.0, Ref. 3"},
                                {"c4", "c4", "c4", "o2", "0.7137", "0.0", "0.2660", "0.0", "-0.2545", "0.0", "Ver. 1.0, Ref. 3"},
                                {"h1", "c4", "c4", "o2", "-0.1435", "0.0", "0.2530", "0.0", "-0.0905", "0.0", "Ver. 1.0, Ref. 3"},
                                {"o2", "c4", "c4", "o2", "1.1000", "0.0", "-0.0500", "0.0", "-0.1441", "0.0", "Ver. 1.0, Ref. 3"},
                                {"c4", "c4", "o2", "c4", "-0.4000", "0.0", "-0.4028", "0.0", "-0.2450", "0.0", "Ver. 1.0, Ref. 3"},
                                {"c4", "c4", "o2", "h1", "-0.6732", "0.0", "-0.4778", "0.0", "-0.1670", "0.0", "Ver. 1.0, Ref. 3"},
                                {"h1", "c4", "o2", "c3a", "0.9513", "0.0", "0.1155", "0.0", "0.0720", "0.0", "Ver. 1.0, Ref. 3"},
                                {"h1", "c4", "o2", "c4", "0.5302", "0.0", "0.0000", "0.0", "-0.3966", "0.0", "Ver. 1.0, Ref. 3"},
                                {"h1", "c4", "o2", "h1", "0.1863", "0.0", "-0.4338", "0.0", "-0.2121", "0.0", "Ver. 1.0, Ref. 3"},
                                {"c3a", "c3a", "c3a", "p4=", "0.0000", "0.0", "5.4770", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 4"},
                                {"h1", "c3a", "c3a", "p4=", "0.0000", "0.0", "2.2700", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 4"},
                                {"c3a", "c3a", "p4=", "h1", "-0.2720", "0.0", "1.1900", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 4"},
                                {"c3a", "c3a", "p4=", "n2=", "-0.2720", "0.0", "1.1900", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 4"},
                                {"c4", "c4", "c4", "n2=", "0.0970", "0.0", "0.0720", "0.0", "-0.2580", "0.0", "Ver. 1.0, Ref. 4"},
                                {"h1", "c4", "c4", "n2=", "-0.1510", "0.0", "0.0100", "0.0", "-0.1860", "0.0", "Ver. 1.0, Ref. 4"},
                                {"c4", "c4", "n2=", "h1", "-5.0720", "0.0", "-0.4980", "0.0", "-0.4380", "0.0", "Ver. 1.0, Ref. 4"},
                                {"h1", "c4", "n2=", "h1", "1.2660", "0.0", "-0.7740", "0.0", "0.0380", "0.0", "Ver. 1.0, Ref. 4"},
                                {"h1", "c4", "p4=", "c4", "0.0000", "0.0", "0.0000", "0.0", "-0.0680", "0.0", "Ver. 1.0, Ref. 4"},
                                {"h1", "c4", "p4=", "h1", "0.0000", "0.0", "0.0000", "0.0", "-0.0680", "0.0", "Ver. 1.0, Ref. 4"},
                                {"h1", "c4", "p4=", "n2=", "0.0000", "0.0", "0.0000", "0.0", "-0.0680", "0.0", "Ver. 1.0, Ref. 4"},
                                {"h1", "n2=", "p4=", "c3a", "0.0000", "0.0", "0.0000", "0.0", "-0.3500", "0.0", "Ver. 1.0, Ref. 4"},
                                {"h1", "n2=", "p4=", "c4", "0.0000", "0.0", "0.0000", "0.0", "-0.3690", "0.0", "Ver. 1.0, Ref. 4"},
                                {"h1", "n2=", "p4=", "h1", "0.0000", "0.0", "0.0000", "0.0", "-0.3500", "0.0", "Ver. 1.0, Ref. 4"},
                                {"h1", "n2=", "p4=", "o2", "0.0000", "0.0", "0.0000", "0.0", "-0.3500", "0.0", "Ver. 1.0, Ref. 4"},
                                {"p4=", "n2=", "p4=", "h1", "0.0000", "0.0", "0.0000", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 4"},
                                {"p4=", "n2=", "p4=", "n2=", "1.8000", "0.0", "0.5000", "0.0", "2.0000", "0.0", "Ver. 1.0, Ref. 4"},
                                {"h1", "n3m", "p4=", "h1", "0.0000", "0.0", "0.0000", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 4"},
                                {"h1", "n3m", "p4=", "n2=", "0.0000", "0.0", "0.0000", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 4"},
                                {"h1", "n3m", "p4=", "o2", "0.0000", "0.0", "0.0000", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 4"},
                                {"h1", "o2", "p4=", "h1", "5.7080", "0.0", "2.1180", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 4"},
                                {"h1", "o2", "p4=", "n2=", "5.7080", "0.0", "2.1180", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 4"},
                                {"h1", "o2", "p4=", "o2", "5.7080", "0.0", "2.1180", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 4"},
                                {"*", "o2", "p4=", "*", "5.7080", "0.0", "2.1180", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 4"},
                                {"*", "c3a", "n2=", "*", "0.0000", "0.0", "1.0000", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 4"},
                                {"*", "c3a", "p4=", "*", "-0.2720", "0.0", "1.1900", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 4"},
                                {"*", "c4", "n2=", "*", "0.0000", "0.0", "0.0000", "0.0", "-0.0200", "0.0", "Ver. 1.0, Ref. 4"},
                                {"*", "c4", "p4=", "*", "0.0000", "0.0", "0.0000", "0.0", "-0.0680", "0.0", "Ver. 1.0, Ref. 4"},
                                {"*", "n2=", "p4=", "*", "0.0000", "0.0", "0.0000", "0.0", "-0.3500", "0.0", "Ver. 1.0, Ref. 4"},
                                {"*", "n3m", "p4=", "*", "0.0000", "0.0", "0.0000", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 4"},
                                {"c3a", "c3a", "c3a", "n3o", "0.0000", "0.0", "7.2124", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 6"},
                                {"h1", "c3a", "c3a", "n3o", "0.0000", "0.0", "2.9126", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 6"},
                                {"c3a", "c3a", "n3o", "o1=", "0.0000", "0.0", "1.1600", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 6"},
                                {"c4", "c4", "n3o", "o1=", "0.0000", "0.0", "0.0000", "0.0", "-0.3500", "0.0", "Ver. 1.0, Ref. 6"},
                                {"h1", "c4", "n3o", "o1=", "0.0000", "0.0", "0.0000", "0.0", "-0.3500", "0.0", "Ver. 1.0, Ref. 6"},
                                {"c4", "c4", "o2", "n3o", "0.0000", "0.0", "-0.4000", "0.0", "-0.2000", "0.0", "Ver. 1.0, Ref. 6"},
                                {"o1=", "n3o", "o2", "c4", "0.0000", "0.0", "2.0000", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 6"},
                                {"c3'", "c3a", "c3a", "c3a", "0.0000", "0.0", "4.6282", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 7"},
                                {"c3'", "c3a", "c3a", "h1", "0.0000", "0.0", "2.1670", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 7"},
                                {"c3'", "n3m", "c3a", "c3a", "0.0000", "0.0", "0.6500", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 7"},
                                {"c3'", "n3m", "c3'", "o1c", "-0.4066", "0.0", "1.2513", "0.0", "-0.7507", "0.0", "Ver. 1.0, Ref. 7"},
                                {"c3'", "o2", "c4", "c4", "0.1302", "0.0", "-0.3250", "0.0", "0.1134", "0.0", "Ver. 1.0, Ref. 7"},
                                {"c3'", "o2", "c4", "h1", "0.9513", "0.0", "0.1155", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 7"},
                                {"c3a", "c3a", "c3'", "o1=", "0.0000", "0.0", "0.7800", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 7"},
                                {"c3a", "c3a", "c3a", "n3m", "0.0000", "0.0", "3.4040", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 7"},
                                {"c3a", "n3m", "c3'", "o1=", "0.0000", "0.0", "2.0521", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 7"},
                                {"c4", "o2", "c3'", "c4", "-2.5594", "0.0", "2.2013", "0.0", "0.0325", "0.0", "Ver. 1.0, Ref. 7"},
                                {"c4", "o2", "c3'", "o1=", "0.8905", "0.0", "3.2644", "0.0", "0.2646", "0.0", "Ver. 1.0, Ref. 7"},
                                {"o1=", "c3'", "c4", "h1", "0.0000", "0.0", "0.0000", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 7"},
                                {"o2", "c3'", "c4", "h1", "0.0000", "0.0", "0.0000", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 7"},
                                {"n3m", "c3a", "c3a", "h1", "0.0000", "0.0", "3.4040", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 7"},
                                {"h1", "n2z", "n2t", "n1z", "0.0000", "0.0", "0.0000", "0.0", "-0.2637", "0.0", "Ver. 1.0, Ref. 9"},
                                {"c4", "n2z", "n2t", "n1z", "0.0000", "0.0", "0.0000", "0.0", "-0.1823", "0.0", "Ver. 1.0, Ref. 9"},
                                {"n2t", "n2z", "c4", "h1", "0.0000", "0.0", "0.0000", "0.0", "-0.2181", "0.0", "Ver. 1.0, Ref. 9"},
                                {"n2t", "n2z", "c4", "c4", "0.0000", "0.0", "0.0000", "0.0", "-0.2021", "0.0", "Ver. 1.0, Ref. 9"},
                                {"n2z", "c4", "c4", "h1", "0.0000", "0.0", "0.0000", "0.0", "-0.2259", "0.0", "Ver. 1.0, Ref. 9"},
                                {"c3a", "c3a", "c3a", "si4", "0.0000", "0.0", "4.3270", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 10"},
                                {"h1", "c3a", "c3a", "si4", "0.0000", "0.0", "1.5093", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 10"},
                                {"c3a", "c3a", "si4", "*", "0.0000", "0.0", "0.0000", "0.0", "-0.0231", "0.0", "Ver. 1.0, Ref. 10"},
                                {"c4", "c4", "si4", "si4", "-0.3500", "0.0", "0.0000", "0.0", "-0.0657", "0.0", "Ver. 1.0, Ref. 10"},
                                {"*", "c4", "si4", "*", "0.0000", "0.0", "0.0000", "0.0", "-0.0657", "0.0", "Ver. 1.0, Ref. 10"},
                                {"*", "si4", "si4", "*", "0.0000", "0.0", "0.0000", "0.0", "-0.0657", "0.0", "Ver. 1.0, Ref. 10"}};

/*
 Inversions
  0, 1, 2, 3 = Keys, 4= K, 5= Chi0, 6= Compass info
  E = K * (Chi - Chi0)^2
*/
char * Compass_inversions[22][7]= {{"c3a", "c3a", "c3a", "c3a", "7.1794", "0.0", "Ver. 1.0, Ref. 1"},
                                {"c3a", "c3a", "c3a", "c4", "7.8153", "0.0", "Ver. 1.0, Ref. 1"},
                                {"c3a", "c3a", "c3a", "h1", "4.8912", "0.0", "Ver. 1.0, Ref. 1"},
                                {"c3a", "c3a", "c3a", "o2", "13.0421", "0.0", "Ver. 1.0, Ref. 3"},
                                {"c3a", "c3a", "c3a", "n2=", "8.0000", "0.0", "Ver. 1.0, Ref. 4"},
                                {"c3a", "c3a", "c3a", "p4=", "6.7090", "0.0", "Ver. 1.0, Ref. 4"},
                                {"c3a", "c3a", "c3a", "n3o", "0.9194", "0.0", "Ver. 1.0, Ref. 6"},
                                {"c3a", "n3o", "o1=", "o1=", "36.2612", "0.0", "Ver. 1.0, Ref. 6"},
                                {"c4", "n3o", "o1=", "o1=", "44.3062", "0.0", "Ver. 1.0, Ref. 6"},
                                {"h1", "n3o", "o1=", "o1=", "38.5581", "0.0", "Ver. 1.0, Ref. 6"},
                                {"o1=", "n3o", "o1=", "o2", "45.0000", "0.0", "Ver. 1.0, Ref. 6"},
                                {"c3'", "c3'", "n3m", "c3a", "0.0000", "0.0", "Ver. 1.0, Ref. 7"},
                                {"c3'", "c3a", "c3a", "c3a", "17.0526", "0.0", "Ver. 1.0, Ref. 7"},
                                {"c3'", "n3m", "c3'", "c3a", "0.0000", "0.0", "Ver. 1.0, Ref. 7"},
                                {"c3a", "c3a", "n3m", "c3a", "17.0526", "0.0", "Ver. 1.0, Ref. 7"},
                                {"c3a", "c3a", "c3a", "n3m", "17.0526", "0.0", "Ver. !1.0, Ref. 7"},
                                {"c3a", "c3'", "n3m", "o1=", "30.0000", "0.0", "Ver. 1.0, Ref. 7"},
                                {"c3a", "c3a", "c3'", "c3a", "17.0526", "0.0", "Ver. !1.0, Ref. 7"},
                                {"c3a", "c3'", "o1=", "n3m", "30.0000", "0.0", "Ver. !1.0, Ref. 7"},
                                {"c3a", "o1=", "c3'", "n3m", "30.0000", "0.0", "Ver. 1.0, Ref. 7"},
                                {"c4", "c3'", "o2", "o1=", "46.9264", "0.0", "Ver. 1.0, Ref. 7"},
                                {"c3a", "c3a", "si4", "c3a", "5.3654", "0.0", "Ver. 1.0, Ref. 10"}};

/*
  Non-bonded (9-6)
 0= Key, 1= r, 2= eps, 3= Compass info
 E = eps(ij) [2(R(ij)/r(ij))^9 - 3(R(ij)/r(ij))^6]
 where    R(ij) = [(r(i)^6 + r(j)^6))/2]^(1/6)

        eps(ij) = 2 sqrt(eps(i) * eps(j)) *
                   r(i)^3 * r(j)^3/[r(i)^6 + r(j)^6]
*/
char * Compass_vdw[46][4]= {{"c3a", "3.9150", "0.0680", "Ver. 1.0, Ref. 1"},
                         {"c4", "3.8540", "0.0620", "Ver. 1.0, Ref. 1"},
                         {"c43", "3.8540", "0.0400", "Ver. 1.0, Ref. 1"},
                         {"c44", "3.8540", "0.0200", "Ver. 1.0, Ref. 1"},
                         {"h1", "2.8780", "0.0230", "Ver. 1.0, Ref. 1"},
                         {"o2z", "3.3000", "0.0800", "Ver. 1.0, Ref. 2"},
                         {"si4", "4.4050", "0.1980", "Ver. 1.0, Ref. 2"},
                         {"si4c", "4.2900", "0.1310", "Ver. 1.0, Ref. 2"},
                         {"c4o", "3.8150", "0.0680", "Ver. 1.0, Ref. 3"},
                         {"h1o", "1.0870", "0.0080", "Ver. 1.0, Ref. 3"},
                         {"o2", "3.3000", "0.0800", "Ver. 1.0, Ref. 3"},
                         {"o2e", "3.3000", "0.1200", "Ver. 1.0, Ref. 3"},
                         {"o2h", "3.5800", "0.0960", "Ver. 1.0, Ref. 3"},
                         {"n2=", "3.8300", "0.0960", "Ver. 1.0, Ref. 4"},
                         {"p4=", "4.2950", "0.0650", "Ver. 1.0, Ref. 4"},
                         {"he", "2.9000", "0.0050", "Ver. 1.0, Ref. 5"},
                         {"ne", "3.2000", "0.0550", "Ver. 1.0, Ref. 5"},
                         {"ar", "3.8800", "0.2000", "Ver. 1.0, Ref. 5"},
                         {"kr", "4.3000", "0.2800", "Ver. 1.0, Ref. 5"},
                         {"xe", "4.2600", "0.3900", "Ver. 1.0, Ref. 5"},
                         {"h1h", "1.4210", "0.0216", "Ver. 1.0, Ref. 5"},
                         {"n1n", "3.8008", "0.0598", "Ver. 1.0, Ref. 5"},
                         {"c1o", "4.0120", "0.0530", "Ver. 1.0, Ref. 5"},
                         {"o1o", "3.4758", "0.0780", "Ver. 1.0, Ref. 5"},
                         {"o1c", "3.6020", "0.0850", "Ver. 1.0, Ref. 5"},
                         {"n1o", "3.4600", "0.1280", "Ver. 1.0, Ref. 5"},
                         {"o1n", "3.3000", "0.1560", "Ver. 1.0, Ref. 5"},
                         {"c2=", "3.9150", "0.0680", "Ver. 1.0, Ref. 5"},
                         {"s2=", "4.0470", "0.1250", "Ver. 1.0, Ref. 5"},
                         {"n2o", "3.5290", "0.3330", "Ver. 1.0, Ref. 5"},
                         {"o1=", "3.4300", "0.1920", "Ver. 1.0, Ref. 5"},
                         {"o1=*", "3.3600", "0.0670", "Ver. 1.0, Ref. 5"},
                         {"s1=", "4.0070", "0.3130", "Ver. 1.0, Ref. 5"},
                         {"n3o", "3.7600", "0.0480", "Ver. 1.0, Ref. 6"},
                         {"o12", "3.4000", "0.0480", "Ver. 1.0, Ref. 6"},
                         {"o2n", "3.6500", "0.2000", "Ver. 1.0, Ref. 6"},
                         {"c3'", "3.9000", "0.0640", "Ver. 1.0, Ref. 7"},
                         {"n3m", "3.7200", "0.1500", "Ver. 1.0, Ref. 7"},
                         {"o2s", "3.3000", "0.0960", "Ver. 1.0, Ref. 7"},
                         {"c4o", "3.8700", "0.0748", "Ver. 1.1, Ref. 8"},
                         {"c41o", "3.8700", "0.1080", "Ver. 1.1, Ref. 8"},
                         {"c43o", "3.8150", "0.0680", "Ver. 1.2, Ref. 10"},
                         {"c4z", "3.6500", "0.0800", "Ver. 1.0, Ref. 9"},
                         {"n1z", "3.5200", "0.0850", "Ver. 1.0, Ref. 9"},
                         {"n2t", "3.3000", "0.0500", "Ver. 1.0, Ref. 9"},
                         {"n2z", "3.4000", "0.1200", "Ver. 1.0, Ref. 9"}};

/*
 Bond increments
 To implement atomic charges
*/
char * Compass_bond_increments[55][5]= {{"c3a", "c3a", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"c3a", "c4", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"c3a", "h1", "-0.1268", "0.1268", "Ver. 1.0, Ref. 1"},
                                      {"c4", "c4", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                      {"c4", "h1", "-0.0530", "0.0530", "Ver. 1.0, Ref. 1"},
                                      {"o2z", "si4", "-0.2225", "0.2225", "Ver. 1.0, Ref. 2"},
                                      {"c3a", "o2e", "0.0420", "-0.0420", "Ver. 1.0, Ref. 3"},
                                      {"c3a", "o2h", "0.0420", "-0.0420", "Ver. 1.0, Ref. 3"},
                                      {"c4", "o2e", "0.1600", "-0.1600", "Ver. 1.0, Ref. 3"},
                                      {"c4", "o2h", "0.1600", "-0.1600", "Ver. 1.0, Ref. 3"},
                                      {"h1", "o2", "0.4200", "-0.4200", "Ver. 1.0, Ref. 3"},
                                      {"h1", "o2h", "0.4200", "-0.4200", "Ver. 1.0, Ref. 3"},
                                      {"c3a", "n2=", "0.1990", "-0.1990", "Ver. 1.0, Ref. 4"},
                                      {"c3a", "p4=", "-0.0600", "0.0600", "Ver. 1.0, Ref. 4"},
                                      {"c4", "n2=", "0.3450", "-0.3450", "Ver. 1.0, Ref. 4"},
                                      {"c4", "p4=", "-0.0500", "0.0500", "Ver. 1.0, Ref. 4"},
                                      {"cl1p", "p4=", "-0.1200", "0.1200", "Ver. 1.0, Ref. 4"},
                                      {"f1p", "p4=", "-0.1800", "0.1800", "Ver. 1.0, Ref. 4"},
                                      {"h1", "n2=", "0.3280", "-0.3280", "Ver. 1.0, Ref. 4"},
                                      {"h1", "p4=", "-0.0500", "0.0500", "Ver. 1.0, Ref. 4"},
                                      {"n2=", "n2=", "0.0000", "0.0000", "Ver. 1.0, Ref. 4"},
                                      {"n2=", "n3", "0.0250", "-0.0250", "Ver. 1.0, Ref. 4"},
                                      {"n2=", "o2", "-0.0430", "0.0430", "Ver. 1.0, Ref. 4"},
                                      {"n2=", "p4=", "-0.3500", "0.3500", "Ver. 1.0, Ref. 4"},
                                      {"n3", "p4=", "-0.1200", "0.1200", "Ver. 1.0, Ref. 4"},
                                      {"o2", "p4=", "-0.1400", "0.1400", "Ver. 1.0, Ref. 4"},
                                      {"c1o", "o1c", "-0.0203", "0.0203", "Ver. 1.0, Ref. 5"},
                                      {"c2=", "o1=", "0.4000", "-0.4000", "Ver. 1.0, Ref. 5"},
                                      {"c2=", "s1=", "0.0258", "-0.0258", "Ver. 1.0, Ref. 5"},
                                      {"n2o", "o1=", "0.0730", "-0.0730", "Ver. 1.0, Ref. 5"},
                                      {"h1h", "h1h", "0.0000", "0.0000", "Ver. 1.0, Ref. 5"},
                                      {"n1n", "n1n", "0.0000", "0.0000", "Ver. 1.0, Ref. 5"},
                                      {"n1o", "o1n", "0.0288", "-0.0288", "Ver. 1.0, Ref. 5"},
                                      {"o1=", "s2=", "-0.2351", "0.2351", "Ver. 1.0, Ref. 5"},
                                      {"o1o", "o1o", "0.0000", "0.0000", "Ver. 1.0, Ref. 5"},
                                      {"c3a", "n3o", "0.2390", "-0.2390", "Ver. 1.0, Ref. 6"},
                                      {"c4", "n3o", "0.2100", "-0.2100", "Ver. 1.0, Ref. 6"},
                                      {"c4", "o2n", "0.3170", "-0.3170", "Ver. 1.0, Ref. 6"},
                                      {"h1", "n3o", "0.1880", "-0.1880", "Ver. 1.0, Ref. 6"},
                                      {"n3o", "o1=", "0.4280", "-0.4280", "Ver. 1.0, Ref. 6"},
                                      {"n3o", "o2n", "0.0010", "-0.0010", "Ver. 1.0, Ref. 6"},
                                      {"c3'", "o2e", "0.1120", "-0.1120", "Ver. 1.0, Ref. 7"},
                                      {"c3'", "c4", "0.0000", "0.0000", "Ver. 1.0, Ref. 7"},
                                      {"c3'", "o1=", "0.4500", "-0.4500", "Ver. 1.0, Ref. 7"},
                                      {"c3'", "c3a", "0.0350", "-0.0350", "Ver. 1.0, Ref. 7"},
                                      {"c3'", "n3m", "0.0000", "0.0000", "Ver. 1.0, Ref. 7"},
                                      {"c3a", "n3m", "0.0950", "-0.0950", "Ver. 1.0, Ref. 7"},
                                      {"h1", "o2h", "0.4100", "-0.4100", "Ver. 1.1, Ref. 8"},
                                      {"n2z", "c4", "-0.3110", "0.3110", "Ver. 1.0, Ref. 9"},
                                      {"n2z", "h1", "-0.3350", "0.3350", "Ver. 1.0, Ref. 9"},
                                      {"n2t", "n1t", "0.3860", "-0.3860", "Ver. 1.0, Ref. 9"},
                                      {"n2t", "n2z", "0.2470", "-0.2470", "Ver. 1.0, Ref. 9"},
                                      {"c3a", "si4", "-0.1170", "0.1170", "Ver. 1.0, Ref. 10"},
                                      {"c4", "si4", "-0.1350", "0.1350", "Ver. 1.0, Ref. 10"},
                                      {"h1", "si4", "-0.1260", "0.1260", "Ver. 1.0, Ref. 10"}};
