/*
 * Copyright (C) 2021-2024 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>

#include "common.h"
#include "xine-toolkit/backend.h"
#include "xine-toolkit/slider.h"
#include "xine-toolkit/labelbutton.h"
#include "xine-toolkit/label.h"
#include "actions.h"
#include "event.h"
#include "acontrol.h"
#include "skins.h"
#include "osd.h"
#include <xine/audio_out.h>

#define NUM_EQ 10

struct xui_actrl_st {
  gui_new_window_t      nw;

  xitk_widget_t        *slid[NUM_EQ + 1], *label[NUM_EQ + 1], *flat, *dismiss;
  int                   val[NUM_EQ + 1];

  int                   skin;
  /* prevent inlining this rare call. */
  void                 (*reset) (gGui_t *gui);
};

/* attempt to save some relocation table entries. */
static const struct {
  char skin[16], hint[8];
} _actrl_names[NUM_EQ + 1] = {
  { "SliderACtl30",  "30Hz" },
  { "SliderACtl60",  "60Hz" },
  { "SliderACtl125", "125Hz" },
  { "SliderACtl250", "250Hz" },
  { "SliderACtl500", "500Hz" },
  { "SliderACtl1k",  "1kHz" },
  { "SliderACtl2k",  "2kHz" },
  { "SliderACtl4k",  "4kHz" },
  { "SliderACtl8k",  "8kHz" },
  { "SliderACtl16k", "16kHz" },
  { "SliderACtlAmp", "" }
};

static void _actrl_flat (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  xui_actrl_t *actrl = (xui_actrl_t *)data;

  (void)w;
  (void)state;
  (void)modifier;
  actrl->reset (actrl->nw.gui);
}

static void _actrl_set_value (xitk_widget_t *w, void *data, int value, unsigned int modifier) {
  xui_actrl_t *actrl = (xui_actrl_t *)data;
  int i = xitk_widget_user_id (w);

  (void)modifier;
  actrl->val[i] = value;
  if (i < NUM_EQ) {
    xine_set_param (actrl->nw.gui->stream, XINE_PARAM_EQ_30HZ + i, value);
    osd_bar (actrl->nw.gui, _actrl_names[i].hint, -50, 50, value, OSD_BAR_POS2);
  } else {
    gui_set_amp_level (actrl->nw.gui, value);
  }
}

void acontrol_update_mixer_display (xui_actrl_t *actrl) {
  if (actrl && (actrl->val[NUM_EQ] != actrl->nw.gui->mixer.level[SOFTWARE_MIXER])) {
    actrl->val[NUM_EQ] = actrl->nw.gui->mixer.level[SOFTWARE_MIXER];
    xitk_slider_set_pos (actrl->slid[NUM_EQ], actrl->val[NUM_EQ]);
  }
}

static void _actrl_exit (xui_actrl_t *actrl) {
  if (actrl->nw.xwin)
    gui_window_delete (&actrl->nw);
  actrl->nw.gui->actrl = NULL;
  free (actrl);
}

static void _actrl_close (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  xui_actrl_t *actrl = (xui_actrl_t *)data;

  (void)w;
  (void)state;
  (void)modifier;
  _actrl_exit (actrl);
}

/* Handle X events here. */
static int _actrl_event (void *data, const xitk_be_event_t *e) {
  xui_actrl_t *actrl = (xui_actrl_t *)data;

  switch (e->type) {
    case XITK_EV_DEL_WIN:
      _actrl_exit (actrl);
      return 1;
    case XITK_EV_KEY_UP:
      if (e->utf8[0] == XITK_CTRL_KEY_PREFIX) {
        if (e->utf8[1] == XITK_KEY_BACKSPACE)
          return xitk_widget_key_event (actrl->flat, e, 1);
        if (e->utf8[1] == XITK_KEY_ESCAPE)
          return xitk_widget_key_event (actrl->dismiss, e, 1);
      }
      break;
    case XITK_EV_KEY_DOWN:
      if (e->utf8[0] == XITK_CTRL_KEY_PREFIX) {
        switch (e->utf8[1]) {
          case XITK_KEY_ESCAPE:
            return xitk_widget_key_event (actrl->dismiss, e, 1);
          case XITK_KEY_UP:
          case XITK_KEY_LEFT:
          case XITK_KEY_DOWN:
          case XITK_KEY_RIGHT:
            /* we dont get keys handled by sliders already here. */
            return xitk_widgets_cycle (actrl->slid, NUM_EQ + 1, e->utf8[1]);
          case XITK_KEY_BACKSPACE:
            return xitk_widget_key_event (actrl->flat, e, 1);
          default: ;
        }
      }
      break;
    default: ;
  }
  return gui_handle_be_event (actrl->nw.gui, e);
}

/* Create control panel window */
void acontrol_main (xitk_widget_t *mode, void *data) {
  gGui_t *gui = (gGui_t *)data;
  xui_actrl_t *actrl;

  if (!gui)
    return;

  actrl = gui->actrl;
  if (mode == XUI_W_OFF) {
    if (!actrl)
      return;
    _actrl_exit (actrl);
    return;
  } else if (mode == XUI_W_ON) {
    if (actrl) {
      gui_raise_window (actrl->nw.gui, actrl->nw.xwin);
      return;
    }
  } else { /* toggle */
    if (actrl) {
      _actrl_exit (actrl);
      return;
    }
  }

  actrl = calloc (1, sizeof (*actrl));
  if (!actrl)
    return;

  actrl->nw.gui = gui;
  actrl->nw.title = _("xine audio control Window");
  actrl->nw.id = "acontrol";
  actrl->nw.skin = "ACtlBG";
  actrl->nw.wfskin = "ACtlWF";
  actrl->nw.adjust = NULL;
  actrl->nw.wr.x = 200;
  actrl->nw.wr.y = 100;
  actrl->nw.wr.width = 25 + 35 * (NUM_EQ + 1);
  actrl->nw.wr.height = 245;
  actrl->skin = gui_window_new (&actrl->nw);
  if (actrl->skin < 0) {
    free (actrl);
    return;
  }

  if (!actrl->skin) {
    uint32_t u;
    /* noskin fallback */
    for (u = 0; u < 4; u++)
      xitk_image_draw_rectangular_box (actrl->nw.bg, 5, 48 + 14 * u, actrl->nw.wr.width - 10, 2, XITK_DRAW_INNER);
    xitk_image_draw_rectangular_box (actrl->nw.bg, 5, 48 + 14 * u, actrl->nw.wr.width - 10, 2, XITK_DRAW_OUTTER);
    for (u = 5; u < 9; u++)
      xitk_image_draw_rectangular_box (actrl->nw.bg, 5, 48 + 14 * u, actrl->nw.wr.width - 10, 2, XITK_DRAW_INNER);
    xitk_window_set_background_image (actrl->nw.xwin, actrl->nw.bg);
  }
  gui->actrl = actrl;

  actrl->reset = acontrol_reset;

  {
    xitk_slider_widget_t sl = {
      .nw = {
        .wl  = actrl->nw.wl,
        .userdata = actrl,
        .skin_element_name = _actrl_names[0].skin,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE,
        .tips = _actrl_names[0].hint
      },
      .min    = -50,
      .max    = 50,
      .step   = 1,
      .type   = XITK_VSLIDER,
      .callback = _actrl_set_value,
      .motion_callback = _actrl_set_value
    };
    xitk_label_widget_t la = {
      .nw = {
        .wl = actrl->nw.wl,
        .userdata = actrl,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE
      }
    };
    unsigned int u;
    int val;

    for (u = 0; u < NUM_EQ; u++) {
      val = xine_get_param (gui->stream, XINE_PARAM_EQ_30HZ + u);

      if (val < -50)
        val = -50;
      if (val > 50)
        val = 50;
      sl.value = actrl->val[u] = val;

      sl.nw.user_id = u;
      la.label = sl.nw.tips;
      if (actrl->skin) {
        actrl->slid[u] = xitk_slider_create (&sl, gui->skin_config);
        actrl->label[u] = NULL;
      } else {
        actrl->slid[u] = xitk_noskin_slider_create (&sl, 15 + 35 * u,  30, 20, 150);
        actrl->label[u] = xitk_noskin_label_create (&la,  5 + 35 * u, 190, 30,  15, fontname);
      }
      sl.nw.skin_element_name += sizeof (_actrl_names[0]);
      sl.nw.tips += sizeof (_actrl_names[0]);
    }

    val = xine_get_param (gui->stream, XINE_PARAM_AUDIO_AMP_LEVEL);
    if (val < 0)
      val = 0;
    if (val > 200)
      val = 200;
    sl.value  = actrl->val[NUM_EQ] = val;
    sl.min    = 0;
    sl.max    = 200;
    sl.nw.tips = _("Audio amplification");
    sl.nw.user_id = NUM_EQ;
    if (actrl->skin) {
      actrl->slid[NUM_EQ] = xitk_slider_create (&sl, gui->skin_config);
      actrl->label[NUM_EQ] = NULL;
    } else {
      la.label = "^";
      actrl->slid[NUM_EQ] = xitk_noskin_slider_create (&sl, 20 + 35 * NUM_EQ, 30, 25, 150);
      actrl->label[NUM_EQ] = xitk_noskin_label_create (&la, 25 + 35 * NUM_EQ, 190, 15, 15, hboldfontname);
    }
  }

  {
    xitk_labelbutton_widget_t lb = {
      .nw = {
        .wl = actrl->nw.wl,
        .userdata = actrl,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE,
        .skin_element_name = "ACtlFlat",
        .tips = _("Neutral sound")
      },
      .button_type    = CLICK_BUTTON,
      .align          = ALIGN_CENTER,
      .label    = "----",
      .callback = _actrl_flat
    };

    if (actrl->skin) {
      actrl->flat = xitk_labelbutton_create (&lb, gui->skin_config);
    } else {
      actrl->flat = xitk_noskin_labelbutton_create (&lb, 10, 210, 70, 25,
        XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, hboldfontname);
    }

    lb.nw.skin_element_name = "ACtlDismiss";
    lb.nw.tips = _("Close control window");
    lb.label    = _("Dismiss");
    lb.callback = _actrl_close;
    if (actrl->skin) {
      actrl->dismiss = xitk_labelbutton_create (&lb, gui->skin_config);
    } else {
      actrl->dismiss = xitk_noskin_labelbutton_create (&lb, actrl->nw.wr.width - 80, 210, 70, 25,
        XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, hboldfontname);
    }
  }

  actrl->nw.key = xitk_be_register_event_handler ("acontrol", actrl->nw.xwin, _actrl_event, actrl, NULL, NULL);

  gui_raise_window (actrl->nw.gui, actrl->nw.xwin);
}

void acontrol_reset (gGui_t *gui) {
  xui_actrl_t *actrl;
  unsigned int u;

  if (!gui)
    return;

  for (u = 0; u < NUM_EQ; u++)
    xine_set_param (gui->stream, XINE_PARAM_EQ_30HZ + u, 0);
  actrl = gui->actrl;
  if (actrl) {
    for (u = 0; u < NUM_EQ; u++) {
      actrl->val[u] = 0;
      xitk_slider_set_pos (actrl->slid[u], 0);
    }
  }
  if (gui->mixer.type_volume != SOFTWARE_MIXER)
    gui_set_amp_level (gui, 100);
  osd_bar (actrl->nw.gui, _("Neutral sound"), -50, 50, 0, OSD_BAR_POS2);
}

/* Change the current skin.
 * Since audio panel skin is optional, just close here. */
void acontrol_change_skins (xui_actrl_t *actrl, int synthetic) {
  (void)synthetic;
  if (actrl)
    _actrl_exit (actrl);
}
