/***************************************************************************
                              kstviewlabel.cpp
                             ------------------
    begin                : Apr 10 2004
    copyright            : (C) 2000 by cbn
                           (C) 2004 by The University of Toronto
    email                :
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "kstviewlabel.h"

#include "enodes.h"
#include "ksdebug.h"
#include "kst.h"
#include "kstdatacollection.h"
#include "labelrenderer.h"
#include "kstsettings.h"
#include "ksttimers.h"


#include <kdatastream.h>
#include <kglobal.h>
#include <klocale.h>
#include <kpopupmenu.h>

#include <qapplication.h>
#include <qbitmap.h>
#include <qmetaobject.h>
#include <qptrstack.h>

#include <stdlib.h>

/*
TODO:
- Optimize
- Rendering testcases
- Show everyone that this rocks
*/
#define MIN_FONT_SIZE 5

KstViewLabel::KstViewLabel(const QString& txt, KstLJustifyType justify, float rotation)
: KstBorderedViewObject("Label") {
  _dataPrecision = 8;
  _autoResize = false; // avoid madness
  _txt = txt;
  _interpret = true;
  _replace = true;
  _rotation = rotation;
  _justify = justify;
  _fontName = KstApp::inst()->defaultFont();
  _fontSize = 0;
  _absFontSize = _fontSize+KstSettings::globalSettings()->plotFontSize;
  _layoutActions |= Delete | Raise | Lower | RaiseToTop | LowerToBottom | Rename | Edit;
  _standardActions |= Delete | Edit;
  _parsed = 0L;
  reparse();
  computeTextSize(_parsed);
  setDirty(false);
  _autoResize = true;
}


KstViewLabel::KstViewLabel(const QDomElement& e) 
: KstBorderedViewObject(e) {
  
  // some defaults and invariants
  _type = "Label";
  _dataPrecision = 8;
  _autoResize = false; // avoid madness
  _txt = "";
  _interpret = true;
  _replace = true;
  _rotation = 0.0;
  _justify = 0L;
  _fontName = KstApp::inst()->defaultFont();
  _fontSize = 0;
  _absFontSize = _fontSize+KstSettings::globalSettings()->plotFontSize;
  _layoutActions |= Delete | Raise | Lower | RaiseToTop | LowerToBottom | Rename | Edit;
  _standardActions |= Delete | Edit;
  _parsed = 0L;
  reparse();
  
  // read the properties
  bool in_autoResize = false;
  QDomNode n = e.firstChild();
  while (!n.isNull()) {
    QDomElement el = n.toElement(); 
    if (!el.isNull()) {
      if (metaObject()->findProperty(el.tagName().latin1(), true) > -1) {
        if (el.tagName() == "autoResize") {
          in_autoResize = QVariant(el.text()).toBool();
        } else {
          setProperty(el.tagName().latin1(), QVariant(el.text()));  
        }
      }  
    }
    n = n.nextSibling();      
  }
  
  _autoResize = in_autoResize;
}


KstViewLabel::~KstViewLabel() {
  delete _parsed;
  _parsed = 0L;
}


void KstViewLabel::reparse() {
  delete _parsed;
  _parsed = Label::parse(_txt, _interpret);
  setDirty();
}


void KstViewLabel::setText(const QString& text) {
  if (_txt != text) {
    _txt = text;
    reparse(); // calls setDirty()
  }
}


const QString& KstViewLabel::text() const {
  return _txt;
}


void KstViewLabel::setRotation(double rotation) {
  if (_rotation != rotation) {
    setDirty();
    _rotation = rotation;
  }
}


void KstViewLabel::setJustification(KstLJustifyType justify) {
  if (_justify != justify) {
    setDirty();
    _justify = justify;
  }
}


void KstViewLabel::resize(const QSize& size) {
  KstBorderedViewObject::resize(size);
  if (!_parsed) {
    reparse();
  }
  if (_parsed) {
    drawToBuffer(_parsed);
  }
}


int KstViewLabel::ascent() const {
  return _ascent;
}


void KstViewLabel::setFontName(const QString& fontName) {
  if (_fontName != fontName) {
    setDirty();
    _fontName = fontName;
  }
}


const QString& KstViewLabel::fontName() const {
  return _fontName;
}


void KstViewLabel::setInterpreted(bool interpreted) {
  if (_interpret != interpreted) {
    _interpret = interpreted;
    reparse(); // calls setDirty();
  }
}


void KstViewLabel::save(QTextStream &ts, const QString& indent) {
  ts << indent << "<" << type() << ">" << endl;
  KstBorderedViewObject::save(ts, indent + "  ");
  ts << indent << "</" << type() << ">" << endl;
}


void KstViewLabel::setDoScalarReplacement(bool replace) {
  if (replace != _replace) {
    setDirty();
    _replace = replace;
  }
}


void KstViewLabel::drawToBuffer(Label::Parsed *lp) {
#if 0
  if (dirty()) {
    computeTextSize(lp); // hmm this is inefficient
  }
#endif

  setDirty(false);

  _backBuffer.buffer().resize(size());
  _backBuffer.buffer().fill(backgroundColor());
  QPainter p(&_backBuffer.buffer());
  QPen pen;
  pen.setColor(foregroundColor());
  p.setPen(pen);
  drawToPainter(lp, p);
}


void KstViewLabel::drawToPainter(Label::Parsed *lp, QPainter& p) {
  int hJust = KST_JUSTIFY_H(_justify);
  if (QApplication::reverseLayout()) {
    if (hJust == KST_JUSTIFY_H_NONE) {
      hJust = KST_JUSTIFY_H_RIGHT;
    }
  } else {
    if (hJust == KST_JUSTIFY_H_NONE) {
      hJust = KST_JUSTIFY_H_LEFT;
    }
  }

  RenderContext rc(_fontName, _absFontSize, &p);
  rc.setSubstituteScalars(_replace);
  rc.precision = _dataPrecision;
  double rotationRadians = M_PI * (int(_rotation) % 360) / 180;
  double absin = fabs(sin(rotationRadians));
  double abcos = fabs(cos(rotationRadians));

  int tx = 0, ty = 0; // translation

  switch (hJust) {
    case KST_JUSTIFY_H_RIGHT:
      rc.x = -_textWidth / 2;
      tx = size().width() - int(_textWidth * abcos + _textHeight * absin) / 2;
      break;
    case KST_JUSTIFY_H_CENTER:
      rc.x = -_textWidth / 2;
      tx = size().width() / 2;
      break;
    case KST_JUSTIFY_H_NONE:
      abort(); // should never be able to happen
    case KST_JUSTIFY_H_LEFT:
    default:
      rc.x = -_textWidth / 2;
      tx = int(_textWidth * abcos + _textHeight * absin) / 2;
      break;
  }

  switch (KST_JUSTIFY_V(_justify)) {
    case KST_JUSTIFY_V_BOTTOM:
      rc.y = _ascent - _textHeight / 2;
      ty = size().height() - int(_textHeight * abcos + _textWidth * absin) / 2;
      break;
    case KST_JUSTIFY_V_CENTER:
      rc.y = _ascent - _textHeight / 2;
      ty = size().height() / 2;
      break;
    case KST_JUSTIFY_V_NONE:
    case KST_JUSTIFY_V_TOP:
    default:
      rc.y = _ascent - _textHeight / 2;
      ty = int(_textHeight * abcos + _textWidth * absin) / 2;
      break;
  }

  p.translate(tx, ty);
  p.rotate(_rotation);

  rc.xStart = rc.x;
#ifdef BENCHMARK
  QTime t;
  t.start();
#endif
  renderLabel(rc, lp->chunk);
#ifdef BENCHMARK
  kstdDebug() << "render took: " << t.elapsed() << endl;
  t.start();
#endif
  QApplication::syncX();
#ifdef BENCHMARK
  kstdDebug() << "sync X took: " << t.elapsed() << endl;
#endif
}


void KstViewLabel::computeTextSize(Label::Parsed *lp) {
  RenderContext rc(_fontName, _absFontSize, 0L);
  rc.setSubstituteScalars(_replace);
  rc.precision = _dataPrecision;
#ifdef BENCHMARK
  QTime t;
  t.start();
#endif
  renderLabel(rc, lp->chunk);
#ifdef BENCHMARK
  kstdDebug() << "compute (false render) took: " << t.elapsed() << endl;
#endif
  _textWidth = rc.xMax;
  _ascent = rc.ascent;
  _textHeight = 1 + rc.ascent + rc.descent;
}


void KstViewLabel::paint(KstPaintType type, QPainter& p, const QRegion& bounds) {
  if (type == P_PRINT) {
    p.save();
    if (_autoResize) {
      adjustSizeForText(p.window());
    } else {
      computeTextSize(_parsed);
    }
    p.setViewport(geometry());
    p.setWindow(0,0,geometry().width(), geometry().height());
    drawToPainter(_parsed, p);
    //setDirty();
    p.restore();
  } else {  
    if (type == P_UPDATE) {
      setDirty();
    }
    bool d = dirty();
    if (d) {
      if (_autoResize) {
        adjustSizeForText(p.window()); // calls computeTextSize and drawToBuffer
      } else {
        computeTextSize(_parsed);
        drawToBuffer(_parsed);
      }
    }

    if (_transparent) {
      QRegion oldRegion = p.clipRegion();
      p.setClipRegion(oldRegion & clipRegion());
      _backBuffer.paintInto(p, geometry());
      p.setClipRegion(oldRegion);
    } else {
      _backBuffer.paintInto(p, geometry());
    }
  }
  KstBorderedViewObject::paint(type, p, bounds);
}


QRegion KstViewLabel::clipRegion() {
  if (!_transparent) {
    return KstBorderedViewObject::clipRegion();
  }

  if (_clipMask.isNull()) {
    QBitmap bm = _backBuffer.buffer().createHeuristicMask(false); // slow but preserves antialiasing...
    _clipMask = QRegion(bm);
    _clipMask.translate(geometry().topLeft().x(), geometry().topLeft().y());
  }

  return _clipMask;
}


void KstViewLabel::setFontSize(int size) {
  if (_fontSize != size) {
    _absFontSize = size + KstSettings::globalSettings()->plotFontSize;
    if (_absFontSize < MIN_FONT_SIZE) {
      _absFontSize = MIN_FONT_SIZE;
    }
    
    _fontSize = size;
    setDirty();
  }
}


int KstViewLabel::fontSize() const {
  return _fontSize;
}


void KstViewLabel::adjustSizeForText(QRect w) {
  if (_autoResize) {
    double x_s, y_s, s;
    
    x_s = y_s = _fontSize + (double)KstSettings::globalSettings()->plotFontSize;

    int x_pix = w.width();
    int y_pix = w.height();

    if (x_pix < y_pix) {
      x_s *= x_pix/540.0;
      y_s *= y_pix/748.0;
    } else {
      y_s *= y_pix/540.0;
      x_s *= x_pix/748.0;
    }

    s = (x_s + y_s)/2.0;

    if (s < MIN_FONT_SIZE) {
      s = MIN_FONT_SIZE;
    }
    _absFontSize = int(s);
    if (_absFontSize < MIN_FONT_SIZE) {
      _absFontSize = MIN_FONT_SIZE;
    }
  }
 
  if (!_parsed) {
    reparse();
  }

  if (_parsed) {
    computeTextSize(_parsed);
  }
  
  if (_rotation != 0 && _rotation != 180) {
    QPointArray pts(4);
    pts[0] = QPoint(0, 0);
    pts[1] = QPoint(0, _textHeight);
    pts[2] = QPoint(_textWidth, 0);
    pts[3] = QPoint(_textWidth, _textHeight);
    double theta = M_PI * (int(_rotation) % 360) / 180;
    double sina = sin(theta);
    double cosa = cos(theta);
    QWMatrix m(cosa, sina, -sina, cosa, 0, 0);

    QPoint to;
    to.setX(_textWidth / 2);
    to.setY(_textHeight / 2);
    pts.translate(-to.x(), -to.y());
    pts = m.map(pts);
    pts.translate(to.x(), to.y());

    if (_parent) {
      QRect r(position(), pts.boundingRect().size());
      resize(r.intersect(_parent->geometry()).size());
    } else {
      resize(pts.boundingRect().size());
    }
  } else {
    QSize sz(_textWidth, _textHeight);
    if (_parent) {
      QRect r(position(), sz);
      resize(r.intersect(_parent->geometry()).size());
    } else {
      resize(sz);
    }
  }
}


bool KstViewLabel::layoutPopupMenu(KPopupMenu *menu, const QPoint& pos, KstViewObjectPtr topLevelParent) {
  KstViewObject::layoutPopupMenu(menu, pos, topLevelParent);
  //menu->insertItem(i18n("&Adjust Size"), this, SLOT(adjustSizeForText()));
  return true;
}


bool KstViewLabel::interpreted() const {
  return _interpret;
}


bool KstViewLabel::doScalarReplacement() const {
  return _replace;
}


void KstViewLabel::setAutoResize(bool on) {
    _autoResize = on;
}


bool KstViewLabel::autoResize() const {
  return _autoResize;
}


KstViewObjectPtr create_KstViewLabel() {
  return KstViewObjectPtr(new KstViewLabel(QString::null));
}


KstViewObjectFactoryMethod KstViewLabel::factory() const {
  return &create_KstViewLabel;
}


void KstViewLabel::writeBinary(QDataStream& str) {
  KstBorderedViewObject::writeBinary(str);
  str << _rotation << _txt << _fontName << _replace << _interpret << _justify;
}


void KstViewLabel::readBinary(QDataStream& str) {
  KstBorderedViewObject::readBinary(str);
  bool b;
  str >> _rotation >> _txt >> _fontName
      >> b;
  _replace = b;
  str >> b;
  _interpret = b;
  str >> b;
  _justify = b;
  reparse(); // FIXME: this should go away and updateFromAspect should be
             //        overridden.  this hack fails when dragging a label
             //        from a smaller parent to a larger one
}


double KstViewLabel::rotation() const { 
  return _rotation; 
}


QMap<QString, QVariant> KstViewLabel::widgetHints(const QString& propertyName) const {
  QMap<QString, QVariant> map = KstBorderedViewObject::widgetHints(propertyName);
  if (!map.empty()) {
    return map;  
  }
  
  if (propertyName == "text") {
    map.insert(QString("_kst_widgetType"), QString("QLineEdit"));
    map.insert(QString("_kst_label"), i18n("Text"));  
  } else if (propertyName == "rotation") {
    map.insert(QString("_kst_widgetType"), QString("KDoubleSpinBox"));
    map.insert(QString("_kst_label"), i18n("Rotation")); 
    map.insert(QString("minValue"), -90.0);
    map.insert(QString("maxValue"), 90.0);
    map.insert(QString("lineStep"), 1.0);
  } else if (propertyName == "font") {
    map.insert(QString("_kst_widgetType"), QString("KFontCombo"));
    map.insert(QString("_kst_label"), i18n("Font"));
  } else if (propertyName == "foreColor") {
    map.insert(QString("_kst_widgetType"), QString("KColorButton"));
    map.insert(QString("_kst_label"), i18n("Font color"));   
  } else if (propertyName == "backColor") {
    map.insert(QString("_kst_widgetType"), QString("KColorButton"));
    map.insert(QString("_kst_label"), i18n("Background color"));   
  } else if (propertyName == "dataPrecision") {
    map.insert(QString("_kst_widgetType"), QString("QSpinBox"));
    map.insert(QString("_kst_label"), i18n("Data precision"));   
    map.insert(QString("minValue"), 0);
    map.insert(QString("maxValue"), 16);
  } else if (propertyName == "fontSize") {
    map.insert(QString("_kst_widgetType"), QString("QSpinBox"));
    map.insert(QString("_kst_label"), i18n("Font size"));
    map.insert(QString("minValue"), MIN_FONT_SIZE - (double)KstSettings::globalSettings()->plotFontSize);   
  } else if (propertyName == "transparent") {
    map.insert(QString("_kst_widgetType"), QString("QCheckBox"));
    map.insert(QString("_kst_label"), QString::null);
    map.insert(QString("text"), i18n("Transparent fill"));   
  } else if (propertyName == "horizJustify") {
    map.insert(QString("_kst_widgetType"), QString("HJustifyCombo"));
    map.insert(QString("_kst_label"), i18n("Horizontal Justification"));
  } else if (propertyName == "vertJustify") {
    map.insert(QString("_kst_widgetType"), QString("VJustifyCombo"));
    map.insert(QString("_kst_label"), i18n("Vertical Justification"));
  } else if (propertyName == "autoResize") {
    map.insert(QString("_kst_widgetType"), QString("QCheckBox"));
    map.insert(QString("_kst_label"), QString::null);
    map.insert(QString("text"), i18n("Resize label for text"));   
  }
  return map;
}


void KstViewLabel::setDataPrecision(int prec) {
  int n;
  
  if (prec < 0) {
    n = 0;  
  } else if (prec > 16) {
    n = 16;  
  } else {
    n = prec;  
  }
  
  if (n != _dataPrecision) {
    setDirty();
    _dataPrecision = n;
  }
}


int KstViewLabel::dataPrecision() const {
  return _dataPrecision;
}


void KstViewLabel::setTransparent(bool transparent) {
  KstViewObject::setTransparent(transparent);  
}


bool KstViewLabel::transparent() const {
  return KstViewObject::transparent();  
}


int KstViewLabel::horizJustifyWrap() const {
  Q_UINT8 justify = KST_JUSTIFY_H(justification());
  switch (justify) {
    case KST_JUSTIFY_H_LEFT:
      return 0;
      break;
    case KST_JUSTIFY_H_RIGHT:
      return 1;
      break;
    case KST_JUSTIFY_H_CENTER:
      return 2;
      break;
    default:
      return 0;  
  }
}


void KstViewLabel::setHorizJustifyWrap(int justify) {
  Q_UINT8 justifySet;
  
  switch (justify) {
    case 0:
      justifySet = KST_JUSTIFY_H_LEFT;
      break;  
    case 1:
      justifySet = KST_JUSTIFY_H_RIGHT;
      break;
    case 2:
      justifySet = KST_JUSTIFY_H_CENTER;
      break;
    default:
      justifySet = KST_JUSTIFY_H_LEFT;
  }
  setJustification(SET_KST_JUSTIFY(justifySet, KST_JUSTIFY_V(justification())));
}
    
        
int KstViewLabel::vertJustifyWrap() const {
  Q_UINT8 justify = KST_JUSTIFY_V(justification());
  switch (justify) {
    case KST_JUSTIFY_V_TOP:
      return 0;
      break;
    case KST_JUSTIFY_V_BOTTOM:
      return 1;
      break;
    case KST_JUSTIFY_V_CENTER:
      return 2;
      break;
    default:
      return 0;  
  }
}


void KstViewLabel::setVertJustifyWrap(int justify) {
  Q_UINT8 justifySet;
  
  switch (justify) {
    case 0:
      justifySet = KST_JUSTIFY_V_TOP;
      break;  
    case 1:
      justifySet = KST_JUSTIFY_V_BOTTOM;
      break;
    case 2:
      justifySet = KST_JUSTIFY_V_CENTER;
      break;
    default:
      justifySet = KST_JUSTIFY_V_TOP;
  }
  setJustification(SET_KST_JUSTIFY(KST_JUSTIFY_H(justification()), justifySet));
}
    

#include "kstviewlabel.moc"
// vim: ts=2 sw=2 et
