/***************************************************************************
                              kst2dplot.h
                             -------------
    begin                : Mar 28, 2004
    copyright            : (C) 2004 The University of Toronto
    email                :
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KST2DPLOT_H
#define KST2DPLOT_H

#include "kstbackbuffer.h"
#include "kstbasecurve.h"
#include "kstimage.h"
#include "kstlabel.h"
#include "kstviewwidget.h"


class KstLegend;
class Kst2DPlot;
typedef KstObjectList<KstSharedPtr<Kst2DPlot> > Kst2DPlotList;

typedef enum { AUTO, AC, FIXED, AUTOUP, NOSPIKE } KstScaleModeType;

struct KstPlotScale {
  double xmin;
  double xmax;
  double ymin;
  double ymax;
  KstScaleModeType xscalemode;
  KstScaleModeType yscalemode;
  bool xlog, ylog;
};

typedef enum { INACTIVE, XY_ZOOMBOX, Y_ZOOMBOX, X_ZOOMBOX, LABEL_TOOL, LAYOUT_TOOL } KstMouseModeType;
struct KstMouse {
  KstMouse() { mode = INACTIVE; label = -1; minMove = 2; }
  KstMouseModeType mode;
  int label, minMove;
  QPoint lastLocation, pressLocation;
  QRect plotGeometry;
  bool zooming() const;
  void zoomStart(KstMouseModeType t, const QPoint& location);
  void zoomUpdate(KstMouseModeType t, const QPoint& location);
  QRect mouseRect() const;
  bool rectBigEnough() const;
};


class Kst2DPlot : public KstPlotBase {
  Q_OBJECT
public:
  Kst2DPlot(const QString& in_tag = "SomePlot",
          KstScaleModeType yscale = AUTO,
          KstScaleModeType xscale = AUTO,
          double xmin = 0, double ymin = 0,
          double xmax = 1, double ymax = 1);
  Kst2DPlot(QDomElement& e);
  virtual ~Kst2DPlot();

  static Kst2DPlotList globalPlotList();
  static bool checkRange(double &min_in, double &max_in, bool bIsLog);
  static void genAxisTickLabelFullPrecision(QString& label, double z, bool isLog);
  static void genAxisTickLabelDifference(QString& label, double zbase, double zvalue, bool isLog);
  static void genAxisTickLabel(QString& label, double z, bool isLog);
  static void genAxisTickLabels(QPainter& p, double Min, double Max, double Org, double Tick,
                              bool bLog, KstLabel* Label, bool &bDelta,
                              double &dMaxWidth, double &dMaxHeight, QStringList &labelList,
                              int& iIndexLo, int& iIndexHi);

  virtual UpdateType update();
  virtual void save(QTextStream& ts);
  virtual void saveTag(QTextStream& ts);

  virtual bool popupMenu(KPopupMenu *menu, const QPoint& pos, KstViewObjectPtr topLevelParent);
  virtual bool layoutPopupMenu(KPopupMenu *menu, const QPoint& pos, KstViewObjectPtr topLevelParent);

  void drawDotAt(QPainter& p, double x, double y);
  double xInternalAlignment();
  void addCurve(KstBaseCurvePtr curve);
  void addLabel(KstLabel* label);
  void removeCurve(KstBaseCurvePtr curve);
  void setXScaleMode(KstScaleModeType scalemode);
  void setYScaleMode(KstScaleModeType scalemode);
  
  /** Set the scale */
  void setScale(double xmin, double ymin,
                double xmax, double ymax);
  /** Set the X scale */
  bool setXScale(double xmin, double xmax);
  /** Set the y scale */
  bool setYScale(double ymin, double ymax);

  /** Set the log_scale */
  bool setLScale(double xmin, double ymin,
                double xmax, double ymax);
  /** Set the X log_scale */
  bool setLXScale(double xmin, double xmax);
  /** Set the y log_scale */
  bool setLYScale(double ymin, double ymax);
  /** Push the scale setting (X,Y, mode) onto scale list */
  void pushScale();
  /** Pop the scale settings off of scale list: true if stack not empty */
  bool popScale();

  void setLog(bool x_log, bool y_log);
  bool isXLog() const;
  bool isYLog() const;

  void getScale(double& xmin, double& ymin, double& xmax, double& ymax) const;
  void getLScale(double& xmin, double& ymin, double& xmax, double& ymax) const;
  
  KstScaleModeType getXScaleMode() const;
  KstScaleModeType getYScaleMode() const;

  void setXLabel(const QString& xlabel);
  void setYLabel(const QString& ylabel);
  void setTopLabel(const QString& toplabel);
  void initFonts(const QFont& font, int font_size = 0);

  QRect GetPlotRegion() const;
  QRect GetWinRegion() const;
  QRect GetTieBoxRegion() const;
  QRect GetPlotAndAxisRegion() const;

  bool isTied() const;
  void toggleTied();
  void setTied(bool is_tied);

  /** Labels */
  KstLabel *XLabel;
  KstLabel *YLabel;
  KstLabel *TopLabel;
  KstLabel *XTickLabel;
  KstLabel *YTickLabel;
  KstLabel *XFullTickLabel;
  KstLabel *YFullTickLabel;
  KstLegend *Legend;

  double _width;
  double _height;
  double _pos_x;
  double _pos_y;

  /** Arbitrary Labels */
  KstLabelList _labelList;

  KstBaseCurveList Curves;

  void GenerateDefaultLabels();

  /* kstview tells kstplot where to offset the plot to */
  void setPixRect(const QRect& RelPlotRegion, const QRect& RelWinRegion, const QRect& RelPlotAndAxisRegion);

  virtual void resize(const QSize&);
  virtual void parentResized();

  virtual bool mouseHandler() const;
  virtual void mouseMoveEvent(QWidget *view, QMouseEvent *e);
  virtual void mousePressEvent(QWidget *view, QMouseEvent *e);
  virtual void mouseReleaseEvent(QWidget *view, QMouseEvent *e);
  virtual void keyPressEvent(QWidget *view, QKeyEvent *e);
  virtual void keyReleaseEvent(QWidget *view, QKeyEvent *e);
  virtual void dragEnterEvent(QWidget *view, QDragEnterEvent *e);
  virtual void dropEvent(QWidget *view, QDropEvent *e);
  virtual void dragMoveEvent(QWidget *view, QDragMoveEvent *e);
  virtual void wheelEvent(QWidget *view, QWheelEvent *e);

  void cancelZoom(QWidget *view);
  bool moveSelfHorizontal(bool left);
  bool moveSelfVertical(bool up);
  bool zoomSelfHorizontal(bool in);
  bool zoomSelfVertical(bool in);
  double xLeft() const;
  void setHasFocus(bool hasFocus);
  void removeFocus(QPainter& p);
  void moveUp(KstViewWidget *);
  void moveDown(KstViewWidget *);
  void moveLeft(KstViewWidget *);
  void moveRight(KstViewWidget *);
  void xZoomIn(KstViewWidget *);
  void yZoomIn(KstViewWidget *);
  void xZoomOut(KstViewWidget *);
  void yZoomOut(KstViewWidget *);
  void xZoomMax(KstViewWidget *);
  void yZoomMax(KstViewWidget *);
  void zoomMax(KstViewWidget *);
  void xLogSlot(KstViewWidget *);
  void yLogSlot(KstViewWidget *);
  void zoomPrev(KstViewWidget *);
  void yZoomAc(KstViewWidget *);
  void zoomSpikeInsensitiveMax(KstViewWidget *);

  //Plot Markers
  bool setPlotMarker(const double xValue);
  bool removePlotMarker(const double xValue);
  void setPlotMarkerList(const QValueList<double>& newMarkers);
  const QValueList<double> plotMarkers(const double minX, const double maxX);
  const QValueList<double>& plotMarkers();
  bool nextMarker(const double currentPosition, double& marker);
  bool prevMarker(const double currentPosition, double& marker);
  void moveToNextMarker(KstViewWidget *);
  void moveToPrevMarker(KstViewWidget *);
  void moveSelfToMarker(bool next);
  //curve related Plot Marker functions
  void setCurveToMarkers(KstBaseCurvePtr curve, bool risingDetect, bool fallingDetect);
  bool hasCurveToMarkers() const;
  void removeCurveToMarkers();
  KstBaseCurvePtr curveToMarkers() const;
  bool curveToMarkersRisingDetect() const;
  bool curveToMarkersFallingDetect() const;

  //Images
  void addImage(KstImagePtr inImage, bool set_dirty = true);
  void removeImage(KstImagePtr inImage, bool set_dirty = true);
  void removeAllImages(bool set_dirty = true);
  bool hasImage(KstImagePtr inImage) const;

signals:
  void modified();

public slots:
  void copy();

  virtual void edit();
  void draw(); // draw into back buffer
  void draw(QPainter &p, KstPaintType type, double resolutionEnhancement = 1); // This actually does the drawing
  virtual void paint(KstPaintType type, QPainter& p);
  void editCurve(int id);
  void editImage(int id);
  void matchAxis(int id);
  void fitCurve(int id);
  void filterCurve(int id);
  void removeCurve(int id);
  void removeImage(int id);
  void setDirty();

protected slots:
  void menuMoveUp();
  void menuMoveDown();
  void menuMoveLeft();
  void menuMoveRight();
  void menuXZoomIn();
  void menuXZoomOut();
  void menuYZoomIn();
  void menuYZoomOut();
  void menuXZoomMax();
  void menuYZoomMax();
  void menuZoomMax();
  void menuXLogSlot();
  void menuYLogSlot();
  void menuZoomPrev();
  void menuYZoomAc();
  void menuZoomSpikeInsensitiveMax();
  void menuNextMarker();
  void menuPrevMarker();

protected:
  virtual KstViewObjectFactoryMethod factory() const;

private:
  template<class T> void updateTiedPlots(void (Kst2DPlot::*method)(T), T arg);
  template<class T> void updateTiedPlots(bool (Kst2DPlot::*method)(T), T arg);

  void updateMousePos(QPoint pos);
  void highlightNearestDataPoint(bool bRepaint, QWidget *view, const QPoint& pos);
  void updateTieBox(QPainter&);
  int labelNumber(QMouseEvent *e);
  bool legendUnder(QMouseEvent *e);
  KstMouseModeType globalZoomType() const;
  void setCursorForMode(QWidget *view, KstMouseModeType mode);
  void setCursorForMode(QWidget *view); // gets the mode from the mouse
  void zoomRectUpdate(QWidget *view, KstMouseModeType t, int x, int y);
  inline void commonConstructor(const QString& in_tag,
                                KstScaleModeType yscale,
                                KstScaleModeType xscale,
                                double xmin,
                                double ymin,
                                double xmax,
                                double ymax,
                                bool x_log = false,
                                bool y_log = false);
  void setBorders(double& xleft_bdr_px, double& xright_bdr_px,
                  double& ytop_bdr_px, double& ybot_bdr_px,
                  double XTick, double Xorg, double YTick, double Yorg,
                  QPainter &p);
  void setTicks(double& tick, double& org,
                double max, double min, bool is_log);

  /** range and domain of plot: not plot dimentions */
  double XMin, XMax, YMin, YMax;
  double _copy_x, _copy_y;

  bool _xLog : 1, _yLog : 1;
  bool _isTied : 1;
  bool _hasFocus : 1;
  bool _dirty : 1;
  bool _zoomPaused : 1;
  bool _highlighting : 1; // lock to prevent reentrancy on data mode
                          // (shouldn't this be avoidable?)
  bool _curveToMarkersRisingDetect : 1;
  bool _curveToMarkersFallingDetect : 1;

  int _draggableLabel : 15; // I think this should be enough

  KstScaleModeType _xScaleMode, _yScaleMode;

  QPtrList<KstPlotScale> _plotScaleList;

  /** Stores border limits.  Set by KstPlot::paint().
      Stored here to be Used to determine mouse mode */
  QRect PlotRegion;
  QRect WinRegion;
  QRect PlotAndAxisRegion;

  void updateScale();

  KstMouse _mouse;
  QPoint _draggablePoint;

  QMap<int, QString> _curveEditMap, _curveFitMap, _curveRemoveMap, _imageEditMap, _imageRemoveMap;
  QMap<int, QGuardedPtr<Kst2DPlot> > _plotMap;

  KstBackBuffer _buffer;
  QGuardedPtr<KstViewWidget> _menuView;
  QSize _oldSize;
  double _oldXAlignment;
  double _m_X, _b_X, _m_Y, _b_Y;

  //the list of more than one image
  KstImageList _images;

  /** Plot Markers.  This needs to remain sorted **/
  QValueList<double> _plotMarkers;
  //the curve used for auto marker generation
  KstBaseCurvePtr _curveToMarkers;
  void updateMarkersFromCurve();  //auto-generates the markers from _curveToMarkers

  /** helper functions for draw(...) **/
  void plotLabels(QPainter& p, int x_px, int y_px, double xleft_bdr_px, double ytop_bdr_px);
  void plotAxes(QPainter& p, QRect& plotRegion, double x_max, double y_max, double x_min, double y_min,
                        double XTick, double Xorg, double xleft_bdr_px, double xright_bdr_px, double x_orig_px, double xtick_px, double xtick_len_px, int x_px,
                        double YTick, double Yorg, double ytop_bdr_px, double ybot_bdr_px, double y_orig_px, double ytick_px, double ytick_len_px, int y_px);
  void plotCurves(QPainter& p,
                           double Lx, double Hx, double Ly, double Hy,
                           double m_X, double m_Y, double b_X, double b_Y, int penWidth);
  void plotImages(QPainter& p,
                           double Lx, double Hx, double Ly, double Hy,
                           double m_X, double m_Y, double b_X, double b_Y,
                           double x_max, double y_max, double x_min, double y_min); 
  void plotMarkers(QPainter& p,
                           double b_X, double b_Y, double x_max, double x_min,
                           double y_px, double ytop_bdr_px, double ybot_bdr_px);
};

typedef KstSharedPtr<Kst2DPlot> Kst2DPlotPtr;
typedef KstObjectMap<Kst2DPlotPtr> Kst2DPlotMap;

template<class T>
void Kst2DPlot::updateTiedPlots(void (Kst2DPlot::*method)(T), T arg) {
  if (isTied()) {
    Kst2DPlotList pl = globalPlotList();

    for (Kst2DPlotList::Iterator i = pl.begin(); i != pl.end(); ++i) {
      Kst2DPlotPtr p = *i;
      if (p->isTied() && p.data() != this) {
        (p->*method)(arg);
        p->pushScale();
        p->setDirty();
      }
    }
  }
}

template<class T>
void Kst2DPlot::updateTiedPlots(bool (Kst2DPlot::*method)(T), T arg) {
  if (isTied()) {
    Kst2DPlotList pl = globalPlotList();

    for (Kst2DPlotList::Iterator i = pl.begin(); i != pl.end(); ++i) {
      Kst2DPlotPtr p = *i;
      if (p->isTied() && p.data() != this) {
        if ((p->*method)(arg)) {
          p->pushScale();
          p->setDirty();
        }
      }
    }
  }
}

#endif
// vim: ts=2 sw=2 et
