/***************************************************************************
                          splineitem.cpp  -  description
                             -------------------
    begin                : Wed Feb 27 2002
    copyright            : (C) 2005 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <math.h>
#include <stdlib.h>
#include <qlibrary.h>
#include <qtextstream.h>
#include <klocale.h>
#include <ksimpleconfig.h>
#include "arrayitem.h"
#include "fitpack.h"
#include "funitem.h"
#include "kgraph.h"
#include "splineitem.h"
#include "utils.h"
#ifndef KPL_CLASSES_ONLY
#include <qlistview.h>
#include "kpldoc.h"
#include "splinedlg.h"
#endif

SplineItem::SplineItem(int k) : logxo(false), kdeg(k), nk(0), nderiv(0),
 kdego(k), nko(0), nderivo(0), xmin(0.0), xmax(0.0), dx(0.0), xlow(0.0),
 xmino(0.0), xmaxo(0.0), dxo(0.0), xlowo(0.0)
{
}

SplineItem::SplineItem(const SplineItem& f) :
 ScaledItem(f), logxo(f.logxo), kdeg(f.kdeg), nk(f.nk), nderiv(f.nderiv),
 kdego(f.kdego), nko(f.nko), nderivo(f.nderivo), xmin(f.xmin), xmax(f.xmax),
 dx(f.dx), xlow(f.xlow), xmino(f.xmino), xmaxo(f.xmaxo), dxo(f.dxo),
 xlowo(f.xlowo), xv(f.xv), yv(f.yv), t(f.t), c(f.c)
{
  xv.detach();
  yv.detach();
  c.detach();
  t.detach();
}

SplineItem::SplineItem(KplNamespace::AutoStruct* aut) : ScaledItem(aut),
 logxo(false), kdeg(3), nk(0), nderiv(0), kdego(3), nko(0), nderivo(0),
 xmin(0.0), xmax(0.0), dx(0.0), xlow(0.0), xmino(0.0), xmaxo(0.0), dxo(0.0),
 xlowo(0.0)
{
}

SplineItem::SplineItem(KSimpleConfig* plo, KplNamespace::AutoStruct* aut,
                       const KURL&) :
 ScaledItem(plo, aut), logxo(false), kdego(3), nko(0), nderivo(0), xmino(0.0),
 xmaxo(0.0), dxo(0.0), xlowo(0.0)
{
  kdeg = plo->readNumEntry("kdeg", 3);
  nk = plo->readNumEntry("nk");
  nderiv = plo->readNumEntry("nderiv");
  xmin = plo->readDoubleNumEntry("xmin");
  xmax = plo->readDoubleNumEntry("xmax", xmin + 1.0);
  dx = plo->readDoubleNumEntry("dx");
  xlow = plo->readDoubleNumEntry("xlow", xmin);
  t.resize(nk);
  QString s;
  for (int i = 0; i < nk; i++)
    t[i] = plo->readDoubleNumEntry(s.sprintf("t%i", i));
  int nc = QMAX(0, nk - kdeg - 1);
  c.resize(nc);
  for (int i = 0; i < nc; i++)
    c[i] = plo->readDoubleNumEntry(s.sprintf("c%i", i));
}

SplineItem::SplineItem(bool act, int fill, int sym, const QString& col,
                       double xn, double yn, double xmi, double xma,
                       double d_x, int k, int deriv, double low,
                       double relSize) :
 ScaledItem(act, fill, sym, col, xn, yn, relSize), logxo(false), kdeg(k),
 nk(0), nderiv(deriv), kdego(k), nko(0), nderivo(0), xmin(xmi), xmax(xma),
 dx(d_x), xlow(low), xmino(0.0), xmaxo(0.0), dxo(0.0), xlowo(0.0)
{
}

SplineItem::~SplineItem()
{
}

SplineItem& SplineItem::operator=(const SplineItem& f)
{
  if (this != &f) {
    *(ScaledItem*)this = f;
    logxo = f.logxo;
    kdeg = f.kdeg;
    nk = f.nk;
    nderiv = f.nderiv;
    kdego = f.kdego;
    nko = f.nko;
    nderivo = f.nderivo;
    xmin = f.xmin;
    xmax = f.xmax;
    dx = f.dx;
    xlow = f.xlow;
    xmino = f.xmino;
    xmaxo = f.xmaxo;
    dxo = f.dxo;
    xlowo = f.xlowo;
    xv = f.xv;
    xv.detach();
    yv = f.yv;
    yv.detach();
    t = f.t;
    t.detach();
    c = f.c;
    c.detach();
  }
  return *this;
}

KplItem::ItemTypes SplineItem::iType() const
{
  return Spline;
}

void SplineItem::draw(KplGraph* g)
{
  if (nk && active) {
    setProperties(g);
    double sav = g->relSize();
    g->setRelSize(relsiz * sav);
    int n = calcTable(g->logx);
    g->plArray(xv.data(), yv.data(), fx, fy, n, fillStyle, true, x0, y0);
    g->setRelSize(sav);
  }
}

#ifndef KPL_CLASSES_ONLY
void SplineItem::writePlo(KSimpleConfig* plo, const KURL& url, bool _abs,
                       KplDoc* m) const
{
  plo->writeEntry("Type", "SPLINEITEM");
  ScaledItem::writePlo(plo, url, _abs, m);
  plo->writeEntry("kdeg", kdeg);
  plo->writeEntry("nk", nk);
  plo->writeEntry("nderiv", nderiv);
  char frm = m->options()->format;
  int prec = m->options()->prec;
  plo->writeEntry("xmin", xmin, true, false, frm, prec);
  plo->writeEntry("xmax", xmax, true, false, frm, prec);
  plo->writeEntry("dx", dx, true, false, frm, prec);
  plo->writeEntry("xlow", xlow, true, false, frm, prec);
  QString s;
  for (int i = 0; i < nk; i++)
    plo->writeEntry(s.sprintf("t%i", i), t[i]);
  for (int i = 0; i < (nk - kdeg - 1); i++)
    plo->writeEntry(s.sprintf("c%i", i), c[i]);
}

void SplineItem::setText(QListViewItem* it, bool*, bool* funcs) const
{
  *funcs = true;
  it->setText(1, i18n("Spline"));
  KGraph g;
  it->setPixmap(2, g.pixmap(symb, color));
  QString s = i18n("degree") + " " + QString::number(kdeg);
  if (nderiv) {
    s += ", ";
    switch (nderiv) {
      case -1:
        s += i18n("integral");
        break;
      case 1:
        s += i18n("1st derivative");
        break;
      case 2:
        s += i18n("2nd derivative");
        break;
      case 3:
        s += i18n("3rd derivative");
        break;
      case 4:
        s += i18n("4th derivative");
        break;
      case 5:
        s += i18n("5th derivative");
    }
  }
  it->setText(2, s);
}

int SplineItem::editItem(QWidget* parent, KplDoc* m, int)
{
  SplineDlg dlg(parent, m, this);
  return dlg.exec();
}

void SplineItem::exportTable(QTextStream& ts, KplDoc* m) const
{
  int n = calcTable(logxo);
  for (int i = 0; i < n; i++)
    ts << m->number(fx * xv[i]) << m->separator()
       << m->number(fy * yv[i]) << "\n";
}
#endif

KplItem* SplineItem::copy() const
{
  return new SplineItem(*this);
}

void SplineItem::expoItem(int* iext, int* ieyt, double* fxt, double* fyt) const
{
  if (nk) {
    Utils::expo(QMAX(fabs(xmin), fabs(xmax)), iext, fxt);
    double xmi, xma, ymi, yma;
    minMax(&xmi, &xma, &ymi, &yma);
    Utils::expo(QMAX(fabs(ymi), fabs(yma)), ieyt, fyt);
  }
}

void SplineItem::minMax(double* xmi, double* xma, double* ymi, double* yma) const
{
  if (nk) {
    *xmi = xmin + x0;
    *xma = xmax + x0;
    int n = calcTable(logxo);
    Utils::minMaxFile(ymi, yma, yv.data(), n);
    *ymi += y0;
    *yma += y0;
  }
}

int SplineItem::roots(double* zero, int mest, int* nz, int nu,
                      double offset) const
{
  int ier;
  FitPack::sproot(t.data(), nk, c.data(), zero, mest, nz, &ier, nu, offset);
  return ier;
}

int SplineItem::interpolation(const double* x0, const double *y0, int np,
                              double xmi, double xma)
{
  QMemArray<double> e0(np);
  e0.fill(1.0);
  int nkt;
  double chisq;
  return fit(x0, y0, e0, np, xmi, xma, &chisq, &nkt);
}

int SplineItem::fit(const ArrayItem* ad, double xmi, double xma, double* chisq,
                    int* nkt, double f, KplNamespace::DataErrorStruct* err,
                    double* avgErr)
{
  return fit(&ad->x[ad->ix][ad->istart], &ad->x[ad->iy][ad->istart],
             &ad->x[ad->ie][ad->istart], ad->n, xmi, xma, chisq, nkt, f, err,
             avgErr);
}

int SplineItem::fit(const double* x0, const double *y0, const double* e0,
                    int np, double xmi, double xma, double* chisq, int* nkt,
                    double f, KplNamespace::DataErrorStruct* err,
                    double* avgErr)
{
  QMemArray<double> x(np);
  QMemArray<double> y(np);
  QMemArray<double> w(np);
  int ier = sortedArrays(x0, y0, e0, np, err, x.data(), y.data(), w.data());
  if (ier)
    return ier;
  if (f)
    for (int i = 0; i < np; i++)
      w[i] = 1.0 / w[i];
  int nest = np + kdeg + 1;
  QMemArray<double> tf(nest);
  QMemArray<double> cf(nest);
  QMemArray<int> iwrk(nest);
  int lwrk = np * (kdeg + 1) + nest * (7 + 3 * kdeg);
  QMemArray<double> wrk(lwrk);
  FitPack::curfit(0, np, x.data(), y.data(), w.data(), xmi, xma, kdeg,
                  f * f * np, nest, nkt, tf.data(), cf.data(), chisq,
                  wrk.data(), lwrk, iwrk.data(), &ier);
  if (ier < 4) {
    xmin = QMAX(xmin, xmi);
    xmax = QMIN(xmax, xma);
    nk = *nkt;
    t.duplicate(tf, nk);
    c.duplicate(cf, nk - kdeg - 1);
    xv.detach();
    xv.resize(0);
    if (avgErr) {
      *avgErr = 0.0;
      if (f) {
        QMemArray<double> fv(np);
        FitPack::splev(t.data(), nk, c.data(), kdeg, x.data(), fv.data(), np,
                       &ier);
        for (int i = 0; i < np; i++) {
          double e = y[i] - fv[i];
          *avgErr += e * e;
        }
        *avgErr = sqrt(*avgErr / np);
      }
    }
  }
  return ier;
}

int SplineItem::calcValues(double* x, double* y, int np, int deriv,
                           double x0) const
{
  int ier = 0;
  if (deriv >= 0)
    FitPack::splder(t.data(), nk, c.data(), kdeg, deriv, x, y, np, &ier);
  else {
    QMemArray<double> wrk(nk);
    for (int i = 0; i < np; i++)
      y[i] = FitPack::splint(t.data(), nk, c.data(), kdeg, x0, x[i],
                             wrk.data());
  }
  return ier;
}

int SplineItem::sortedArrays(const double* x0, const double* y0,
                             const double* e0, int np,
                             KplNamespace::DataErrorStruct* err, double* x,
                             double* y, double* e)
{
  int ier = 0;
  double (*fktErrMod)(double, const double*);
  QLibrary* lib;
  bool errcol = true;
  if (err) {
    errcol = err->fitErrCol;
    if (!errcol) {
      if (!FunItem::getFuncAddr(err->errModPath, err->errModName, &lib,
                               &fktErrMod))
        return 5;
    }
  }
  int nc = errcol ? 3 : 2;
  double* tmp = new double[nc * np];
  int j = 0;
  for (int i = 0; i < np; i++) {
    tmp[j++] = x0[i];
    tmp[j++] = y0[i];
    if (errcol)
      tmp[j++] = e0[i];
  }
  qsort(tmp, np, nc * sizeof(double), cmpasc);
  j = 0;
  for (int i = 0; i < np; i++) {
    x[i] = tmp[j++];
    y[i] = tmp[j++];
    if (errcol)
      e[i] = tmp[j++];
  }
  delete [] tmp;
  for (int i = 0; i < np; i++) {
    if (!errcol)
      e[i] = fktErrMod(err->errModArg ? y[i] : x[i], err->pErrMod);
    if (e[i] <= 0.0) {
      ier = 4;
      break;
    }
  }
  if (!errcol)
    delete lib;
  return ier;
}

int SplineItem::cmpasc(const void *e1, const void *e2)
{
  double a1 = *((double*) e1);
  double a2 = *((double*) e2);
  if (a1 < a2)
    return -1;
  if (a1 > a2)
    return 1;
  return 0;
}

int SplineItem::calcTable(bool logx) const
{
  bool newv = xv.isEmpty();
  double d;
  int n;
  if (dx) {
    n = int(fabs((xmax - xmin) / dx)) + 1;
    d = logx ? (log10(xmax / xmin) / (n - 1)) : dx;
  } else {
    n = 101;
    d = 0.01 * (logx ? log10(xmax / xmin) : (xmax - xmin));
  }
  int i;
  if (newv || (logx != logxo) || (kdeg != kdego) || (nk != nko) ||
      (nderiv != nderivo) || (xmin != xmino) || (xmax != xmaxo) ||
      (d != dxo) || (xlow != xlowo)) {
    xv.detach();
    xv.resize(n);
    yv.detach();
    yv.resize(n);
    int n1 = n - 1;
    for (i = 0; i < n; i++)
      xv[i] = (i == n1) ? xmax : (logx ? pow(10.0, log10(xmin) + i * d) :
                                         (xmin + i * d));
    calcValues(xv.data(), yv.data(), n, nderiv, xlow);
  }
  logxo = logx;
  kdego = kdeg;
  nko = nk;
  nderivo = nderiv;
  xmino = xmin;
  xmaxo = xmax;
  dxo = d;
  xlowo = xlow;
  return n;
}
