/***************************************************************************
                          fitdlg.cpp  -  description
                             -------------------
    begin                : Tue May 25 1999
    copyright            : (C) 2005 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <math.h>
#include <stdlib.h>
#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qdir.h>
#include <qfileinfo.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlibrary.h>
#include <qlistbox.h>
#include <qlineedit.h>
#include <qpainter.h>
#include <qradiobutton.h>
#include <qregexp.h>
#include <qtable.h>
#include <qtimer.h>
#include <kapplication.h>
#include <kcolorbutton.h>
#include <kconfig.h>
#include <kglobal.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <knotifyclient.h>
#include <kseparator.h>
#include <kurl.h>
#include "arrayitem.h"
#include "choosefuncdlg.h"
#include "editparameterdlg.h"
#include "fitdlg.h"
#include "frameitem.h"
#include "funitem.h"
#include "kpldoc.h"
#include "kpldoubleedit.h"
#include "kplgraph.h"
#include "kplspinbox.h"
#include "lmfit.h"
#include "residualsdlg.h"
#include "sourcewidget.h"
#include "utils.h"

CorrDlg::CorrDlg(QWidget* _parent, int np, const int* ip, const double* corr) :
  KDialog(_parent, 0, true)
{
  setPlainCaption(i18n("Parameter correlation coefficients"));
  QTable* table = new QTable(np, np, this);
  table->setMinimumSize(100, 40);
  QString s;
  for (int iy = 0; iy < np; iy++)
    table->verticalHeader()->setLabel(iy, s.sprintf("p%i", ip[iy]));
  for (int ix = 0; ix < np; ix++) {
    table->horizontalHeader()->setLabel(ix, s.sprintf("p%i", ip[ix]));
    for (int iy = 0; iy < np; iy++) {
      double temp = corr[np * iy + ix];
      if ((temp >= -1.0004) && (temp  <= 1.0004))
        s.setNum(temp, 'f', 3);
      else
        s = "NaN";
      table->setItem(iy, ix, new QTableItem(table, QTableItem::Never, s));
    }
    table->adjustColumn(ix);
  }
  table->adjustSize();
  (new QVBoxLayout(this, 0, 0))->addWidget(table);
}

CorrDlg::~CorrDlg()
{
}

ErrModDlg::ErrModDlg(QWidget* _parent, KplDoc* model,
                     KplNamespace::DataErrorStruct* err0) :
 KDialogBase(Plain, i18n("Error model function"), Help | Ok | Cancel, Ok,
            _parent, 0, true, true), m(model), err(err0)
{
  errt = new KplNamespace::DataErrorStruct(*err);
  QFrame* frame = plainPage();
  QGridLayout* grid = new QGridLayout(frame, 4, 2, 0, spacingHint());
  grid->addWidget(new QLabel(i18n("Library"), frame), 0, 0);
  QHBoxLayout* hbox = new QHBoxLayout();
  grid->addLayout(hbox, 0, 1);
  hbox->addWidget(fileName = new QLineEdit(err->errModPath, frame));
  QPushButton* b = new QPushButton("...", frame);
  b->setFixedWidth(28);
  hbox->addWidget(b);
  connect(b, SIGNAL(clicked()), SLOT(slotFile()));
  grid->addWidget(new QLabel(i18n("Function"), frame), 1, 0);
  grid->addLayout(hbox = new QHBoxLayout(), 1, 1);
  hbox->addWidget(func = new QLineEdit(err->errModName, frame));
  hbox->addWidget(b = new QPushButton("...", frame));
  b->setFixedWidth(28);
  grid->addWidget(new QLabel(i18n("Argument"), frame), 2, 0);
  grid->addLayout(hbox = new QHBoxLayout(), 2, 1);
  bg = new QButtonGroup();
  QRadioButton* r;
  hbox->addWidget(r = new QRadioButton(i18n("x column"), frame));
  bg->insert(r);
  hbox->addWidget(r = new QRadioButton(i18n("y column"), frame));
  bg->insert(r);
  ((QRadioButton*) bg->find(err->errModArg))->setChecked(true);
  hbox->addItem(new QSpacerItem(10, 10, QSizePolicy::Expanding));
  setHelp("ERRORMODEL");
  connect(b, SIGNAL(clicked()), SLOT(slotFunc()));
  grid->addWidget(new QLabel(i18n("Parameter"), frame), 3, 0);
  grid->addLayout(hbox = new QHBoxLayout(), 3, 1);
  hbox->addWidget(b = new QPushButton(i18n("Edit"), frame));
  connect(b, SIGNAL(clicked()), SLOT(slotEditParameter()));
  hbox->addWidget(b = new QPushButton(i18n("Load"), frame));
  connect(b, SIGNAL(clicked()), SLOT(slotGetParameter()));
  hbox->addWidget(b = new QPushButton(i18n("Save"), frame));
  connect(b, SIGNAL(clicked()), SLOT(slotSaveParameter()));
}

ErrModDlg::~ErrModDlg()
{
  delete errt;
  delete bg;
}

void ErrModDlg::slotFile()
{
  m->getFile(fileName);
}

void ErrModDlg::slotFunc()
{
  QFileInfo fi(fileName->text());
  QString path = fi.dirPath(true);
  QFile f(path + "/" + fi.baseName() + ".def");
  if (f.open(IO_ReadOnly)) {
    ChooseFuncDlg dlg(this, &f, func);
    dlg.exec();
  } else
    KMessageBox::error(this,
                       i18n("while trying to open module definition file"));
}

void ErrModDlg::slotGetParameter()
{
  m->getPar(this, errt->pErrMod);
}

void ErrModDlg::slotEditParameter()
{
  EditParameterDlg dlg(this, m, errt->pErrMod, fileName->text(), func->text());
  dlg.exec();
}

void ErrModDlg::slotSaveParameter()
{
  m->saveFunPar(this, errt->pErrMod);
}

void ErrModDlg::slotOk()
{
  errt->errModPath = fileName->text();
  errt->errModName = func->text();
  errt->errModArg = bg->id(bg->selected());
  QLibrary* lib;
  double (*fkt)(double, const double *);
  if (!FunItem::getFuncAddr(errt->errModPath, errt->errModName, &lib, &fkt))
    return;
  delete lib;
  *err = *errt;
  accept();
}

FitDlg::FitDlg(QWidget* _parent, KplDoc* model, QPtrList<ArrayItem>* ad0,
               QPtrList<FunItem>* fd0, int mode) :
 KDialogBase(Plain, i18n("Parameter fit"),
             Help | Ok | Apply | Cancel | User1 | User2,
             Ok, _parent, 0, true, true, i18n("&Start"), i18n("&Residuals")),
 running(false), dlgMode(mode), m(model), ad(ad0), fd(fd0), sw(0)
{
  aut = new KplNamespace::AutoStruct(*m->options());
  config = KGlobal::config();
  err = new KplNamespace::DataErrorStruct[ad->count()];
  int i;
  QString s;
  ArrayItem* a;
  for (i = 0; i < (int) ad->count(); i++) {
    err[i] = aut->err;
    s.sprintf("DataErrors%i", i);
    if (config->hasGroup(s)) {
      config->setGroup(s);
      err[i].errModPath = config->readEntry("ErrorModelPath",
                                            aut->err.errModPath);
      err[i].errModName = config->readEntry("ErrorModelName",
                                            aut->err.errModName);
      QStringList list = config->readListEntry("ErrorModelParam", ' ');
      int cnt = list.count();
      for (int j = 0; j < KPL_NPMAX; j++)
        err[i].pErrMod[j] = (j < cnt) ? list[j].toDouble() : 0.0;
      a = ad->at(i);
      if (a->ie < a->ncols)
        err[i].fitErrCol = config->readBoolEntry("FitErrorColumn",
                                                 aut->err.fitErrCol);
      else
        err[i].fitErrCol = false;
      err[i].errModArg = config->readNumEntry("ErrorModelArgument",
                                              aut->err.errModArg);
    }
  }
  FunItem* ff = fd->first();
  for (i = 0; i < KPL_NPMAX; i++) {
    m->pLast[i] = ff->py[i];
    m->pErr[i] = 0.0;
  }
  if (dlgMode & ShowDlg) {
    Utils::setSize(this, "FitDialog");
    QHBoxLayout* hbox = 0;
    QVBoxLayout* vbox;
    QFrame* frame = plainPage();
    if (m->options()->showSource) {
      hbox = new QHBoxLayout(frame, 11, spacingHint());
      vbox = new QVBoxLayout(hbox);
    } else
      vbox = new QVBoxLayout(frame, 11, spacingHint());
    QGroupBox* g = new QGroupBox(0, Qt::Vertical, i18n("Parameter"), frame);
    vbox->addWidget(g);
    QVBoxLayout* vbox2 = new QVBoxLayout(g->layout(), spacingHint());
    int nGr = 1 + (KPL_NPMAX - 1) / 10;
    QGridLayout* grid = new QGridLayout(vbox2, QMIN(KPL_NPMAX, 10),
                                        5 * nGr - 1);
    for (i = 1; i < nGr; i++)
      grid->addItem(new QSpacerItem(20, 10), 0, 4 * i);
    char frm = m->options()->format;
    int prec = m->options()->prec;
    int w = 0;
    QString s0 = m->number(0.0);
    for (i = 0; i < KPL_NPMAX; i++) {
      int ix = 5 * (i / 10);
      int iy = i % 10;
      grid->addWidget(enFit[i] = new QCheckBox(s.sprintf("p%i", i), g),
                      iy, ix);
      enFit[i]->setChecked(aut->fitPar[i]);
      widgetList.append(enFit[i]);
      grid->addWidget(par[i] = new KplDoubleEdit(m->pLast[i], g, frm, prec),
                      iy, ix + 1);
      if (!i)
        w = QMAX(80, par[0]->fontMetrics().width(m->number(-1.0e-123 / 3.0)) +
                                                 3);
      par[i]->setFixedWidth(w);
      widgetList.append(par[i]);
      grid->addWidget(new QLabel("\261", g), iy, ix + 2);
      grid->addWidget(epar[i] = new QLabel(s0, g), iy, ix + 3);
      epar[i]->setFrameStyle(QFrame::Panel | QFrame::Sunken);
      epar[i]->setFixedWidth(w);
    }
    QHBoxLayout* hbox2 = new QHBoxLayout(vbox2, spacingHint());
    hbox2->addWidget(nonLin = new QCheckBox(i18n("Nonlinear fit"), g));
    nonLin->setChecked(aut->fitNonLin);
    widgetList.append(nonLin);
    QPushButton* b = new QPushButton(i18n("Load"), g);
    hbox2->addWidget(b);
    connect(b, SIGNAL(clicked()), SLOT(slotGetParameter()));
    widgetList.append(b);
    hbox2->addWidget(b = new QPushButton(i18n("Save"), g));
    connect(b, SIGNAL(clicked()), SLOT(slotSaveParameter()));
    widgetList.append(b);
    hbox2->addWidget(showCorr = new QPushButton(i18n("Correlations"), g));
    showCorr->setEnabled(false);
    connect(showCorr, SIGNAL(clicked()), SLOT(slotCorr()));
    g = new QGroupBox(0, Qt::Horizontal, i18n("Data errors"), frame);
    hbox2 = new QHBoxLayout(g->layout(), spacingHint());
    hbox2->addWidget(new QLabel(i18n("Array"), g));
    a = ad->at(0);
    hbox2->addWidget(sArr = new KplSpinBox(0, ad->count() - 1, 1, g));
    sArr->setValue(0);
    hbox2->addWidget(lArr = new QLabel(g));
    lArr->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    hbox2->addWidget(errCol = new QCheckBox(i18n("Error column"), g));
    hbox2->addItem(new QSpacerItem(10, 10, QSizePolicy::Expanding));
    hbox2->addWidget(errMod = new QPushButton(i18n("Model"), g));
    connect(errMod, SIGNAL(clicked()), SLOT(slotErrMod()));
    vbox->addWidget(g);
    hbox2 = new QHBoxLayout(vbox, spacingHint());
    hbox2->addWidget(new QLabel(i18n("Maximum iterations"), frame));
    hbox2->addWidget(sItmax = new KplSpinBox(1, 200, 1, frame));
    sItmax->setValue(aut->fitMaxIt);
    widgetList.append(sItmax);
    hbox2->addItem(new QSpacerItem(10, 10, QSizePolicy::Expanding));
    hbox2->addWidget(new QLabel(i18n("Tolerance"), frame));
    hbox2->addWidget(eTol = new KplDoubleEdit(aut->fitTol, 0.0, 1.0, frame,
                                              'g', 6));
    widgetList.append(eTol);
    vbox->addWidget(results = new QListBox(frame));
    results->setMinimumHeight(40);
    widgetList.append(actionButton(Ok));
    widgetList.append(actionButton(Apply));
    widgetList.append(actionButton(User2));
    enableButton(User2, false);
    if (m->options()->showSource) {
      hbox->addWidget(sw = new SourceWidget(frame, 0), 1);
      sw->setMinimumWidth(200);
    }
    setHelp("SEC-FIT");
    connect(nonLin, SIGNAL(toggled(bool)), SLOT(enableNonLin(bool)));
    connect(errCol, SIGNAL(toggled(bool)), SLOT(errColToggled(bool)));
    connect(sArr, SIGNAL(valueChanged(int)), SLOT(updArray(int)));
    enableNonLin(aut->fitNonLin);
    updArray(0);
    if (m->options()->showSource)
      QTimer::singleShot(0, sw, SLOT(slotHighlight()));
  }
}

FitDlg::~FitDlg()
{
  if (dlgMode & ShowDlg)
    Utils::saveSize(this, "FitDialog");
  delete [] err;
  delete aut;
}

void FitDlg::getValues(bool ok)
{
  int i, j;
  if (dlgMode & ShowDlg) {
    aut->fitNonLin = nonLin->isChecked();
    for (i = 0; i < KPL_NPMAX; i++)
      aut->fitPar[i] = enFit[i]->isChecked();
    if (aut->fitNonLin) {
      aut->fitMaxIt = sItmax->interpretedValue();
      aut->fitTol = eTol->value();
    }
    m->setOptions(aut);
    QString s;
    for (j = 0; j < (int) fd->count(); j++) {
      s.sprintf("DataErrors%i", j);
      config->setGroup(s);
      config->writeEntry("FitErrorColumn", err[j].fitErrCol);
      config->writeEntry("ErrorModelPath", err[j].errModPath);
      config->writeEntry("ErrorModelName", err[j].errModName);
      QStringList s1;
      for (i = 0; i < KPL_NPMAX; i++)
        s1.append(m->number(err[j].pErrMod[i]));
      config->writeEntry("ErrorModelParam", s1, ' ');
      config->writeEntry("ErrorModelArgument", err[j].errModArg);
    }
  }
  for (j = 0; j < (int) fd->count(); j++)
    for (i = 0; i < KPL_NPMAX; i++)
      fd->at(j)->py[i] = m->pLast[i];
  m->setModified();
  m->backupItems();
  if (ok)
    accept();
}

void FitDlg::slotMessage(const QString& msg)
{
  results->insertItem(msg);
  results->setTopItem(results->count() - 1);
  kapp->processEvents();
}

void FitDlg::slotGetParameter()
{
  m->getPar(this, m->pLast);
  for (int i = 0; i < KPL_NPMAX; i++) {
    par[i]->setValue(m->pLast[i]);
    m->pErr[i] = 0.0;
    epar[i]->setText("0");
  }
}

void FitDlg::slotSaveParameter()
{
  KURL url;
  if (m->getWriteURL(this, url, "*.par\n*"))
    m->saveParameter(this, url, m->pLast, m->pErr);
}

void FitDlg::errColToggled(bool state)
{
  errMod->setEnabled(!state);
  err[sArr->interpretedValue()].fitErrCol = state;
}

void FitDlg::enableNonLin(bool on)
{
  eTol->setEnabled(on);
  sItmax->setEnabled(on);
}

void FitDlg::updArray(int i)
{
  ArrayItem* a = ad->at(i);
  lArr->setText(a->url.fileName() + " " + QString::number(a->ix) + " " +
                QString::number(a->iy) + " " + QString::number(a->ie));
  errCol->setChecked(err[i].fitErrCol);
  errCol->setEnabled(a->ie < a->ncols);
  if (m->options()->showSource) {
    FunItem* ff = fd->at(i);
    QFileInfo fi(ff->pathy.path());
    QRegExp rx2("\\S+__FdPCd");
    QRegExp rx3("_Z\\d+\\S+dPKd");
    int pp2 = rx2.search(ff->namey);
    int pp3 = rx3.search(ff->namey);
    QString s = fi.dirPath(true) + "/" + fi.baseName();
    bool guessC = pp2 && pp3;
    QFile f(s + (guessC ? ".c" : ".cpp"));
    bool open = f.open(IO_ReadOnly);
    if (!open) {
      f.setName(s + (guessC ? ".cpp" : ".c"));
      open = f.open(IO_ReadOnly);
    }
    if (open) {
      sw->fill(&f);
      if (!pp2)
        sw->highlight(ff->namey.left(rx2.matchedLength() - 7));
      else
        if (!pp3)
          sw->highlight(ff->namey.mid(3, rx3.matchedLength() - 7));
        else
          sw->highlight(ff->namey);
    }
  }
}

void FitDlg::slotOk()
{
  getValues(true);
}

void FitDlg::slotCancel()
{
  if (running)
    lm->userBreak = true;
  else
    reject();
}

void FitDlg::slotApply()
{
  getValues(false);
}

void FitDlg::slotUser1()
{
  fit();
}

void FitDlg::fit()
{
  int i;
  np = 0;
  for (i = 0; i < KPL_NPMAX; i++) {
    if (dlgMode & ShowDlg)
      aut->fitPar[i] = enFit[i]->isChecked();
    if (aut->fitPar[i])
      np++;
  }
  if (!np) {
    KMessageBox::sorry(this, i18n("No parameter to fit!"));
    return;
  }
  lm = new LMFit(ad, fd, aut->fitPar, m->pLast, this);
  lm->sig = sig.data();
  bool nLin = aut->fitNonLin;
  if (dlgMode & ShowDlg) {
    nLin = nonLin->isChecked();
    if (nLin) {
      aut->fitTol = eTol->value();
      aut->fitMaxIt = sItmax->interpretedValue();
    }
    showCorr->setEnabled(false);
    results->clear();
    connect(lm, SIGNAL(updateMessage(const QString &)),
            SLOT(slotMessage(const QString &)));
    for (i = 0; i < KPL_NPMAX; i++)
      m->pLast[i] = par[i]->value();
    for (QWidget* w = widgetList.first(); w; w = widgetList.next())
      w->setEnabled(false);
  }
  kapp->processEvents();
  double avgErr;
  running = true;
  int info = FunItem::fit(ad, fd, aut->fitPar, m->pLast, &fvec, &sig, err,
                          nLin, aut->fitTol, aut->fitMaxIt, lm, &m->chisq,
                          corr, m->pErr, &avgErr, this);
  if (info == -101)
    KMessageBox::error(this, i18n("no function address!"));
  QString s;
  if (info >= -100) {
    if (nLin) {
      if (info >= 8)
        info = 4;
      if (dlgMode & ShowDlg) {
        switch (info) {
          case 0:
            s = i18n("Improper input parameters");
            break;
          case 1:
            s = i18n("Relative error in chi-square is at most tolerance");
            break;
          case 2:
            s = i18n("Relative error in parameters is at most tolerance");
            break;
          case 3:
            s = i18n("Relative error in chi-square and in parameters is at most "
                     "tolerance");
            break;
          case 4:
            s = i18n("Function vector is orthogonal to the columns of "
                     "the jacobian");
            break;
          case 5:
            s = i18n("Number of iterations has reached or exceeded maximum");
            break;
          case 6:
            s = i18n("Tolerance too small. No further reduction of "
                     "chi-square possible");
            break;
          case 7:
            s = i18n("Tolerance too small. No further improvement of "
                     "parameters possible");
            break;
          default:
            s = i18n("Terminated by user");
        }
        slotMessage(s);
      }
    }
  }
  running = false;
  if (info > -100) {
    int k = 0;
    for (i = 0; i < KPL_NPMAX; i++) {
      if (aut->fitPar[i])
        ip[k++] = i;
      if (dlgMode & ShowDlg) {
        par[i]->setValue(m->pLast[i]);
        epar[i]->setText(m->number(m->pErr[i]));
      }
      if (dlgMode & Follow)
        aut->pFit[i] = m->pLast[i];
    }
    if (dlgMode & ShowDlg) {
      int n = 0;
      for (ArrayItem* a = ad->first(); a; a = ad->next())
        n += a->n;
      int ny = n - np;
      double q = ny ? LMFit::igamc(0.5 * ny, 0.5 * m->chisq) : 0.0;
      slotMessage(s.sprintf(i18n("Average error = %.3g    ny = %i    "
                                 "Significance Q = %.3g"),
                            avgErr, ny, q));
    }
    if (dlgMode & SavePar) {
      KURL url = m->URL();
      QFileInfo fi(url.path());
      url.setPath(fi.dirPath(true) + "/" + fi.baseName() + ".par");
      m->saveParameter(this, url, m->pLast, m->pErr);
    }
    if (!lm->userBreak)
      KNotifyClient::event(
#if KDE_VERSION >= 0x030101
                           winId(),
#endif
                           "FitCompleted", i18n("Parameter fit completed."));
  }
  delete lm;
  if (dlgMode & ShowDlg) {
    showCorr->setEnabled(true);
    for (QWidget* w = widgetList.first(); w; w = widgetList.next())
      w->setEnabled(true);
    enableNonLin(nonLin->isChecked());
  }
}

void FitDlg::slotErrMod()
{
  ErrModDlg dlg(this, m, &err[sArr->interpretedValue()]);
  dlg.exec();
}

void FitDlg::slotUser2()
{
  QPtrList<KplItem> items;
  items.setAutoDelete(true);
  FrameItem* fd = new FrameItem(true, false, false, -1, -1, 5, 2,
                                KplGraph::GridWithLabels, 0, 0, "0", "0",
                                4.0, 15.0, 3.0, 10.0, 0.0, 15.0, 0.0, 10.0,
                                5.0, 2.0, 1.0, "x", i18n("Residuals"), "",
                                m->options());
  items.append(fd);
  int j = 0;
  for (ArrayItem* a = ad->first(); a; a = ad->next()) {
    ArrayItem* ar = new ArrayItem(true, false, a->symb, "0", 1.0, 1.0, 0, 1, 2,
                                  0, a->n, 1, "", true);
    ar->color = a->color;
    for (int i = 0; i < a->n; i++) {
      ar->x[0][i] = a->x[a->ix][a->istart + i];
      ar->x[1][i] = fvec[j] * sig[j];
      ar->x[2][i] = sig[j++];
    }
    items.append(ar);
  }
  fd->autoScale(true, &items);
  ResidualsDlg dlg(this, m, &items);
  dlg.exec();
}

void FitDlg::slotCorr()
{
  CorrDlg dlg(this, np, ip, corr);
  dlg.exec();
}
