% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/runMI-score.R
\name{lavTestScore.mi}
\alias{lavTestScore.mi}
\title{Score Test for Multiple Imputations}
\usage{
lavTestScore.mi(object, add = NULL, release = NULL, test = c("D2",
  "Rubin"), scale.W = FALSE, asymptotic = !is.null(add),
  univariate = TRUE, cumulative = FALSE, epc = FALSE,
  verbose = FALSE, warn = TRUE, information = "expected")
}
\arguments{
\item{object}{An object of class \code{\linkS4class{lavaan.mi}}.}

\item{add}{Either a \code{character} string (typically between single
quotes) or a parameter table containing additional (currently fixed-to-zero)
parameters for which the score test must be computed.}

\item{release}{Vector of \code{integer}s. The indices of the \emph{equality}
constraints that should be released. The indices correspond to the order of
the equality constraints as they appear in the parameter table.}

\item{test}{\code{character} indicating which pooling method to use.
\code{"Rubin"} indicates Rubin's (1987) rules will be applied to the
 gradient and information, and those pooled values will be used to
 calculate modification indices in the usual manner. \code{"D2"} (default),
\code{"LMRR"}, or \code{"Li.et.al"} indicate that modification indices
 calculated from each imputed data set will be pooled across imputations,
 as described in Li, Meng, Raghunathan, & Rubin (1991) and Enders (2010).}

\item{scale.W}{\code{logical}. If \code{FALSE} (default), the pooled
information matrix is calculated as the weighted sum of the
within-imputation and between-imputation components. Otherwise, the pooled
information is calculated by scaling the within-imputation component by the
average relative increase in variance (ARIV; see Enders, 2010, p. 235).
Not recommended, and ignored (irrelevant) if \code{test = "D2"}.}

\item{asymptotic}{\code{logical}. If \code{FALSE} (default when using
\code{add} to test adding fixed parameters to the model), the pooled test
will be returned as an \emph{F}-distributed variable with numerator
(\code{df1}) and denominator (\code{df2}) degrees of freedom.
If \code{TRUE}, the pooled \emph{F} statistic will be multiplied by its
\code{df1} on the assumption that its \code{df2} is sufficiently large
enough that the statistic will be asymptotically \eqn{\chi^2} distributed
with \code{df1}. When using the \code{release} argument, \code{asymptotic}
will be set to \code{TRUE} because (A)RIV can only be calculated for
\code{add}ed parameters.}

\item{univariate}{\code{logical}. If \code{TRUE}, compute the univariate
score statistics, one for each constraint.}

\item{cumulative}{\code{logical}. If \code{TRUE}, order the univariate score
statistics from large to small, and compute a series of multivariate
score statistics, each time including an additional constraint in the test.}

\item{epc}{\code{logical}. If \code{TRUE}, and we are releasing existing
constraints, compute the expected parameter changes for the existing (free)
parameters (and any specified with \code{add}), if all constraints
were released. For EPCs associated with a particular (1-\emph{df})
constraint, only specify one parameter in \code{add} or one constraint in
\code{release}.}

\item{verbose}{\code{logical}. Not used for now.}

\item{warn}{\code{logical}. If \code{TRUE}, print warnings if they occur.}

\item{information}{\code{character} indicating the type of information
matrix to use (check \code{\link{lavInspect}} for available options).
\code{"expected"} information is the default, which provides better
control of Type I errors.}
}
\value{
A list containing at least one \code{data.frame}:
 \itemize{
   \item{\code{$test}: The total score test, with columns for the score
     test statistic (\code{X2}), the degrees of freedom (\code{df}), and
     a \emph{p} value under the \eqn{\chi^2} distribution (\code{p.value}).}
   \item{\code{$uni}: Optional (if \code{univariate=TRUE}).
     Each 1-\emph{df} score test, equivalent to modification indices.}
   \item{\code{$cumulative}: Optional (if \code{cumulative=TRUE}).
     Cumulative score tests.}
   \item{\code{$epc}: Optional (if \code{epc=TRUE}). Parameter estimates,
     expected parameter changes, and expected parameter values if all
     the tested constraints were freed.}
 }
See \code{\link[lavaan]{lavTestScore}} for details.
}
\description{
Score test (or "Lagrange multiplier" test) for lavaan models fitted to
multiple imputed data sets. Statistics for releasing one or more
fixed or constrained parameters in model can be calculated by pooling
the gradient and information matrices pooled across imputed data sets
using Rubin's (1987) rules, or by pooling the score test statistics
across imputed data sets (Li, Meng, Raghunathan, & Rubin, 1991).
}
\examples{
 \dontrun{
## impose missing data for example
HSMiss <- HolzingerSwineford1939[ , c(paste("x", 1:9, sep = ""),
                                      "ageyr","agemo","school")]
set.seed(12345)
HSMiss$x5 <- ifelse(HSMiss$x5 <= quantile(HSMiss$x5, .3), NA, HSMiss$x5)
age <- HSMiss$ageyr + HSMiss$agemo/12
HSMiss$x9 <- ifelse(age <= quantile(age, .3), NA, HSMiss$x9)

## impute missing data
library(Amelia)
set.seed(12345)
HS.amelia <- amelia(HSMiss, m = 20, noms = "school", p2s = FALSE)
imps <- HS.amelia$imputations

## specify CFA model from lavaan's ?cfa help page
HS.model <- '
  speed =~ c(L1, L1)*x7 + c(L1, L1)*x8 + c(L1, L1)*x9
'

out <- cfa.mi(HS.model, data = imps, group = "school", std.lv = TRUE)

## Mode 1: Score test for releasing equality constraints

## default test: Li et al.'s (1991) "D2" method
lavTestScore.mi(out, cumulative = TRUE)
## Rubin's rules
lavTestScore.mi(out, test = "Rubin")

## Mode 2: Score test for adding currently fixed-to-zero parameters
lavTestScore.mi(out, add = 'x7 ~~ x8 + x9')

}

}
\references{
Bentler, P. M., & Chou, C.-P. (1992). Some new covariance structure model
  improvement statistics. \emph{Sociological Methods & Research, 21}(2),
  259--282. doi:10.1177/0049124192021002006

  Enders, C. K. (2010). \emph{Applied missing data analysis}.
  New York, NY: Guilford.

  Li, K.-H., Meng, X.-L., Raghunathan, T. E., & Rubin, D. B. (1991).
  Significance levels from repeated \emph{p}-values with multiply-imputed
  data. \emph{Statistica Sinica, 1}(1), 65--92. Retrieved from
  \url{https://www.jstor.org/stable/24303994}

  Rubin, D. B. (1987). \emph{Multiple imputation for nonresponse in surveys}.
  New York, NY: Wiley.
}
\seealso{
\code{\link[lavaan]{lavTestScore}}
}
\author{
Terrence D. Jorgensen (University of Amsterdam; \email{TJorgensen314@gmail.com})

Adapted from \pkg{lavaan} source code, written by
  Yves Rosseel (Ghent University; \email{Yves.Rosseel@UGent.be})

\code{test = "Rubin"} method proposed by
  Maxwell Mansolf (University of California, Los Angeles;
  \email{mamansolf@gmail.com})
}
