/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		SocketUtils.cpp

	Contains:	Implements utility functions defined in SocketUtils.h
					
	$Log: Socket.cpp,v $
	Revision 1.2  1999/02/19 23:13:18  ds
	Created
	
	
*/

#include <string.h>

#ifndef __MW_
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/utsname.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/ioctl.h>
#include <unistd.h>
#endif

#include "SocketUtils.h"

UInt32							SocketUtils::sNumIPAddrs = 0;
SocketUtils::IPAddrInfo*		SocketUtils::sIPAddrInfoArray = NULL;

void SocketUtils::Initialize()
{
	//Most of this code is similar to the SIOCGIFCONF code presented in Stevens,
	//Unix Network Programming, section 16.6
	
	//Use the SIOCGIFCONF ioctl call to iterate through the network interfaces
	static const UInt32 kMaxAddrBufferSize = 2048;
	
	struct ifconf ifc;
	struct ifreq* ifr;
	char buffer[kMaxAddrBufferSize];
	
	int tempSocket = ::socket(AF_INET, SOCK_DGRAM, 0);
	if (tempSocket == -1)
		return;
		
	ifc.ifc_len = kMaxAddrBufferSize;
	ifc.ifc_buf = buffer;
#if __linux__ || __MacOSX__
	int err = ::ioctl(tempSocket, SIOCGIFCONF, (char*)&ifc);
#elif __FreeBSD__
	int err = ::ioctl(tempSocket, OSIOCGIFCONF, (char*)&ifc);
#else
	#error
#endif
	if (err == -1)
		return;
	
#if __FreeBSD__
	int netdev1, netdev2;
	struct ifreq *netdevifr;
	netdevifr = ifc.ifc_req;
	netdev1 = ifc.ifc_len / sizeof(struct ifreq);
	for (netdev2=netdev1-1; netdev2>=0; netdev2--)
		{
		if (ioctl(tempSocket, SIOCGIFADDR, &netdevifr[netdev2]) != 0)
			continue;
		}
#endif
	
	::close(tempSocket);
	tempSocket = -1;

	//walk through the list of IP addrs twice. Once to find out how many,
	//the second time to actually grab their information
	char* ifReqIter = NULL;
	sNumIPAddrs = 0;
	
	for (ifReqIter = buffer; ifReqIter < (buffer + ifc.ifc_len);)
	{
		ifr = (struct ifreq*)ifReqIter;
		if (!SocketUtils::IncrementIfReqIter(&ifReqIter, ifr))
			return;
		
		//Only count interfaces in the AF_INET family.
		//And don't count localhost, loopback interfaces
		if ((ifr->ifr_addr.sa_family == AF_INET) && (::strncmp(ifr->ifr_name, "lo", 2) != 0))
			sNumIPAddrs++;
	}
	
	//allocate the IPAddrInfo array. Unfortunately we can't allocate this
	//array the proper way due to a GCC bug
	UInt8* addrInfoMem = new UInt8[sizeof(IPAddrInfo) * sNumIPAddrs];
	::memset(addrInfoMem, 0, sizeof(IPAddrInfo) * sNumIPAddrs);
	sIPAddrInfoArray = (IPAddrInfo*)addrInfoMem;
	
	//Now extract all the necessary information about each interface
	//and put it into the array
	UInt32 currentIndex = 0;
	
	for (ifReqIter = buffer; ifReqIter < (buffer + ifc.ifc_len);)
	{
		ifr = (struct ifreq*)ifReqIter;
		if (!SocketUtils::IncrementIfReqIter(&ifReqIter, ifr))
		{
			Assert(0);//we should have already detected this error
			return;
		}
		
		//Only count interfaces in the AF_INET family
		//And don't count localhost, loopback interfaces
		if ((ifr->ifr_addr.sa_family == AF_INET) && (::strncmp(ifr->ifr_name, "lo", 2) != 0))
		{
			struct sockaddr_in* addrPtr = (struct sockaddr_in*)&ifr->ifr_addr;	
			char* theAddrStr = ::inet_ntoa(addrPtr->sin_addr);

			//store the IP addr
			sIPAddrInfoArray[currentIndex].fIPAddr = ntohl(addrPtr->sin_addr.s_addr);
			
			//store the IP addr as a string
			sIPAddrInfoArray[currentIndex].fIPAddrStr.Len = ::strlen(theAddrStr);
			sIPAddrInfoArray[currentIndex].fIPAddrStr.Ptr = new char[sIPAddrInfoArray[currentIndex].fIPAddrStr.Len + 2];
			::strcpy(sIPAddrInfoArray[currentIndex].fIPAddrStr.Ptr, theAddrStr);

			//convert this addr to a dns name, and store it
			struct hostent* theDNSName = ::gethostbyaddr((char *)&addrPtr->sin_addr,
															sizeof(addrPtr->sin_addr), AF_INET);
			if (theDNSName != NULL)
			{
				sIPAddrInfoArray[currentIndex].fDNSNameStr.Len = ::strlen(theDNSName->h_name);
				sIPAddrInfoArray[currentIndex].fDNSNameStr.Ptr = new char[sIPAddrInfoArray[currentIndex].fDNSNameStr.Len + 2];
				::strcpy(sIPAddrInfoArray[currentIndex].fDNSNameStr.Ptr, theDNSName->h_name);
			}
			else
			{
				//if we failed to look up the DNS name, just store the IP addr as a string
				sIPAddrInfoArray[currentIndex].fDNSNameStr.Len = sIPAddrInfoArray[currentIndex].fIPAddrStr.Len;
				sIPAddrInfoArray[currentIndex].fDNSNameStr.Ptr = new char[sIPAddrInfoArray[currentIndex].fDNSNameStr.Len + 2];
				::strcpy(sIPAddrInfoArray[currentIndex].fDNSNameStr.Ptr, sIPAddrInfoArray[currentIndex].fIPAddrStr.Ptr);
			}
			
			//move onto the next array index
			currentIndex++;
		}
	}
	Assert(currentIndex == sNumIPAddrs);
}

bool SocketUtils::IncrementIfReqIter(char** inIfReqIter, ifreq* ifr)
{
	//returns true if successful, false otherwise

#if __MacOSX__
	*inIfReqIter += sizeof(ifr->ifr_name) + ifr->ifr_addr.sa_len;

	//if the length of the addr is 0, use the family to determine
	//what the addr size is
	if (ifr->ifr_addr.sa_len == 0)
#else
	*inIfReqIter += sizeof(ifr->ifr_name) + 0;
#endif
	{
		switch (ifr->ifr_addr.sa_family)
		{
			case AF_INET:
				*inIfReqIter += sizeof(struct sockaddr_in);
				break;
			default:
				*inIfReqIter += sizeof(struct sockaddr);
//				Assert(0);
//				sNumIPAddrs = 0;
//				return false;
		}
	}
	return true;
}

bool SocketUtils::IsMulticastIPAddr(UInt32 inAddress)
{
	return ((inAddress>>8) & 0x00f00000) == 0x00e00000;	//	multicast addresses == "class D" == 0xExxxxxxx == 1,1,1,0,<28 bits>
}

bool SocketUtils::IsLocalIPAddr(UInt32 inAddress)
{
	for (UInt32 x = 0; x < sNumIPAddrs; x++)
		if (sIPAddrInfoArray[x].fIPAddr == inAddress)
			return true;
	return false;
}

void SocketUtils::ConvertAddrToString(const struct in_addr& theAddr, StrPtrLen* ioStr)
{
	//re-entrant version of code below
	//inet_ntop(AF_INET, &theAddr, ioStr->Ptr, ioStr->Len);
	//ioStr->Len = ::strlen(ioStr->Ptr);
	
	char* addr = inet_ntoa(theAddr);
	strcpy(ioStr->Ptr, addr);
	ioStr->Len = ::strlen(ioStr->Ptr);
}

