/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTSPModule.h

	Contains:	Base class that defines the API for RTSP modules to implement. RTSP modules
				are responsible for handling, modifying, and providing custom services for
				RTSP requests. They may elect to handle the request and send a response to
				the client. They do so by creating an RTSPHandler object.
				
				A default RTSPHandler for modules is provided (RTSPModuleHandler). This handler
				will simply invoke the module for request processing. If a module chooses, it
				may subclass this object and provide custom information about the request,
				and state information for async processing.
	
	$Log: RTSPModule.h,v $
	Revision 1.2  1999/02/19 23:08:29  ds
	

*/

#ifndef __RTSPMODULE_H__
#define __RTSPMODULE_H__

//INCLUDES:
#include "RTSPProtocol.h"
#include "RTSPRequestInterface.h"
#include "RTSPServerInterface.h"
#include "RTSPSessionInterface.h"
#include "RTSPMessages.h"

class RTSPModule {
public:
	
	//CONSTRUCTOR / DESTRUCTOR:
	
	RTSPModule() : fRollMask(0) {}
	virtual ~RTSPModule() {}
	
	//MEMBER FUNCTIONS:
	
	virtual bool		Initialize() { return true; }
	virtual void		Shutdown() {}
	
	//WHAT TYPE OF MODULE IS THIS???
	enum
	{
		kFilterModule = 			0x00000001,	//All are UInt32s
		kRequestProcessingModule = 	0x00000002,
		kSessionClosingModule = 	0x00000004,
		kErrorLoggingModule = 		0x00000008
	};
	
	bool	IsFilterModule()			{ return fRollMask & kFilterModule; }
	bool	IsRequestProcessingModule() { return fRollMask & kRequestProcessingModule; }
	bool	IsSessionClosingModule() 	{ return fRollMask & kSessionClosingModule; }
	bool	IsLoggingModule() 			{ return fRollMask & kErrorLoggingModule; }
	
	//REQUEST PROCESSING
	
	//Modules get invoked in this role immediately after the full request has arrived, but
	//before the server has had a chance to muck with it. They can read the full request,
	//and if they want, send a response right off the bat.
	virtual void		FilterRequest(RTSPRequestInterface*/*request*/)	{}

	//Module invoked in this method is responsible for sending the RTSP response to the client.
	//If it encounters an error, it should return an RTSPHandler object that can handle the error,
	//and in turn send a response to the client 
	virtual RTSPProtocol::RTSPStatusCode ProcessRequest(RTSPRequestInterface */*request*/) 	{ return RTSPProtocol::kSuccessOK; }
	
	virtual void		SessionClosing(RTSPSessionInterface* /*inSession*/) {}
											
	//LOGGING
	
	//There are three types of logging actions a module can implement. The first one
	//is request logging. At the end of each RTSP request, modules have the option to log
	//it. The latter two are error logging, where general system warnings and failures can be logged
	
	typedef UInt32 Verbosity;
	typedef UInt32 ErrorNumber;
	
	enum
	{
		kFatal = 0,		//All are of type "Verbosity"
		kWarning = 1,
		kAssert = 2,
		kDebugging = 3
	};
	
	virtual void	LogError(	const Verbosity /*inVerbosity*/,
								const char* /*inString*/) {}
	virtual void	LogError(	const Verbosity /*inVerbosity*/,
								const RTSPMessages::Message /*inString*/,
								const ErrorNumber /*inNumber*/) {}

protected:

	//What roles does this module want to be invoked in?
	UInt32	fRollMask;
};

class RTSPModuleInterface
{
	public:
	
		//These utility functions allow the server to easily invoke all loaded modules
		//to process a certain event.
		static void				FilterRequest(RTSPRequestInterface* inRequest);
		
		static RTSPProtocol::RTSPStatusCode ProcessRequest(RTSPRequestInterface* inRequest);
		
		static void				LogError(	const RTSPModule::Verbosity inVerbosity,
											const char* inMessage);
		static void				LogError(	const RTSPModule::Verbosity inVerbosity,
											const RTSPMessages::Message inMessage,
											const RTSPModule::ErrorNumber inNumber);
		
		static void				SessionClosing(RTSPSessionInterface* inSession);
		
	protected:
	
		static RTSPModule** sRTSPModuleList;
		static UInt32 sNumRTSPModules;
};
#endif // __RTSPMODULE_H__

