/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTPStream.h

	Contains:	Represents a single client stream (audio, video, etc).
				Control API is similar to overall session API.
				
				Contains all stream-specific data & resources, used by Session when it
				wants to send out or receive data for this stream
				
	$Log: RTPStream.h,v $
	Revision 1.4  1999/02/24 19:28:06  ds
	RTPStream now uses new UDPSocketPool interface for getting its UDP sockets
	
	Revision 1.2  1999/02/19 23:08:27  ds
	Created
	

*/

#ifndef __RTPSTREAM_H__
#define __RTPSTREAM_H__


#include "RTPSession.h"
#include "RTSPRequestInterface.h"
#include "RTSPProtocol.h"
#include "UDPDemuxer.h"
#include "StreamDictionary.h"

class RTPStream : public UDPDemuxerTask
{
	public:
		
		//This class performs all the duties necessary for maintaining an RTP stream.
		//It also processes RTCP packets.
		RTPStream(UInt32 inStreamID, RTPSession* inSession,
					StrPtrLen* inCodecName, UInt32 inCodecType);
		virtual ~RTPStream();
		
		//CODEC TYPE
		enum
		{
			kUnknownCodecType = 0,	//UInt32s
			kVideoCodecType = 1,
			kAudioCodecType = 2
		};
		
		//STREAM QUALITY LEVEL
		//If the client doesn't have enough bandwidth to support all the bandwidth the
		//server is producing, the quality will suck. Therefore, modules may opt to
		//detect this (using the RTCP or other means available) and alter the quality
		//level. 
		
		UInt32	GetQualityLevel()			{ return fQualityLevel; }

		void	IncreaseQuality() { if (fQualityLevel > 0) fQualityLevel--; }
		void	DecreaseQuality(){ if ((fSession->GetNumQualityLevels() == 0) || (fQualityLevel < (fSession->GetNumQualityLevels() - 1))) fQualityLevel++; }
		
		//ACCESS FUNCTIONS
		UInt32	GetStreamID() 				{ return fStreamID; }
		RTPSession*	GetSession() 			{ return fSession; }
		UInt32	GetSSRC()					{ return fSsrc; }
		UInt32	GetTimeStampRandomOffset() 	{ return fTimestampRandomOffset; }
		UInt16	GetSeqNumRandomOffset()		{ return fSeqRandomOffset; }
		UDPSocketPair* GetSocketPair()		{ return fSockets; }
		UInt32	GetRemoteAddr()				{ return fRemoteAddr; }
		UInt16	GetRemoteRTCPPort()			{ return fRemoteRTCPPort; }
		StreamDictionary* GetDictionary()	{ return &fStreamDictionary; }
		
		//Codec name may be empty if it is unknown
		StrPtrLen*	GetCodecName()			{ return &fCodecName; }
		UInt32		GetCodecType()			{ return fCodecType; }
		
		//This function sets up the stream based on the information in the RTSP request.
		//If an error occurs a handler is returned to handle the error
		//
		//This is a no-op if called more than once
		RTSPProtocol::RTSPStatusCode Setup(RTSPRequestInterface* request);
		
		//This function prepares the stream for playing. If this function completes
		//successfully, your derived GetNextPacket() function will be called.
		//
		//This is a no-op if already playing
		//
		//You may opt to have the server send RTCP packets or not, and you may opt
		//to have the server write the RTP header or not
		void	Play(RTSPRequestInterface* request, bool inSendRTCP, bool inShouldWriteHeader);
		
		//This sends data to the client. The input must be a complete RTP packet.
		//This function will add the seq # & timestamp random offsets. It will write
		//these directly into the input buffer!
		void			SendRTP(char* inData, UInt32 inLen);
		void			SendRTCP(char* inData, UInt32 inLen);
		
		//UTILITY FUNCTIONS:
		//These are not necessary to call and do not manipulate the state of the
		//stream. They may, however, be useful services exported by the server

		//Formats a transport header for this stream. 
		void			AppendTransport(RTSPRequestInterface* request);
		
		//Formats a RTP-Info header for this stream.
		//Isn't useful unless you've already called Play()
		void			AppendRTPInfo(RTSPProtocol::RTSPHeader inHeader,
										RTSPRequestInterface* request);

		//When we get a new RTCP packet, we can directly invoke the RTP session and tell it
		//to process the packet right now!
		virtual void ProcessPacket(StrPtrLen* inPacket);
	void DumpDictionary();

		// Kind of a hack. The reflector needs this info to do thinning
		UInt16			GetNextSeqNumber() 		{ return fNextSeqNumber; }
		UInt16			GetSeqNumberOffset() 	{ return fSeqNumberOffset; }
		
		void			SetNextSeqNumber(UInt16 newValue) { fNextSeqNumber = newValue; }
		void			SetSeqNumberOffset(UInt16 newValue) { fSeqNumberOffset = newValue; }
		
	private:
	
		enum
		{
			kMaxRTCPPacketSizeInBytes = 1024,	//All are UInt32s
			kMaxSsrcSizeInBytes = 25,
			kMaxStreamURLSizeInBytes = 32,
			kSenderReportIntervalInSecs = 7
		};
	
		//Module assigns a streamID to this object
		UInt32			fStreamID;
		RTPSession*		fSession;
		
		//either pointers to the statically allocated sockets (maintained by the server)
		//or fresh ones (only fresh in extreme special cases)
		UDPSocketPair* fSockets;

		//who am i sending to?
		UInt32		fRemoteAddr;
		UInt16		fRemoteRTPPort;
		UInt16		fRemoteRTCPPort;
		
		//RTCP stuff 
		SInt64		fLastSenderReportTime;
		UInt32		fPacketCount;
		UInt32		fByteCount;
		
		//low-level RTP stuff 
		UInt32		fSsrc;
		char		fSsrcString[kMaxSsrcSizeInBytes];
		StrPtrLen	fSsrcStringPtr;
		
		//Sequence numbers & statistics
		UInt16		fSeqRandomOffset; //add this to each seq. from module
		UInt16		fLastSequenceNumber;//used for creating new random offsets
		UInt16		fFirstSeqNumber;//used in sending the play response
		bool		fHasFirstSeqNumber;
		
		//RTP timestamp stuff
		UInt32		fFirstRandomOffset;
		UInt32		fTimestampRandomOffset;//RTP time
		UInt32		fFirstTimeStamp;//RTP time
		bool		fHasFirstTimeStamp;
		
		//what is the URL for this stream?
		char		fStreamURL[kMaxStreamURLSizeInBytes];
		StrPtrLen	fStreamURLPtr;
		
		//so the session can queue all its streams up
		OSQueueElem	fQueueElem;
		
		//are we supposed to send RTCP sender reports?
		bool		fSendingRTCP;
		bool		fWritingHeader;
		UInt32		fQualityLevel;
		
		//Codec name and type
		StrPtrLen	fCodecName;
		UInt32		fCodecType;
		
		StreamDictionary fStreamDictionary;

		//Thinning information
		UInt16		fNextSeqNumber;
		UInt16		fSeqNumberOffset;
};
#endif // __RTPSTREAM_H__
