/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTPServerInterface.h

	Contains:	Defines an API for server-wide services that modules can access &
				manipulate. 
					
	$Log: RTPServerInterface.h,v $
	Revision 1.3  1999/02/26 04:56:40  cl
	Fixed total vs. current sessions bug
	
	Revision 1.2  1999/02/19 23:08:26  ds
	Created
	

*/

#ifndef _RTPSERVERINTERFACE_H_
#define _RTPSERVERINTERFACE_H_

#include "OSRef.h"
#include "UDPSocketPool.h"
#include "RTSPProtocol.h"
#include "atomic.h"

class RTPTotalByteUpdater;
class RTPServerInterface;
class RTSPRequestInterface;

//PRIVATE CLASSES USED BY RTP SERVER INTERFACE
class IncomingRTCPDataTask : public Task
{
	public:
		//This task handles all incoming RTCP data. It just polls, so make sure
		//to start the polling process by signalling a start event.
		IncomingRTCPDataTask() : Task() {}
		virtual ~IncomingRTCPDataTask() {}
	
	private:
		virtual SInt64 Run();
		
		enum
		{
			//We use non-blocking sockets for RTCP and poll to see if they have data
			kRTCPPollIntervalInMilSecs = 100 //UInt32
		};
		
		friend class RTPServerInterface;
};

class RTPSocketPool : public UDPSocketPool
{
	public:
		RTPSocketPool() {}
		~RTPSocketPool() {}
		
	private:	
		virtual UDPSocketPair*	ConstructUDPSocketPair();
		virtual void			DestructUDPSocketPair(UDPSocketPair* inPair);
		
		friend class RTPServerInterface;
};

class RTPServerInterface
{
	public:
	
		//Server tracks how much load is on it, and if it is too overloaded, tells you here
		//If it is overloaded, it returns an RTSPHandler to handle the error
		static RTSPProtocol::RTSPStatusCode IsOkToAddNewSession(RTSPRequestInterface* inRTSPRequest);
		
		//Allows you to map RTP session IDs (strings) to actual RTP session objects
		static OSRefTable*	GetRTPSessionMap() { return sRTPMap; }
	
		//Server provides a statically created & bound UDPSocket / Demuxer pair
		//for each IP address setup to serve RTP. You access those pairs through
		//this function. This returns a pair pre-bound to the IPAddr specified.
		static UDPSocketPool*	GetSocketPool() { return sSocketPool; }
		
		//RTCP support requires generating unique CNames for each stream.
		//This function generates a proper cName and returns its length. The buffer
		//passed in must be at least kMaxCNameLen
		enum
		{
			kMaxCNameLen = 20	//Uint32
		};
		static UInt32			GetACName(char* ioCNameBuffer);

		//Modules most certainly don't NEED to use this function, but it is awfully handy
		//if they want to take advantage of it. Using the SDP data provided in the iovec,
		//this function sends a standard describe response.
		//NOTE: THE FIRST ENTRY OF THE IOVEC MUST BE EMPTY!!!!
		static void SendDescribeResponse(RTSPRequestInterface* inRequest, iovec* describeData,
											UInt32 inNumVectors, UInt32 inTotalLength);
		
		//This function just takes care of formatting & sending a standard DESCRIBE header.
		static void			SendDescribeHeader(RTSPRequestInterface* inRequest, UInt32 responseLen);
		
		//statistics tracking
		static void			IncrementTotalSessions() { (void)atomic_add(&sTotalSessions, 1); }
		static UInt32		GetTotalSessions() { return sTotalSessions; }
		
		//Current bandwidth tracking. This is stored / reported in BYTES / second
		static UInt32		GetCurrentBandwidthInBits()	{ return sCurrentBandwidthInBits; }
		//Avg bandwidth... over a period of time.
		static UInt32		GetAvgBandwidthInBits()		{ return sAvgBandwidthInBits; }
		//packets per second for the server
		static UInt32		GetPacketsPerSecond()		{ return sPacketsPerSecond; }
		
		//total bytes sent by the server
		static void			IncrementTotalBytes(UInt32 bytes) 	{ (void)atomic_add(&sPeriodicBytes, bytes); }
		static UInt64		GetTotalBytes() 					{ return sTotalBytes; }

		//total packets sent by the server
		static void			IncrementTotalPackets() { (void)atomic_add(&sPeriodicPackets, 1); }
				
		//Server state:
		enum
		{
			kStartingUpState = 0,		//All are UInt32s
			kRunningState = 1,
			kRefusingConnectionsState = 2,
			kFatalErrorState = 3,//a fatal error has occurred, not shutting down yet
			kShuttingDownState = 4
		};	

		static UInt32		GetServerState();
		static void			SetServerState(UInt32 serverState);
		static UInt32  GetCurrentSessionCount() { return sRTPMap->GetNumRefsInTable(); }


	protected:
	
		//Sockets are allocated global to the server, and arbitrated through this pool here.
		//RTCP data is processed completely within the following task.
		static RTPSocketPool*			sSocketPool;
		static IncomingRTCPDataTask*	sRTCPDataTask;
		
		//stores the total number of connections since startup.
		//I don't use mactypes here because these variables are being
		//passed into Umesh's atomic routines.
		static unsigned int		sTotalSessions;
		//stores the total number of bytes served since startup
		static UInt64			sTotalBytes;
		//because there is no 64 bit atomic add (for obvious reasons), we efficiently
		//implement total byte counting by atomic adding to this variable, then every
		//once in awhile updating the sTotalBytes.
		static unsigned int		sPeriodicBytes;
		static unsigned int		sPeriodicPackets;
		
		//stores the current served bandwidth in BITS per second
		static UInt32			sCurrentBandwidthInBits;
		static UInt32			sAvgBandwidthInBits;
		static UInt32			sPacketsPerSecond;
		//is the server refusing connections or not?
		static UInt32			sIsRefusingConnections;
		
		static StrPtrLen 		sSDPSuffix;	
		static char*			sCNameBase;
		static OSRefTable*		sRTPMap;
		
		//stores the server state
		static UInt32			sServerState;
		
		friend class RTPSocketPool;
		friend class IncomingRTCPDataTask;
		friend class RTPTotalByteUpdater;
};

#endif //_RTPSERVERINTERFACE_H_