/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		TimeoutTask.cpp

	Contains:	Implementation of TimeoutTask
					
	$Log: TimeoutTask.cpp,v $
	Revision 1.2  1999/02/19 23:14:14  ds
	Created
	
	
*/

#include "TimeoutTask.h"
#include "OS.h"

TimeoutTaskThread*	TimeoutTask::sThread = NULL;

void TimeoutTask::Initialize()
{
	if (sThread == NULL)
	{
		sThread = new ('tmTT') TimeoutTaskThread();
		sThread->Signal(Task::kStartEvent);
	}
}

TimeoutTask::TimeoutTask(Task* inTask)
: fTask(inTask), fTimeout(0), fQueueElem(this)
{
	Assert(fTask != NULL);
	OSMutexLocker locker(&sThread->fMutex);
	sThread->fQueue.EnQueue(&fQueueElem);
}

TimeoutTask::~TimeoutTask()
{
	OSMutexLocker locker(&sThread->fMutex);
	sThread->fQueue.Remove(&fQueueElem);
}

void TimeoutTask::SetTimeoutMilSecs(SInt64 inMilliseconds)
{
	if (inMilliseconds > 0)
		fTimeout = OS::Milliseconds() + inMilliseconds;
	else
		fTimeout = 0;
}



SInt64 TimeoutTaskThread::Run()
{
	//ok, check for timeouts now. Go through the whole queue
	OSMutexLocker locker(&fMutex);
	SInt64 curTime = OS::Milliseconds();
	
	for (OSQueueIter iter(&fQueue); !iter.IsDone(); iter.Next())
	{
		TimeoutTask* theTimeoutTask = (TimeoutTask*)iter.GetCurrent()->GetEnclosingObject();
		
		//if it's time to time this task out, signal it
		if ((theTimeoutTask->fTimeout > 0) && (curTime >= theTimeoutTask->fTimeout))
		{
#if TIMEOUT_DEBUGGING
			printf("TimeoutTask %ld timed out. Curtime = %ld, timeout time = %ld\n",(SInt32)theTimeoutTask, curTime, theTimeoutTask->fTimeout);
#endif
			theTimeoutTask->fTask->Signal(Task::kTimeoutEvent);
		}
#if TIMEOUT_DEBUGGING
		else
			printf("TimeoutTask %ld not being timed out. Curtime = %ld. timeout time = %ld\n", (SInt32)theTimeoutTask, curTime, theTimeoutTask->fTimeout);
#endif
	}
	(void)this->GetEvents();//we must clear the event mask!
	this->SetIdleTimer(kIntervalSeconds * 1000);
	return 0;//don't delete me!
}
