/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		ReflectorSession.h

	Contains:	This object supports reflecting an RTP multicast stream to N
				RTPStreams. It spaces out the packet send times in order to
				maximize the randomness of the sending pattern and smooth
				the stream.
					
	$Log: ReflectorSession.h,v $
	Revision 1.1  1999/02/19 23:10:05  ds
	Created
	

*/

#ifndef _REFLECTOR_SESSION_H_
#define _REFLECTOR_SESSION_H_

#include "IdleTask.h"
#include "RTPStream.h"
#include "RTPSession.h"
#include "SDPParser.h"
#include "RTSPProtocol.h"
#include "RTPServerInterface.h"

//This will add some printfs that are useful for checking the thinning
#define REFLECTOR_THINNING_DEBUGGING 0 


class ReflectorStream;
class ReflectorBuffer;
class ReflectorPacket;
class ReflectorSender;

class ReflectorSession
{
	public:

		//This object gets scheduled and sends out packets
		
		ReflectorSession(SDPParser* inParser, StrPtrLen* inUniqueID);
		~ReflectorSession();
			
		RTSPProtocol::RTSPStatusCode	BindReflectorSockets(RTSPRequestInterface* inRTSPRequest);
		void			CloseSourceSockets();
		
		OSMutex*		GetMutex() 	{ return &fMutex; }
		OSRef*			GetRef()	{ return &fRef; }
		UInt32			GetNumStreams() { return fNumStreams; }
		
		inline StrPtrLen*		GetCodecName(UInt32 trackID);
		inline UInt32			GetCodecType(UInt32 trackID);
		
		//This object needs to track how many of its RTPStreams are playing.
		void			ChangePlayCount(int amount) { atomic_add(&fNumPlaying, amount); }
			
		//RTP sessions can attach and unattach to this reflector session as they
		//wish. This is how they do so. RemoveStreamsForSessions automatically
		//removes all the RTPStreams associated with a given RTPSession
		void	AddSession(RTPSession* inSession);
		void 	RemoveSession(RTPSession* inSession);
	
		//Reflector quality levels:
		enum
		{
			kAudioOnlyQuality = 1,		//UInt32
			kNormalQuality = 0,			//UInt32
			kNumQualityLevels = 2		//UInt32
		};

	private:
	
		void 		AllocateSessionArray(UInt32 inNumBuckets);

		//Data structures can't be modified while we are sending packets.
		OSMutex		fMutex;
	
		ReflectorStream** 	fStreamArray;
		UInt32				fNumStreams;
		
		//for storage in the session map. A reflector session is identified
		//by the URL pointing to its SDP file.
		char* fURL;
		OSRef  fRef;

		//RTPSessions are kept in a 2-dimensional array
		enum
		{
			kMinNumBuckets = 64	//UInt32
		};
		typedef RTPSession** Bucket;
		Bucket*		fSessionArray;
		UInt32		fBucketSize;
		UInt32		fNumBuckets;
		UInt32		fFirstEmptyBucket;//first bucket that's completely empty
	
		//time (in msec) between each client sending out their packets
		SInt64		fClientSpacing;

		UInt32		fNumElements;
		unsigned int fNumPlaying;//tracks how many of these streams are playing

		friend class ReflectorStream;
		friend class ReflectorSender;
};

class ReflectorPacket
{
	private:
	
		ReflectorPacket() : fQueueElem(this), fPacketPtr(fPacketData, 0) {}
		~ReflectorPacket() {}

		enum
		{
			kMaxReflectorPacketSize = 2048	//UInt32
		};

		UInt32 		fBucketsSeenThisPacket;
		SInt64 		fTimeArrived;
		OSQueueElem	fQueueElem;
		char		fPacketData[kMaxReflectorPacketSize];
		StrPtrLen	fPacketPtr;
				
		friend class ReflectorSender;
		friend class ReflectorSocket;
};

//Custom UDP socket classes for doing reflector packet retrieval, socket management
class ReflectorSocket : public UDPSocket, public IdleTask
{
	public:

		ReflectorSocket();
		virtual ~ReflectorSocket();
		
		void	AddSender(ReflectorSender* inSender);
		void	RemoveSender(ReflectorSender* inStreamElem);
		bool	HasSender() { return (fSenderQueue.GetLength() > 0); }
	
	private:
		
		virtual SInt64		Run();
		ReflectorPacket* 	GetPacket();
		void 				GetIncomingData(const SInt64& inMilliseconds);
		
		//Number of packets to allocate when the socket is first created
		enum
		{
			kNumPreallocatedPackets = 20	//UInt32
		};

		//Queue of available ReflectorPackets
		OSQueue	fFreeQueue;
		//Queue of attached senders
		OSQueue	fSenderQueue;
		//Can't have new streams attaching to this socket while we
		//are processing data. We could get rid of this mutex if we used an atomic queue
		OSMutex	fMutex;
		
		SInt64	fSleepTime;
};


class ReflectorSocketPool : public UDPSocketPool
{
	public:
	
		ReflectorSocketPool() {}
		virtual ~ReflectorSocketPool() {}
		
		virtual UDPSocketPair*	ConstructUDPSocketPair();
		virtual void			DestructUDPSocketPair(UDPSocketPair *inPair);
};

class ReflectorSender
{
	ReflectorSender(ReflectorStream* inStream, ReflectorSession* inSession,
						bool isRTCP, UInt32 inStreamID);
	~ReflectorSender();
	
	//Used for adjusting sequence numbers in light of thinning
	UInt16		GetPacketSeqNumber(const StrPtrLen& inPacket);
	void		SetPacketSeqNumber(const StrPtrLen& inPacket, UInt16 inSeqNumber);
	bool 		PacketShouldBeThinned(RTPStream* inStream, const StrPtrLen& inPacket);

	//We want to make sure that ReflectPackets only gets invoked when there
	//is actually work to do, because it is an expensive function
	bool		ShouldReflectNow(const SInt64& inCurrentTime, SInt64* ioWakeupTime);
	
	//This function gets data from the multicast source and reflects.
	//Returns the time at which it next needs to be invoked
	void		ReflectPackets(SInt64* ioWakeupTime, OSQueue* inFreeQueue);

	ReflectorStream* 	fStream;
	ReflectorSession* 	fSession;
	bool				fIsRTCP;
	//so we can pick out which RTP stream we are linked up with.
	UInt32		fStreamID;
	
	OSQueueElem	fSocketQueueElem;
	
	OSQueue		fPacketQueue;
	//these serve as an optimization, keeping track of when this
	//sender needs to run so it doesn't run unnecessarily
	bool		fHasNewPackets;
	SInt64		fNextTimeToRun;
			
	//how often to send RRs to the source
	enum
	{
		kRRInterval = 5000,		//SInt64 (every 5 seconds)
		kSmallPacketCutoff = 200//UInt32
	};

	SInt64		fLastRRTime;
	
	friend class ReflectorSocket;
	friend class ReflectorStream;
};


class ReflectorStream
{
	public:
	
		ReflectorStream(ReflectorSession* inSession, UInt32 inSrcAddr,
						UInt16 inSrcPort, UInt16 inTimeToLive, UInt32 inStreamID,
						UInt32 inCodecType, StrPtrLen* inCodecName);
		~ReflectorStream() { if (fCodecName.Ptr != NULL) delete [] fCodecName.Ptr; }
		
		//Call this to initialize the reflector sockets. Returns an RTSPHandler
		//in the event of an error
		RTSPProtocol::RTSPStatusCode BindSockets(RTSPRequestInterface* inRTSPRequest);
		void			CloseSockets();
		
		//Sends an RTCP receiver report to the multicast source
		void	SendReceiverReport();
		
		StrPtrLen*	GetCodecName() 	{ return &fCodecName; }
		UInt32		GetCodecType()	{ return fCodecType; }
		
	private:
	
		UDPSocketPair*		fSockets;
		ReflectorSession*	fSession;
		ReflectorSender		fRTPSender;
		ReflectorSender		fRTCPSender;
		
		//Codec information
		UInt32		fCodecType;
		StrPtrLen	fCodecName;
		
		//Multicast (source) address and port
		UInt32		fAddr;
		UInt16		fPort;
		UInt16		fTtl;
		
		//for issuing RTCP RRs to source
		enum
		{
			kReceiverReportSize = 16,	//UInt32
			kAppSize = 36				//UInt32
		};
	
		char		fReceiverReportBuffer[kReceiverReportSize + kAppSize +
										RTPServerInterface::kMaxCNameLen];
		UInt32*		fEyeLocation;//place in the buffer to write the eye information
		UInt32		fReceiverReportSize;
		
		static ReflectorSocketPool	sSocketPool;
		
		friend class ReflectorSession;
		friend class ReflectorSocket;
};

inline StrPtrLen* ReflectorSession::GetCodecName(UInt32 trackID)
{
	//The input is a trackID, which are arrayindex + 1
	Assert(trackID <= fNumStreams);
	return fStreamArray[trackID - 1]->GetCodecName();
}

inline UInt32 ReflectorSession::GetCodecType(UInt32 trackID)
{
	Assert(trackID <= fNumStreams);
	return fStreamArray[trackID - 1]->GetCodecType();
}


#endif //_REFLECTOR_SESSION_H_
