/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTSPLoggingModule.cpp

	Contains:	Implementation of object defined in .h file.

	$Log: RTSPLoggingModule.cpp,v $
	Revision 1.2  1999/02/19 23:08:28  ds
	Created
	

*/

#include <string.h>
#include "RTSPLoggingModule.h"
#include "RTSPMessages.h"
#include "RTSPRollingLog.h"
#include "OS.h"

RTSPLoggingModule*	RTSPLoggingModule::sLoggingModule = NULL;

bool RTSPLoggingModule::Initialize()
{

	OSMutexLocker locker(&fLogMutex);
	sLoggingModule = this;
	
	//Tell the server that this module is a logging module
	fRollMask = kErrorLoggingModule;
	
	this->CheckErrorLogState();
		
	//format a date for the startup time
	char theDateBuffer[RTSPRollingLog::kMaxDateBufferSizeInBytes];
	bool result = RTSPRollingLog::FormatDate(theDateBuffer);
	
	char tempBuffer[1024];
	if (result)
		::sprintf(tempBuffer, "#Quicktime Streaming Startup STARTUP %s\n", theDateBuffer);

	//log startup message to error log as well.
	if ((result) && (fErrorLog != NULL))
		fErrorLog->WriteToLog(tempBuffer, !kAllowLogToRoll);
	
	return true;

}

void RTSPLoggingModule::Shutdown()
{
	OSMutexLocker locker(&fLogMutex);
	sLoggingModule = NULL;
	
	//log shutdown message
	//format a date for the shutdown time
	char theDateBuffer[RTSPRollingLog::kMaxDateBufferSizeInBytes];
	bool result = RTSPRollingLog::FormatDate(theDateBuffer);
	
	char tempBuffer[1024];
	if (result)
		::sprintf(tempBuffer, "#Quicktime Streaming Shutdown SHUTDOWN %s\n", theDateBuffer);

	if ((result) && (fErrorLog != NULL))
		fErrorLog->WriteToLog(tempBuffer, !kAllowLogToRoll);
}




void RTSPLoggingModule::LogError(	const RTSPModule::Verbosity inVerbosity,
									const RTSPMessages::Message inMessage,
									const RTSPModule::ErrorNumber inErrorNumber)
{
	RTSPLoggingModule::LogErrorPrivate(inVerbosity,
				RTSPServerInterface::GetRTSPMessages()->GetMessage(inMessage)->Ptr, inErrorNumber);
}

void RTSPLoggingModule::LogError(const RTSPModule::Verbosity inVerbosity, const char* inString)
{
	RTSPLoggingModule::LogErrorPrivate(inVerbosity, inString, 0);
}


void RTSPLoggingModule::LogErrorPrivate(	const RTSPModule::Verbosity inVerbosity,
											const char* inMessageP,
											const RTSPModule::ErrorNumber inErrorNumber)
{
	Assert(NULL != inMessageP);
	if (inMessageP == NULL)
		inMessageP = RTSPServerInterface::GetRTSPMessages()->GetMessage(RTSPMessages::kNoMessage)->Ptr;
	if (inMessageP == NULL)
		return;
		
	OSMutexLocker locker(&fLogMutex);
	if (RTSPServerInterface::GetRTSPPrefs()->GetErrorLogVerbosity() >= inVerbosity)
	{
		//The error logger is the bottleneck for any and all messages printed by the server.
		//For debugging purposes, these messages can be printed to stdout as well.
		if (RTSPServerInterface::GetRTSPPrefs()->IsScreenLoggingEnabled())
			::printf("%s (%ld)\n", inMessageP, inErrorNumber);
		
		this->CheckErrorLogState();
		
		if (fErrorLog == NULL)
			return;
			
		//timestamp the error
		char theDateBuffer[RTSPRollingLog::kMaxDateBufferSizeInBytes];
		bool result = RTSPRollingLog::FormatDate(theDateBuffer);
		//for now, just ignore the error.
		if (!result)
			theDateBuffer[0] = '\0';

		Assert(strlen(theDateBuffer) + strlen(inMessageP) < 1024);
		char tempBuffer[1024];
		::sprintf(tempBuffer, "%s: %s (%ld)\n", theDateBuffer, inMessageP, inErrorNumber);
		
		fErrorLog->WriteToLog(tempBuffer, kAllowLogToRoll);
	}
}



void RTSPLoggingModule::RollErrorLog()
{
	const bool kForceEnable = true;

	OSMutexLocker locker(&sLoggingModule->fLogMutex);
	if (sLoggingModule != NULL) 
	{
		sLoggingModule->CheckErrorLogState(kForceEnable);
		
		if (sLoggingModule->fErrorLog != NULL)
			sLoggingModule->fErrorLog->RollLog();
			
		sLoggingModule->CheckErrorLogState(!kForceEnable);
	}
}

void RTSPLoggingModule::CheckErrorLogState(bool forceEnable)
{
	//this function makes sure the logging state is in synch with the preferences.
	//extern variable declared in QTSSPreferences.h

	//check error log.
	if ((NULL == fErrorLog) && (forceEnable || RTSPServerInterface::GetRTSPPrefs()->IsErrorLogEnabled()))
	{
		fErrorLog = new ('erlg') RTSPErrorLog;
		fErrorLog->EnableLog();
	}

	if ((NULL != fErrorLog) && (!forceEnable && !!RTSPServerInterface::GetRTSPPrefs()->IsErrorLogEnabled()))
	{
		delete fErrorLog;
		fErrorLog = NULL;
	}
}

