/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		OSRef.cpp

	Contains:	Implementation of OSRef class. 
					
	$Log: OSRef.cpp,v $
	Revision 1.2  1999/02/19 23:06:14  ds
	Created
	

*/

#include "OSRef.h"

UInt32 	OSRefTableUtils::HashString(StrPtrLen* inString)
{
	Assert(inString != NULL);
	Assert(inString->Ptr != NULL);
	Assert(inString->Len > 0);
	
	//make sure to convert to unsigned here, as there may be binary
	//data in this string
	UInt8* theData = (UInt8*)inString->Ptr;
	
	//divide by 4 and take the characters at quarter points in the string,
	//use those as the basis for the hash value
	UInt32 quarterLen = inString->Len >> 2;
	return (inString->Len * (theData[0] + theData[quarterLen] +
			theData[quarterLen * 2] + theData[quarterLen * 3] +
			theData[inString->Len - 1]));
}

QTSS_ErrorCode OSRefTable::Register(OSRef* ref)
{
	//function currently doesn't check for duplicates
	Assert(ref != NULL);
	if (ref->fInATable)
		return QTSS_InvalidParameter;
		
	Assert(ref->fRefCount == 0);
	
	OSMutexLocker locker(&fMutex);
	//this new entry had better not be a duplicate!
	OSRef* duplicateRef = this->Resolve(&ref->fString);
	if (duplicateRef != NULL)
		return QTSS_DupName;
		
	ref->fInATable = true;
	fTable.Add(ref);
	return QTSS_NoErr;
}



void OSRefTable::UnRegister(OSRef* ref, UInt32 refCount)
{
	Assert(ref != NULL);
	OSMutexLocker locker(&fMutex);

	//make sure that no one else is using the object
	while (ref->fRefCount > refCount)
		ref->fCond.Wait(&fMutex);
	
#if DEBUG
	OSRefKey key(&ref->fString);
	Assert(fTable.Map(&key) != NULL);
#endif
	
	//ok, we now definitely have no one else using this object, so
	//remove it from the table
	fTable.Remove(ref);
	ref->fInATable = false;
}


OSRef*	OSRefTable::Resolve(StrPtrLen* inUniqueID)
{
	Assert(inUniqueID != NULL);
	OSRefKey key(inUniqueID);

	//this must be done atomically wrt the table
	OSMutexLocker locker(&fMutex);
	OSRef* ref = fTable.Map(&key);
	if (ref != NULL)
		ref->fRefCount++;
	return ref;
}

void	OSRefTable::Release(OSRef* ref)
{
	Assert(ref != NULL);
	OSMutexLocker locker(&fMutex);
	ref->fRefCount--;
	//make sure to wakeup anyone who may be waiting for this resource to be released
	ref->fCond.Signal();
}
