/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		OS.h

	Contains:  	OS utility functions

	$Log: OS.cpp,v $
	Revision 1.2  1999/02/19 23:05:30  ds
	Created
		
	
*/

#include <stdlib.h>
#include <string.h>

#ifndef __MW_
#include <sys/time.h>
#include <sys/stat.h>
#endif

#include "OS.h"
#include "OSThread.h"

#if __MacOSX__
#include "timestamp.h"//for fast milliseconds routine
#endif

double 	OS::sDivisor = 0;
double OS::sMicroDivisor = 0;
SInt32 	OS::sMemoryErr = 0;
SInt64  OS::sMsecSince1900 = 0;
SInt64  OS::sInitialMsec = 0;

#if MEMORY_DEBUGGING
OSQueue OS::sMemoryQueue;
OSQueue OS::sTagQueue;
UInt32  OS::sAllocatedBytes = 0;
OSMutex  OS::sMutex('Memy');
#endif

void* operator new (size_t s)
{
	return OS::New (s, '0000', false);
}

void* operator new[](size_t s)
{
	return OS::New (s, '0000', false);
}

void* operator new(size_t s, UInt32 tag)
{
	return OS::New (s, tag, true);
}

void* operator new[](size_t s, UInt32 tag)
{
	return OS::New (s, tag, false);
}

void operator delete(void* mem)
{
#if MEMORY_DEBUGGING
	OS::DebugDelete(mem);
#else
	Assert(mem != NULL);
	free(mem);
#endif
}

void operator delete[](void* mem)
{
	operator delete(mem);
}

void* OS::New(size_t s, UInt32 tag, bool sizeCheck)
{
#if MEMORY_DEBUGGING
	return OS::DebugNew(s, tag, sizeCheck);
#else
	void *m = calloc(1, s);
	if (m == NULL)
		exit(sMemoryErr);
	return m;
#endif
}

#if MEMORY_DEBUGGING
void* OS::DebugNew(size_t s, UInt32 tag, bool sizeCheck)
{
	//also allocate enough space for a Q elem and a long to store the length of this
	//allocation block
	OSMutexLocker locker(&sMutex);
	
	#if VALIDATE_MEMORY_POOL
	ValidateMemoryQueue();
	#endif
	
	UInt32 actualSize = s + sizeof(MemoryDebugging) + (2 * sizeof(tag));
	char *m = (char *)malloc(actualSize);
	if (m == NULL)
		exit(sMemoryErr);

	//mark the beginning and the end with the tag
	memset(m, 0xfe, actualSize);//mark the block with an easily identifiable pattern
	memcpy(m, &tag, sizeof(tag));
	memcpy((m + actualSize) - sizeof(tag), &tag, sizeof(tag));

	//put this chunk on the global chunk queue
	MemoryDebugging* header = (MemoryDebugging*)(m + sizeof(tag));
	memset(header, 0, sizeof(MemoryDebugging));
	header->size = s;
	header->tag = tag;
	header->elem.SetEnclosingObject(header);
	sMemoryQueue.EnQueue(&header->elem);
	sAllocatedBytes += s;
	
	//also update the tag queue
	for (OSQueueIter iter(&sTagQueue); !iter.IsDone(); iter.Next())
	{
		TagElem* elem = (TagElem*)iter.GetCurrent()->GetEnclosingObject();
		if (*elem->tagPtr == tag)
		{
			//verify that the size of this allocation is the same as all others
			//(if requested... some tags are of variable size)
			if (sizeCheck)
				Assert(s == elem->tagSize);
			elem->totMemory += s;
			elem->numObjects++;
			return header + 1;
		}
	}
	//if we've gotten here, this tag doesn't exist, so let's add it.
	TagElem* elem = (TagElem*)malloc(sizeof(TagElem));
	if (elem == NULL)
		exit(sMemoryErr);
	memset(elem, 0, sizeof(TagElem));
	elem->elem.SetEnclosingObject(elem);
	memcpy(elem->tag, &tag, sizeof(tag));
	elem->tagPtr = (UInt32*)&elem->tag;
	elem->tagSize = s;
	elem->totMemory = s;
	elem->numObjects = 1;
	sTagQueue.EnQueue(&elem->elem);
	return m + sizeof(tag) + sizeof(MemoryDebugging);
}

void OS::DebugDelete(void *mem)
{
	OSMutexLocker locker(&sMutex);
	
	#if VALIDATE_MEMORY_POOL
	ValidateMemoryQueue();
	#endif
	
	char* memPtr = (char*)mem;
	MemoryDebugging* memInfo = (MemoryDebugging*)mem;
	memInfo--;//get a pointer to the MemoryDebugging structure
	Assert(memInfo->elem.IsMemberOfAnyQueue());//must be on the memory Queue
	//double check it's on the memory queue
	bool found  = false;
	for (OSQueueIter iter(&sMemoryQueue); !iter.IsDone(); iter.Next())
	{
		MemoryDebugging* check = (MemoryDebugging*)iter.GetCurrent()->GetEnclosingObject();
		if (check == memInfo)
		{
			found = true;
			break;
		}
	}
	Assert(found == true);
	sMemoryQueue.Remove(&memInfo->elem);
	Assert(!memInfo->elem.IsMemberOfAnyQueue());
	sAllocatedBytes -= memInfo->size;
	
	//verify that the tags placed at the very beginning and very end of the
	//block still exist
	memPtr += memInfo->size;
	UInt32* tagPtr = (UInt32*)memPtr;
	Assert(*tagPtr == memInfo->tag);
	memPtr -= sizeof(MemoryDebugging) + sizeof(UInt32) + memInfo->size;
	tagPtr = (UInt32*)memPtr;
	Assert(*tagPtr == memInfo->tag);
	
	//also update the tag queue
	for (OSQueueIter iter2(&sTagQueue); !iter2.IsDone(); iter2.Next())
	{
		TagElem* elem = (TagElem*)iter2.GetCurrent()->GetEnclosingObject();
		if (*elem->tagPtr == memInfo->tag)
		{
			Assert(elem->numObjects > 0);
			elem->numObjects--;
			elem->totMemory -= memInfo->size;
			memset(mem, 0xfd,memInfo->size);
			free(memPtr);
			return;
		}
	}
	Assert(0);//we should always find this tag on the tag queue.
}

void OS::ValidateMemoryQueue()
{
	OSMutexLocker locker(&sMutex);
	for(OSQueueIter iter(&sMemoryQueue); !iter.IsDone(); iter.Next())
	{
		MemoryDebugging* elem = (MemoryDebugging*)iter.GetCurrent()->GetEnclosingObject();
		char* rawmem = (char*)elem;
		rawmem -= sizeof(UInt32);
		UInt32* tagPtr = (UInt32*)rawmem;
		rawmem += sizeof(UInt32) + sizeof(MemoryDebugging) + elem->size;
		tagPtr = (UInt32*)rawmem;
		Assert(*tagPtr == elem->tag);
	}
	for(OSQueueIter iter2(&sMemoryQueue); !iter2.IsDone(); iter2.Next())
	{
		MemoryDebugging* elem2 = (MemoryDebugging*)iter2.GetCurrent()->GetEnclosingObject();
		char* rawmem2 = (char*)elem2;
		rawmem2 -= sizeof(UInt32);
		UInt32* tagPtr2 = (UInt32*)rawmem2;
		Assert(*tagPtr2 == elem2->tag);
	}
}


bool OS::MemoryDebuggingTest()
{
	static char* s20 = "this is 20 characte";
	static char* s30 = "this is 30 characters long, o";
	static char* s40 = "this is 40 characters long, okey dokeys";
	
	void* victim = DebugNew(20, 'tsta', true);
	strcpy((char*)victim, s20);
	MemoryDebugging* victimInfo = (MemoryDebugging*)victim;
	
	ValidateMemoryQueue();
	victimInfo--;
	if (victimInfo->tag != 'tsta')
		return false;
	if (victimInfo->size != 20)
		return false;
		
	void* victim2 = DebugNew(30, 'tstb', true);
	strcpy((char*)victim2, s30);
	ValidateMemoryQueue();
	void* victim3 = DebugNew(20, 'tsta', true);
	strcpy((char*)victim3, s20);
	ValidateMemoryQueue();
	void* victim4 = DebugNew(40, 'tstc', true);
	strcpy((char*)victim4, s40);
	ValidateMemoryQueue();
	void* victim5 = DebugNew(30, 'tstb', true);
	strcpy((char*)victim5, s30);
	ValidateMemoryQueue();
	
	if (sTagQueue.GetLength() != 3)
		return false;
	for (OSQueueIter iter(&sTagQueue); !iter.IsDone(); iter.Next())
	{
		TagElem* elem = (TagElem*)iter.GetCurrent()->GetEnclosingObject();
		if (*elem->tagPtr == 'tstb')
		{
			if (elem->tagSize != 30)
				return false;
			if (elem->numObjects != 2)
				return false;
		}
		else if (*elem->tagPtr == 'tsta')
		{
			if (elem->tagSize != 20)
				return false;
			if (elem->numObjects != 2)
				return false;
		}
		else if (*elem->tagPtr == 'tstc')
		{
			if (elem->tagSize != 40)
				return false;
			if (elem->numObjects != 1)
				return false;
		}
		else
			return false;
	}
	
	DebugDelete(victim3);
	ValidateMemoryQueue();
	DebugDelete(victim4);
	ValidateMemoryQueue();

	if (sTagQueue.GetLength() != 3)
		return false;
	for (OSQueueIter iter2(&sTagQueue); !iter2.IsDone(); iter2.Next())
	{
		TagElem* elem = (TagElem*)iter2.GetCurrent()->GetEnclosingObject();
		if (*elem->tagPtr == 'tstb')
		{
			if (elem->tagSize != 30)
				return false;
			if (elem->numObjects != 2)
				return false;
		}
		else if (*elem->tagPtr == 'tsta')
		{
			if (elem->tagSize != 20)
				return false;
			if (elem->numObjects != 1)
				return false;
		}
		else if (*elem->tagPtr == 'tstc')
		{
			if (elem->tagSize != 40)
				return false;
			if (elem->numObjects != 0)
				return false;
		}
		else
			return false;
	}
	
	if (sMemoryQueue.GetLength() != 3)
		return false;
	DebugDelete(victim);
	ValidateMemoryQueue();
	if (sMemoryQueue.GetLength() != 2)
		return false;
	DebugDelete(victim5);
	ValidateMemoryQueue();
	if (sMemoryQueue.GetLength() != 1)
		return false;
	DebugDelete(victim2);
	ValidateMemoryQueue();
	if (sMemoryQueue.GetLength() != 0)
		return false;
	DebugDelete(victim4);
	return true;
}
#endif

void OS::Initialize()
{
#if __MacOSX__
	//Setup divisor for milliseconds
	double divisorNumerator = 0;
	double divisorDenominator = 0;
	struct timescale theTimescale;
	
	(void)utimescale(&theTimescale);
	
	divisorNumerator = (double)theTimescale.tsc_numerator;
	divisorDenominator = (double)theTimescale.tsc_denominator;//we want milliseconds, not micro
	sMicroDivisor = divisorNumerator / divisorDenominator;
	divisorDenominator *= 1000;//we want milliseconds, not micro
	sDivisor = divisorNumerator / divisorDenominator;
#endif

	//setup t0 value for msec since 1900
	struct timeval t;
	struct timezone tz;
	int theErr = ::gettimeofday(&t, &tz);
	Assert(theErr == 0);

	//t.tv_sec is number of seconds since Jan 1, 1970. Convert to seconds since 1900
	sMsecSince1900 = 24 * 60 * 60;
	sMsecSince1900 *= (70 * 365) + 17;
	sMsecSince1900 += (SInt64)t.tv_sec;
	sMsecSince1900 *= 1000;//convert to msec from sec
	
	sInitialMsec = Milliseconds();
}

SInt64 OS::Milliseconds()
{
#if __MacOSX__
	//DMS - here is that superfast way of getting milliseconds. Note that this is time
	//relative to when the machine booted.
	UInt64 theCurTime = timestamp();
	Assert(sDivisor > 0);
	
	double theDoubleTime = (double)theCurTime;
	theDoubleTime *= sDivisor;
	return ((SInt64)theDoubleTime) - sInitialMsec;
#else
	struct timeval t;
	struct timezone tz;
	int theErr = ::gettimeofday(&t, &tz);
	Assert(theErr == 0);

	SInt64 curTime;
	curTime = t.tv_sec;
	curTime *= 1000;				// sec -> msec
	curTime += t.tv_usec / 1000;	// usec -> msec

	return curTime - sInitialMsec;
#endif
}

SInt64 OS::Microseconds()
{
#if __MacOSX__
	UInt64 theCurTime = timestamp();	
	double theDoubleTime = (double)theCurTime;
	theDoubleTime *= sMicroDivisor;
	return (SInt64)theDoubleTime;
#else
	struct timeval t;
	struct timezone tz;
	int theErr = ::gettimeofday(&t, &tz);
	Assert(theErr == 0);

	SInt64 curTime;
	curTime = t.tv_sec;
	curTime *= 1000;		// sec -> msec
	//curTime = t.tv_sec * 1000;
	curTime += t.tv_usec;

	return curTime - (sInitialMsec * 1000);
#endif
}

SInt64	OS::HostToNetworkSInt64(SInt64 hostOrdered)
{
	// THIS IS BAD! Really, we should compile in the byteswapping based on the endian-ness
	// of the processor, not whether we are on MacOSX or not (in this scheme, MacOSX is
	// big-endian and all other OSes aren't). Currently, I don't know of a way to do this.
#if __MacOSX__
	return hostOrdered;
#else
	//let's figure out a proper scheme for enabling this code when we move to a littlendian platform
	return ( (hostOrdered << 56) | (0x00ff000000000000 & (hostOrdered << 40))
		| (0x00ff0000000000 & (hostOrdered << 24)) | (0x00ff00000000 & (hostOrdered << 8))
		| (0x00ff000000 & (hostOrdered >> 8)) | (0x00ff0000 & (hostOrdered >> 24))
		| (0x00ff00 & (hostOrdered >> 40)) | (0x00ff & (hostOrdered >> 56)) );
#endif
}
		


QTSS_ErrorCode OS::MakeDir(char *inPath, mode_t mode)
{
	struct stat theStatBuffer;
	if (::stat(inPath, &theStatBuffer) == -1)
	{
		//this directory doesn't exist, so let's try to create it
		if (::mkdir(inPath, mode) == -1)
			return (QTSS_ErrorCode)OSThread::GetErrno();
	}
	else if (!S_ISDIR(theStatBuffer.st_mode))
		return QTSS_FileExists;//there is a file at this point in the path!

	//directory exists
	return QTSS_NoErr;
}

QTSS_ErrorCode OS::RecursiveMakeDir(char *inPath, mode_t mode)
{
	Assert(inPath != NULL);
	
	//iterate through the path, replacing '/' with '\0' as we go
	char *thePathTraverser = inPath;
	
	//skip over the first / in the path.
	if (*thePathTraverser == '/')
		thePathTraverser++;
		
	while (*thePathTraverser != '\0')
	{
		if (*thePathTraverser == '/')
		{
			//we've found a filename divider. Now that we have a complete
			//filename, see if this partial path exists.
			
			//make the partial path into a C string
			*thePathTraverser = '\0';
			QTSS_ErrorCode theErr = MakeDir(inPath, mode);
			//there is a directory here. Just continue in our traversal
			*thePathTraverser = '/';

			if (theErr != QTSS_NoErr)
				return theErr;
		}
		thePathTraverser++;
	}
	
	//need to create the last directory in the path
	return MakeDir(inPath, mode);
}

