/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */

#include "playlist_elements.h"
#include "playlist_utils.h"



// ************************
//
// MediaStream
//
// ************************

MediaStream::MediaStream(void)
{
	memset( (char *)&fData, '\0', sizeof(fData));
	fSend = true;
}
MediaStream::~MediaStream()
{
	if (fData.fSoundDescriptionBuffer) delete[] fData.fSoundDescriptionBuffer;
}


UInt32 MediaStream::GetACName(char* ioCNameBuffer)
{
	//clear out the whole buffer
	::memset(ioCNameBuffer, 0, kMaxCNameLen);
	
	//cName identifier
	ioCNameBuffer[0] = 1;
	
	//Unique cname is constructed from the base name and the current time
	::sprintf(&ioCNameBuffer[1], " %s%qd", "QTSS", PlayListUtils::Milliseconds() / 1000);
	UInt32 cNameLen = ::strlen(ioCNameBuffer);
	//2nd byte of CName should be length
	ioCNameBuffer[1] = cNameLen - 2;//don't count indicator or length byte

	//pad length to a 4 byte boundary
	UInt32 padLength = cNameLen % 4;
	if (padLength > 0)
		cNameLen += 4 - padLength;
	
	return cNameLen;
}

void MediaStream::TestAndIncSoundDescriptor(RTpPacket *packetPtr)
{ // currently not executed
	short test = 0;
	do 
	{
		if (!fData.fIsSoundStream) break;
		if (!packetPtr->HasSoundDescription()) break;
		if (!fData.fSoundDescriptionBuffer) break;

		SoundDescription *packetSDPtr = NULL;
		SoundDescription *savedSDPtr = (SoundDescription *) fData.fSoundDescriptionBuffer;
		(void) packetPtr->GetSoundDescriptionRef(&packetSDPtr);
		long descSize = packetPtr->fSoundDescriptionLen;
		
		if (descSize == 0) break;
		if (descSize > eMaxSoundDescriptionSize) break;
		
		if (fData.fSavedSoundDescSize == descSize)
		{	test = ::memcmp(packetSDPtr, fData.fSoundDescriptionBuffer, descSize);
		} 
		else test = 1; // they are different sizes so it is a new sample description

		if (test != 0) // they are different
		{	::memcpy(savedSDPtr, packetSDPtr, descSize); 
			fData.fSavedSoundDescSize = descSize;
			fData.fSavedDataRefIndex ++ ; // it is different than saved so change the index
		}
		packetSDPtr->dataRefIndex = htons(fData.fSavedDataRefIndex);
		 
	} while (false);
}


void MediaStream::UpdatePacketInStream(RTpPacket *packetPtr)
{
	UInt32	curSSRC = 0;
	UInt32	curRTpTimeStamp = 0;
	UInt16	curRTpSequenceNumber = 0;
	UInt32	newRTpTimeStamp = 0;
	UInt16	newRTpSequenceNumber = 0;
	UInt32	newSSRC = 0;
	unsigned char curPayload;
	unsigned char newPayload;
	packetPtr->GetHeaderInfo(&curRTpTimeStamp, &curRTpSequenceNumber, &curSSRC, &curPayload);

	newSSRC = fData.fInitSSRC;
	MapToStream(curRTpTimeStamp, curRTpSequenceNumber, curPayload, &newRTpTimeStamp, &newRTpSequenceNumber, &newPayload);
	
	if (fData.fIsVideoStream) LogStr("video ");
	if (fData.fIsSoundStream) LogStr("audio ");
	
	packetPtr->SetHeaderInfo(newRTpTimeStamp, newRTpSequenceNumber,newSSRC,newPayload);

	//TestAndIncSoundDescriptor(packetPtr); // put in to track QuickTime format sound descriptors and flag change in sample types
}

void MediaStream::MapToStream(UInt32 curRTpTimeStamp, UInt16 curRTpSequenceNumber, unsigned char curPayload, UInt32 *outRTpTimeStampPtr, UInt16 *outRTpSequenceNumberPtr, unsigned char *outPayloadPtr)
{
	if (fData.fNewMovieStarted == true)		// this is the first packet in a new movie
	{	
		fData.fNewMovieStarted = false;	
	}  
			
	fData.fCurStreamRTpSequenceNumber++; //  the stream sequence number
	
	UInt64 curSequenceNumber = (UInt64) fData.fCurStreamRTpSequenceNumber + (UInt64) fData.fSeqRandomOffset;	
	UInt64 curTimeStamp = (UInt64) curRTpTimeStamp + (UInt64) fData.fMediaStartOffsetMediaScale + (UInt64) fData.fRTpRandomOffset;
	 	
	UInt32 outTime = (UInt32) ( (UInt64) curTimeStamp & (UInt64) 0xFFFFFFFF ); 
	UInt16 outSeq =  (UInt16) ( (UInt64) curSequenceNumber & (UInt64) 0xFFFF ); 
	PayLoad *firstPayLoadPtr =  (fData.fMovieMediaTypePtr)->fPayLoadTypes.Begin(); // potential problem; assumes first payload per track is this payload
	short	*payLoadIDPtr = (fData.fMovieMediaTypePtr)->fPayLoads.Begin();
	unsigned char outPayload = curPayload;
	if (firstPayLoadPtr)
	{	
		outPayload = (unsigned char) ( 0x7F & firstPayLoadPtr->payloadID);
		outPayload |= (curPayload & 0x80);// the movie payload marker
	}
		
	if (outRTpTimeStampPtr) *outRTpTimeStampPtr = outTime;
	if (outRTpSequenceNumberPtr) *outRTpSequenceNumberPtr = outSeq;
	if (outPayloadPtr)  *outPayloadPtr = outPayload;
}

void MediaStream::BuildStaticRTCpReport()
{
	char theTempCName[kMaxCNameLen];
	UInt32 cNameLen = GetACName(theTempCName);
	
	//write the SR & SDES headers
	UInt32* theSRWriter = (UInt32*)&fData.fSenderReportBuffer;
	*theSRWriter = htonl(0x80c80006);
	theSRWriter += 7;
	//SDES length is the length of the CName, plus 2 32bit words, minus 1
	*theSRWriter = htonl(0x81ca0000 + (cNameLen >> 2) + 1);
	::memcpy(&fData.fSenderReportBuffer[kSenderReportSizeInBytes], theTempCName, cNameLen);
	fData.fSenderReportSize = kSenderReportSizeInBytes + cNameLen;
}

void MediaStream::InitIfAudio()
{
	if (fData.fStreamMediaTypePtr)
	{
		SimpleString audioStr("audio");
		fData.fIsSoundStream = SimpleParser::Compare(&audioStr, &(fData.fStreamMediaTypePtr->fTheTypeStr) );
		if (fData.fIsSoundStream)
		{				
			fData.fSoundDescriptionBuffer = new char[eMaxSoundDescriptionSize];
			::memset(fData.fSoundDescriptionBuffer, 0, eMaxSoundDescriptionSize );
		}
		else
		{
			SimpleString videoStr("video");
			fData.fIsVideoStream = SimpleParser::Compare(&videoStr, &(fData.fStreamMediaTypePtr->fTheTypeStr) );
		}
	}	

}


void MediaStream::StreamStart(SInt64 startTime)
{

	fData.fStreamStartTime = startTime;
	fData.fMovieEndTime = startTime;
	fData.fLastSenderReportTime = 0;

	//for RTCp SRs, we also need to store the play time in NTP
	fData.fNTPPlayTime = PlayListUtils::ConvertToSecondsSince1900(startTime);

	fData.fCurStreamRTpSequenceNumber =  0;
	fData.fMovieStartOffsetMicro = 0;	
	
	fData.fNewStreamStarted = true;
	fData.fSeqRandomOffset = PlayListUtils::Random();
	fData.fRTpRandomOffset = PlayListUtils::Random();
	
//	fData.fSeqRandomOffset = -1000; // test roll over
//	fData.fRTpRandomOffset = -100000; // test roll over
	
	InitIfAudio();
	
	//Build a static RTCp sender report (this way, only the info that changes
	//on the fly will have to be written)
	BuildStaticRTCpReport();	
}

void MediaStream::MovieStart(SInt64 startTime)
{
	fData.fNewMovieStarted = true;
	fData.fMovieStartTime = startTime;
	
	if (fData.fMovieMediaTypePtr && fData.fRTPFilePtr)
	{	UInt32 trackID = fData.fMovieMediaTypePtr->fTrackID;
		fData.fMovieMediaTypePtr->fTimeScale = fData.fRTPFilePtr->GetTrackTimeScale(trackID);
		
		UInt64 lastMoviedurationMicro = (UInt64) ( (Float64)  (fData.fLastMovieDurationSecs * (Float64) PlayListUtils::eMicro) ) ; // add the length of the last movie
		fData.fMovieStartOffsetMicro += lastMoviedurationMicro; 
//zz		printf("lastMoviedurationMicro = %lu \n", (UInt32) lastMoviedurationMicro);
//zz		printf("fMovieStartOffsetMicro = %lu \n", (UInt32) fData.fMovieStartOffsetMicro);
		
		if (fData.fNewStreamStarted)
		{	fData.fNewStreamStarted = false;
			fData.fMovieEndTime = startTime; // first movie in stream has 0 movieInterval time.
		}
		
//		UInt64 movieInterval = startTime - fData.fMovieEndTime; // important
//		fData.fMovieStartOffsetMicro += movieInterval; // important accounts for delay between movies
		
//zz		printf("movieInterval = %lu \n", (UInt32) movieInterval);
//zz		printf("fMovieStartOffsetMicro = %lu \n", (UInt32) fData.fMovieStartOffsetMicro);
		
		Float64 mediaOffSet =   (Float64) fData.fMovieStartOffsetMicro / (Float64)  PlayListUtils::eMicro; // convert to float seconds
		mediaOffSet = mediaOffSet  * (Float64)  fData.fMovieMediaTypePtr->fTimeScale; // mediaOffset in media time scale
		fData.fMediaStartOffsetMediaScale =  (UInt64) mediaOffSet; // convert float time units to UInt64

//zz		printf("fMediaStartOffsetMediaScale = %lu \n", (UInt32) fData.fMediaStartOffsetMediaScale);

		fData.fLastMovieDurationSecs = fData.fRTPFilePtr->GetMovieDuration();
	}


}

void MediaStream::MovieEnd(SInt64 endTime)
{
	fData.fMovieEndTime = endTime;
	fData.fMovieMediaTypePtr = NULL;
	fData.fRTPFilePtr = NULL;
}

SInt16	MediaStream::Accounting(RTpPacket *packetPtr)
{
	SInt16 result = -1;
	
	fData.fBytesSent += packetPtr->fLength;
	fData.fPacketsSent ++;
		
	UInt32 lastTime = fData.fTimeStamp; 	
	unsigned char payload;
	packetPtr->GetHeaderInfo(&fData.fTimeStamp, &fData.fLastSequenceNumber, &fData.fSsrc, &payload);
	
	fData.fLastTimeStampOffset = fData.fTimeStamp - lastTime; 	
	
	result = 0;
	
	return result;
}

SInt16	MediaStream::Send(RTpPacket *packetPtr)
{

	SInt16 result = -1;
	
	do 
	{
		if (fData.fMovieMediaTypePtr == NULL) break;
		
		UpdatePacketInStream(packetPtr);
		
		result = Accounting(packetPtr);
		if (result) break;
		
		if (fSend)
		{	result = fData.fSocketPair->SendRTp(packetPtr->fThePacket, packetPtr->fLength);
		}
	}
	while (false);
	
	return result;
}

void MediaStream::ReceiveOnPorts()
{
	fData.fSocketPair->RecvRTp(fData.fPortRTpReadBuff.fReadBuffer, ReceiveBuffer::kReadBufferSize, &fData.fPortRTpReadBuff.fReceiveLen);
	fData.fSocketPair->RecvRTCp(fData.fPortRTCpReadBuff.fReadBuffer, ReceiveBuffer::kReadBufferSize, &fData.fPortRTCpReadBuff.fReceiveLen);
}

int MediaStream::UpdateSenderReport(SInt64 theTime)
{
	int result = 0;
	
	SInt64 timeToSend = fData.fLastSenderReportTime + (kSenderReportIntervalInSecs * PlayListUtils::eMilli);

	if (theTime > timeToSend )
	{
		fData.fLastSenderReportTime = theTime;
		UInt32* theReport = (UInt32*) fData.fSenderReportBuffer;
		
		theReport++;
		*theReport = htonl(fData.fSsrc);
		
		theReport++;
		SInt64* theNTPTimestampP = (SInt64*)theReport;		
		*theNTPTimestampP = PlayListUtils::HostToNetworkSInt64(fData.fNTPPlayTime +
								PlayListUtils::ConvertMsecToFixed64Sec(theTime - fData.fStreamStartTime));
								
		theReport += 2;
		*theReport = htonl(fData.fTimeStamp);

		theReport++;		
		fData.fPacketCount = (UInt32) fData.fPacketsSent;
		*theReport = htonl(fData.fPacketCount);
	
		theReport++;
		fData.fByteCount = (UInt32)  fData.fBytesSent; 
		*theReport = htonl(fData.fByteCount);
		
		theReport += 2;
		*theReport = htonl(fData.fSsrc);
		
		LogStr("Sender Report\n");
		LogUInt("NTP ",(UInt32) ((*theNTPTimestampP) >> 32)," ");
		LogUInt(" ",(UInt32) ((*theNTPTimestampP) & 0xFFFFFFFF), "\n" );
		LogUInt("time stamp = ", fData.fTimeStamp, "\n");
		LogInt("SSRC = ", fData.fSsrc, "\n");
		LogUInt("Packets sent = ", fData.fPacketCount, "\n");
		LogUInt("Bytes sent = ", fData.fByteCount, "\n");
		LogBuffer();
		result = fData.fSocketPair->SendRTCp(fData.fSenderReportBuffer, fData.fSenderReportSize);

	}

	return result;
}

// ************************
//
// UDPSOCKETPAIR
//
// ************************


void UDPSocketPair::Close()
{
	if (fSocketRTp != 0) ::close(fSocketRTp);
	if (fSocketRTCp != 0) ::close(fSocketRTCp);
	fSocketRTp = 0;
	fSocketRTCp = 0;
}

SInt16 UDPSocketPair::Open()
{
	SInt16 result = 0;

	do
	{
		Close();
		
		fSocketRTp = ::socket(PF_INET, SOCK_DGRAM, 0);
		if (fSocketRTp == kInvalidSocket)
		{	result = kInvalidSocket; 
			break;
		}
		
		fSocketRTCp = ::socket(PF_INET, SOCK_DGRAM, 0);
		if (fSocketRTCp == kInvalidSocket)
		{	result = kInvalidSocket; 
			break;
		}
		
		int flag; 
		int val;
		flag = ::fcntl(fSocketRTp, F_GETFL, 0);
		val = ::fcntl(fSocketRTp, F_SETFL, flag | O_NONBLOCK);
		if( val < 0 ) { result = -1; break; }

		flag = ::fcntl(fSocketRTCp, F_GETFL, 0);
		val = ::fcntl(fSocketRTCp, F_SETFL, flag | O_NONBLOCK);
		if( val < 0 ) { result = -1; break; }
			
	} while (false);
	
	if (result != 0)
	{	if (fSocketRTp != 0) ::close(fSocketRTp);
		if (fSocketRTCp != 0) ::close(fSocketRTCp);
		fSocketRTp = 0;
		fSocketRTCp = 0;
	}
		
	return result;
}

void UDPSocketPair::InitPorts(UInt32 addr)
{
	::memset(&fLocalAddrRTp, 0, sizeof(fLocalAddrRTp));
	fLocalAddrRTp.sin_family = PF_INET;
	fLocalAddrRTp.sin_port = htons(0);
	fLocalAddrRTp.sin_addr.s_addr = htonl(addr);

	::memset(&fLocalAddrRTCp, 0, sizeof(fLocalAddrRTCp));
	fLocalAddrRTCp.sin_family = PF_INET;
	fLocalAddrRTCp.sin_port = htons(0);
	fLocalAddrRTCp.sin_addr.s_addr = htonl(addr);

	::memset(&fDestAddrRTp, 0, sizeof(fDestAddrRTp));
	fDestAddrRTp.sin_family = PF_INET;
	fDestAddrRTp.sin_port = htons(0);
	fDestAddrRTp.sin_addr.s_addr = htonl(addr);

	::memset(&fDestAddrRTCp, 0, sizeof(fDestAddrRTCp));
	fDestAddrRTCp.sin_family = PF_INET;
	fDestAddrRTCp.sin_port = htons(0);
	fDestAddrRTCp.sin_addr.s_addr = htonl(addr);
}

SInt16 UDPSocketPair::Bind(UInt32 addr)
{
	int err = -1;
	
	if ( (fSocketRTp == kInvalidSocket) || (fSocketRTCp == kInvalidSocket) )
		return -1;
	
	InitPorts(addr);

	for (int count = 0; count < eBindMaxTries; count ++)
	{	UInt32 PortRTp = eSourcePortStart + (PlayListUtils::Random() % eSourcPortRange);
		UInt32 PortRTCp = PortRTp + 1; // keep them together for clarity

		fLocalAddrRTp.sin_port = htons(PortRTp);
		fLocalAddrRTCp.sin_port = htons(PortRTCp);
		
		err = ::bind(fSocketRTp, (sockaddr *)&fLocalAddrRTp, sizeof(fLocalAddrRTp));
		if (err != 0)
		{
			fLocalAddrRTp.sin_port = 0;
			fLocalAddrRTp.sin_addr.s_addr = 0;
			continue;
		}

		err = ::bind(fSocketRTCp, (sockaddr *)&fLocalAddrRTCp, sizeof(fLocalAddrRTCp));	
		if (err != 0)
		{
			err = ::close(fSocketRTp);
			fLocalAddrRTp.sin_port = 0;
			fLocalAddrRTp.sin_addr.s_addr = 0;

			fLocalAddrRTCp.sin_port = 0;
			fLocalAddrRTCp.sin_addr.s_addr = 0;
			continue;
		}  
		
		if (err == 0) 
		{	fState |= kBound;
			break;
		}
	}
	
	return err;
}


SInt16 UDPSocketPair::SendTo(int socket, sockaddr *destAddrPtr, char* inBuffer, UInt32 inLength )
{
	SInt16 result = -1;
	do
	{
		if (inBuffer == NULL) break;		
		if (destAddrPtr == NULL) break;			
		if (socket == kInvalidSocket) break;
		
		::sendto(socket, inBuffer, inLength, 0, destAddrPtr, sizeof(sockaddr));
		
		result = 0;
	} while (false);
		
	if (result != 0) printf("UDP SENDTO ERROR!\n");
	return result;
}


SInt16 UDPSocketPair::SendRTp(char* inBuffer, UInt32 inLength)
{	
	return SendTo(fSocketRTp, (sockaddr*)&fDestAddrRTp, inBuffer, inLength );
}

SInt16 UDPSocketPair::SendRTCp(char* inBuffer, UInt32 inLength)
{	
	return SendTo(fSocketRTCp, (sockaddr*)&fDestAddrRTCp, inBuffer, inLength );
}

SInt16  UDPSocketPair::SetDestination (char *destAddress,UInt16 destPortRTp, UInt16 destPortRTCp)
{
	SInt16 result = -1;

	if (destAddress != NULL)
	{
		fDestAddrRTp = fLocalAddrRTp; 
		fDestAddrRTp.sin_port = htons(destPortRTp); 
		inet_aton(destAddress,&fDestAddrRTp.sin_addr);
		
		fDestAddrRTCp = fLocalAddrRTCp;
		fDestAddrRTCp.sin_port = htons(destPortRTCp);		
		inet_aton(destAddress,&fDestAddrRTCp.sin_addr);
		
		result = 0;
	}
	return result;
}

SInt16 UDPSocketPair::RecvFrom(sockaddr *RecvRTpddrPtr, int socket, char* ioBuffer, UInt32 inBufLen, UInt32* outRecvLen)
{
	SInt16 	result = -1;
	
	do
	{	
		if (ioBuffer == NULL) break;
		if (RecvRTpddrPtr == NULL) break;
		if (socket == kInvalidSocket) break;
		
		int addrLen = sizeof(sockaddr);
		
		SInt32 theRecvLen = ::recvfrom(socket, ioBuffer, inBufLen, 0, RecvRTpddrPtr, &addrLen);		
		if (theRecvLen == -1) break;
		
		if (outRecvLen) *outRecvLen = (UInt32)theRecvLen;
	} while (false);
	return result;		
}

SInt16 UDPSocketPair::RecvRTp(char* ioBuffer, UInt32 inBufLen, UInt32* outRecvLen)
{
	return RecvFrom( (sockaddr *)&fDestAddrRTp.sin_addr, fSocketRTp, ioBuffer, inBufLen, outRecvLen);
}

SInt16 UDPSocketPair::RecvRTCp(char* ioBuffer, UInt32 inBufLen, UInt32* outRecvLen)
{
	return  RecvFrom( (sockaddr *)&fDestAddrRTCp, fSocketRTCp, ioBuffer, inBufLen, outRecvLen);
}

// ************************
//
// RTpPacket
//
// ************************

SInt16 RTpPacket::GetSoundDescriptionRef(SoundDescription **soundDescriptionPtr)
{
	SInt16 result = -1;
	
	if (fThePacket && soundDescriptionPtr)
	{
		//::memset(soundDescriptionPtr, 0, *maxLen);
		long minSoundLength = sizeof(SoundHeader) + sizeof(SoundDescription) + kRTpHeaderSize;
		if ( fLength >= minSoundLength  )
		{
			char *offsetPtr = fThePacket + kRTpHeaderSize + sizeof(SoundHeader);
			//::memcpy( (char *) soundDescriptionPtr, offsetPtr, sizeof(long) );
			*soundDescriptionPtr = (SoundDescription *) offsetPtr;
			long descSize = ntohl( (**soundDescriptionPtr).descSize);
			//::memcpy( (char *) soundDescriptionPtr, offsetPtr, descSize );
			fSoundDescriptionLen = descSize;
			result = 0;
		}
	}
	
	return result;
}

bool RTpPacket::HasSoundDescription() 
{
	bool result = false;
	
	if (fThePacket)
	{
	
//		WritePacketToLog(fThePacket, fLength);
		long minSoundLength = sizeof(SoundHeader) + sizeof(SoundDescription) + kRTpHeaderSize;
		if (fLength >= minSoundLength ) 
		{
			char *offsetPtr = fThePacket + kRTpHeaderSize;
			SoundHeader* testHeaderPtr = (SoundHeader*) offsetPtr;
			do
			{
				if (testHeaderPtr->bytes[0] != 0x17) break;
				if (testHeaderPtr->sndtype[0] != 's') break;
				if (testHeaderPtr->sndtype[1] != 'o') break;
				if (testHeaderPtr->sndtype[2] != 'u') break;
				if (testHeaderPtr->sndtype[3] != 'n') break;

				fHasSoundDescription = result = true;
				
			} while (false);
		}
		
		if (result) 
		{	
			LogStr("has sound description soun\n");
		}
		PrintLogBuffer(true);
	}
	
	return result;
}

SInt16 RTpPacket::GetHeaderInfo(UInt32 *timeStampPtr, UInt16 *sequencePtr,UInt32 *SSRCPtr, unsigned char*payloadTypeAndMarkPtr)
{
	SInt16 result = -1;
	
	
	unsigned char *header8Ptr = (unsigned char *) fThePacket;
	UInt16* header16Ptr = (UInt16*)fThePacket;
	UInt32* header32Ptr = (UInt32*)fThePacket;	
	
	if (fThePacket && timeStampPtr && sequencePtr && SSRCPtr && payloadTypeAndMarkPtr)
	{
		*payloadTypeAndMarkPtr = header8Ptr[cPayloadType];
		*sequencePtr = ntohs(header16Ptr[cSequenceNumber]);
		*timeStampPtr = ntohl(header32Ptr[cTimeStamp]);
		*SSRCPtr = ntohl(header32Ptr[cSSRC]);
		result = 0;	
	}
	
	return result;
}



SInt16 RTpPacket::SetHeaderInfo(UInt32 timeStamp, UInt16 sequence, UInt32 SSRC, unsigned char payloadTypeAndMark)
{
	SInt16 result = -1;
	
	unsigned char *header8Ptr = (unsigned char *) fThePacket;
	UInt16* header16Ptr = (UInt16*)fThePacket;
	UInt32* header32Ptr = (UInt32*)fThePacket;	
	
	
	if (fThePacket)
	{
		LogInt("sequence = ", sequence, " ");
		LogUInt("time = ", timeStamp, " ");
//		LogUInt("old payload = ",( UInt32 ) (header8Ptr[cPayloadType] & 0x7F)," ");
		LogUInt("payload = ",( UInt32 ) (payloadTypeAndMark & 0x7F) ," ");
//		LogUInt("old marker = ", ( UInt32 ) (header8Ptr[cPayloadType] & 0x80) ," ");
//		LogUInt("marker = ", ( UInt32 ) (payloadTypeAndMark & 0x80) ," ");
		LogUInt("ssrc = ", SSRC, "\n");

		header8Ptr[cPayloadType] = payloadTypeAndMark;
		header16Ptr[cSequenceNumber] = htons(sequence);
		header32Ptr[cTimeStamp] = htonl(timeStamp);		
		header32Ptr[cSSRC] = htonl(SSRC);		
		result = 0;	

		LogBuffer();
	}

	
	return result;
}


