/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */

/*

	8.3.99 - rt added loop detection.
	
	7.30.99 - jm added *PLAY-LIST* check
			- added error messages
*/

#include "plassert.h"

#include "PickerFromFile.h"


#include <stdlib.h>	
#include "GetWord.h"
#include "Trim.h"

#include "PathDelimiter.h"

#define kMaxPickerPath 512

#include <stdio.h>	

static bool CompareNameToElement( PLDoubleLinkedListNode<LoopDetectionListElement>* node, void* name );
static void DisplayPickerErr( int pickErr, const char *message, const char*fname, int lineCount, const char*lineBuff );



static bool CompareNameToElement( PLDoubleLinkedListNode<LoopDetectionListElement>* node, void* name )
{
	if ( !::strcmp( node->mElement->mPathName, (const char*)name ) )
		return true;
	
	return false;
}


static void DisplayPickerErr( int pickErr, const char *message, const char*fname, int lineCount, const char*lineBuff )
{
	char *errMessage;
	
	::printf( "- %s:\n", message );
	
	if ( lineCount )
		::printf( "  Playlist: %s, line# %i\n", fname, lineCount );
	else
		::printf( "  Playlist: %s\n", fname );
	
	if ( lineBuff )
		::printf( "  Playlist text: %s", lineBuff ); // lineBuff already includes a \n
	
	switch ( pickErr )
	{
		case kPickerPopulateLoopDetected:
			errMessage = "Include would create a loop.\n";
			break;

		case kPickerPopulateBadFormat:
			errMessage = "Playlist file is missing *PLAY-LIST* identifier.\n";
			break;
			
		case kPickerPopulateFileError:
			errMessage = "Playlist file could not be opened.\n";
			break;
			
		case kPickerPopulateNoMem:
		default:
			errMessage = "Internal error occurred.\n";
			break;
	
	}
	::printf( "  Reason: %s\n", errMessage);

}


int PopulatePickerFromFile( PlaylistPicker* picker, char* fname, const char* basePath, LoopDetectionList *ldList )
{
	ASSERT( picker );
	ASSERT( fname );
	
	FILE*						weightings = NULL;
	LoopDetectionListElement*	ldElement = NULL;
	LoopDetectionNode*			ldNode = NULL;
	int							lineCount = 0;
	int 						pickErr = kPickerPopulateNoErr;	
	char						path[kMaxPickerPath];
	
	
		
#if kPartialPathBeginsWithDelimiter
	if ( *fname == kPathDelimiterChar )
	{
		if ( *basePath )
			fname++;
#else
	if ( *fname != kPathDelimiterChar )
	{
#endif
		// it's a partial path, expand it to include all
		// previously traversed paths
		::strcpy( path, basePath );
		::strcat( path, fname );
			
	}
	else
	{
		// it's an absolute reference. use the path
		// part of this for the new basePath
		::strcpy( path, fname );
		
	}
	
	// path is now either an absolute or working directory
	// referenced partial path to the playlist file.
	int len = strlen(path);
	char lastChar = path[len-1];
	if (lastChar == '\n' || lastChar == '\r' || lastChar == ' ')
		path[len-1] = '\0';

	// ldList is passed as NULL by the initial caller.  recursive calls
	// pass along the ldList we create hre
	if ( ldList == NULL )
		ldList = new LoopDetectionList;

	ASSERT( ldList );
	
	if ( !ldList )
		pickErr = kPickerPopulateNoMem;
	
	
	if ( !pickErr )
	{
		if ( ldList->ForEachUntil( CompareNameToElement, path ) )
		{
			// we're already in the include chain, this is a loop
			// print a warning (error?) and continue past the loop.
			//printf("- Playlists include loop at file: %s\n", path );
			pickErr = kPickerPopulateLoopDetected;
		}
	}
	
	
	
	if ( !pickErr )
	{
		ldElement = new LoopDetectionListElement( path );
		
		ASSERT( ldElement );
		
		if ( ldElement )
		{	ldNode = new LoopDetectionNode( ldElement );
			ASSERT( ldNode );
			if ( !ldNode )
				pickErr = kPickerPopulateNoMem;
		}
		else
			pickErr = kPickerPopulateNoMem;
	}
	
	
	if ( !pickErr )
	{
		weightings = ::fopen( path, "r" );

		if (!weightings) 
		{
			//printf("- Playlist picker failed opening list file %s\n", path);
			pickErr = kPickerPopulateFileError;
		}
	}
	
	
	
	if ( !pickErr )
	{
		long	lineBuffSize = kMaxPickerPath *2;
		long	wordBuffSize = kMaxPickerPath;
		
		
		char 	lineBuff[kMaxPickerPath * 2];
		char	wordBuff[kMaxPickerPath];
		char*	next;
		char*	pathEnd;
		char*	thisLine;
		
		// add ourselves to the list
		ldList->AddElement( ldNode );
		
		// trim off the file name to get just the path part
		pathEnd = ::strrchr( path, kPathDelimiterChar );
		
		if ( pathEnd )
		{	
			pathEnd++;
		
			*pathEnd = 0;
		}
		else
			*path = 0;
		
		

		thisLine = lineBuff;
		
		if ( ::fgets( lineBuff, lineBuffSize, weightings ) != NULL )
		{
			lineCount++;
			
			thisLine = ::TrimLeft( lineBuff );
			
			if ( 0 != ::strncmp(thisLine,"*PLAY-LIST*",11) )
			{	
				//::printf("- Playlist file missing *PLAY-LIST* identifier as first line:\n");
				//::printf("  %s%s\n", basePath, fname);
				pickErr = kPickerPopulateBadFormat;
			}
		}
		
			
		if ( !pickErr )
		{
			do 
			{	
				next = lineBuff;
				
				if ( ::fgets( lineBuff, lineBuffSize, weightings ) == NULL )
					break;
				
				lineCount++;
				
				next = ::TrimLeft( lineBuff );
				
				if ( *next == '#' )
				{
					// it's a comment - just toss
					
					//if ( *next )
					//	printf( "comment: %s" , &lineBuff[1] );
					
				}
				else if (*next == '+') // a list
				{
					next = ::TrimLeft( next+1 );	// skip past + include
					
					if ( *next == '"' )	// get the name from the next part of the buff
						next = ::GetQuotedWord( wordBuff, next, wordBuffSize );
					else
						next = ::GetWord( wordBuff, next, wordBuffSize );
						
					
					
					// recusively populate from the include file.
					pickErr = PopulatePickerFromFile( picker, wordBuff, path, ldList );
					
					if ( pickErr )
					{	
						DisplayPickerErr( pickErr, "Playlist Include failed",  fname, lineCount, lineBuff );
						pickErr = kPickerPopulateNoErr;
					}
				}
				else if ( *next )
				{
					char	numBuff[32];
					char	expandedFileName[kMaxPickerPath];
					int 	weight = 10;	// default weight is 10

					// get the movie file name
					if ( *next == '"' )
						next = ::GetQuotedWord( wordBuff, next, wordBuffSize );
					else
						next = ::GetWord( wordBuff, next, wordBuffSize );
				
					if (*wordBuff)
					{
						#if kPartialPathBeginsWithDelimiter
						if ( *wordBuff == kPathDelimiterChar )
						{
							if ( *path )
								wordBuff++;
						#else
						if ( *wordBuff != kPathDelimiterChar )
						{
						#endif
							// it's a partial path..
							
							// cat the path and fname to form the 
							// full or partial path to the movie
							::strcpy( expandedFileName, path );
							::strcat( expandedFileName, wordBuff );
						}
						else
						{	// it's an absolute path..
							::strcpy( expandedFileName, wordBuff );
						}
						
						// then get the weighting ( if supplied )
						next = ::GetWord( numBuff, next, 32 );

						if ( *numBuff )
							weight = ::atoi(numBuff);

						if ( !picker->AddToList( expandedFileName, weight ) )
							pickErr = kPickerPopulateNoMem;
					}
				}
				
			} while ( feof( weightings ) == 0 && pickErr == kPickerPopulateNoErr );
		}
		
		// remove ourselves from the list
		ldList->RemoveElement( ldNode );
		
	}
	
	
	// only report unreported errors.
	if ( ldList && ldList->GetNumElements() == 0 && pickErr )
		DisplayPickerErr( pickErr, "Playlist error", fname, lineCount, NULL );
		
	
	if ( ldNode )
		delete ldNode; // node deletes element
	else if ( ldElement )
		delete ldElement;
	

	if ( weightings )
		(void)::fclose( weightings );

	if ( ldList && ldList->GetNumElements() == 0 )
	{
		// all done now!
		delete ldList;
		ldList = NULL;
	
	}
	
	
	return pickErr;
	
	
}