
#ifndef __pldoublelinkedlist__
#define __pldoublelinkedlist__

/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */


/*

	8.3.99 rt added __PLDoubleLinkedListDEBUG__ compile switch to turn off ASSERTs and Debugging validation
*/

#include <stdlib.h>
#include <string.h>

#include "OSHeaders.h"
#include "plassert.h"

#ifndef __PLDoubleLinkedListDEBUG__
#define __PLDoubleLinkedListDEBUG__ 0
#endif

template <class T> class PLDoubleLinkedList;

template <class S> class PLDoubleLinkedListNode
{
	friend class PLDoubleLinkedList <class S>;

	public:
		PLDoubleLinkedListNode( class S* element )
		{
				// the node takes ownership of "element"
				mElement = element;
				mNext = NULL;
				mPrev = NULL;
		}
		virtual ~PLDoubleLinkedListNode()
				{
					#if __PLDoubleLinkedListDEBUG__
					ASSERT( mPrev == NULL && mNext == NULL );
					#endif
					
					delete  mElement;
				}
		
		class S* mElement;

	protected:
		PLDoubleLinkedListNode *mNext;
		PLDoubleLinkedListNode *mPrev;
};



template <class T> class PLDoubleLinkedList
{
	
	public:
		PLDoubleLinkedList()
		{
			mHead = NULL;
			mTail = NULL;
			mNumElements = 0;
		}
		
		virtual ~PLDoubleLinkedList()
				{
					ClearList();
				}
		
		#if __PLDoubleLinkedListDEBUG__
		
		void					ValidateLinks()
								{
									PLDoubleLinkedListNode<T> *nextNode;
									
									ASSERT( mHead == NULL || mHead->mPrev == NULL );
									ASSERT( mTail == NULL || mTail->mNext == NULL );
									
									
									if ( mTail == mHead && mTail != NULL )
									{
										ASSERT(  mTail->mPrev == NULL && mTail->mNext == NULL );
									}
									
									if ( mHead  )
									{
										ASSERT(  mTail != NULL  );
									}

									if ( mTail  )
									{
										ASSERT(  mHead != NULL  );
									}

									
									if ( mTail && mTail->mPrev )
										ASSERT( mTail->mPrev->mNext == mTail  );

									if ( mHead && mHead->mNext )
										ASSERT( mHead->mNext->mPrev == mHead  );
									
									
										
									nextNode = mHead;
									
									while ( nextNode )
									{	
										ASSERT( mHead == nextNode || nextNode->mPrev->mNext == nextNode );
										ASSERT( mTail == nextNode || nextNode->mNext->mPrev == nextNode );
										
										if ( !nextNode->mNext )
											ASSERT( mTail == nextNode );
											
										nextNode = nextNode->mNext;
									}

									nextNode = mTail;
									
									while ( nextNode )
									{	
										ASSERT( mHead == nextNode || nextNode->mPrev->mNext == nextNode );
										ASSERT( mTail == nextNode || nextNode->mNext->mPrev == nextNode );
										
										if ( !nextNode->mPrev )
											ASSERT( mHead == nextNode );
											
										nextNode = nextNode->mPrev;
									}
								}
		#endif // __PLDoubleLinkedListDEBUG__
		
		PLDoubleLinkedListNode<T>	* GetFirst() { return mHead; };
		
		void 					AddElementToTail(PLDoubleLinkedListNode<T> *element)
								{
									
									
									#if __PLDoubleLinkedListDEBUG__
									// must not be associated with another list
									ASSERT( element->mPrev == NULL && element->mNext == NULL );
									#endif
									
									
									if ( mTail )
										mTail->mNext = element;

									
									element->mPrev = mTail;
									element->mNext = NULL;
									
									mTail = element;
										
									if ( !mHead )
										mHead = element;

									mNumElements++;
									
									#if __PLDoubleLinkedListDEBUG__
									ValidateLinks();
									#endif
								}		
		
		void					AddElement(PLDoubleLinkedListNode<T> *element )
								{
									#if __PLDoubleLinkedListDEBUG__
									// must not be associated with another list
									ASSERT( element->mPrev == NULL && element->mNext == NULL );
									#endif
									
									if ( mHead )
										mHead->mPrev = element;

									
									element->mPrev = NULL;
									element->mNext = mHead;
									
									mHead = element;
										
									if ( !mTail )
										mTail = element;

									mNumElements++;
									
									#if __PLDoubleLinkedListDEBUG__
									ValidateLinks();
									#endif
									
								}

		void 					RemoveElement(PLDoubleLinkedListNode<T> *element)
								{
									
									#if __PLDoubleLinkedListDEBUG__
									// must be associated with this list	
									ASSERT( mHead == element || element->mPrev->mNext == element );
									
									// must be associated with this list
									ASSERT( mTail == element || element->mNext->mPrev == element );
									#endif
									
									
									if ( mHead == element)
										mHead = element->mNext;
									else
										element->mPrev->mNext = element->mNext;

									if ( mTail == element)
										mTail = element->mPrev;
									else
										element->mNext->mPrev = element->mPrev;

									
									element->mPrev = NULL;
									element->mNext = NULL;

									
									
									mNumElements--;
									
									#if __PLDoubleLinkedListDEBUG__
									ValidateLinks();
									#endif
									
								}
								
		PLDoubleLinkedListNode<T>	*ForEachUntil( bool (*doFunc)( PLDoubleLinkedListNode<T> *element,  void *userData), void *userData )
									{
										PLDoubleLinkedListNode<T> *nextElement, *curElement;
										bool					stopIteration = false;
										
										curElement = mHead;
										
										while ( curElement && !stopIteration )
										{
											nextElement = curElement->mNext;
										
											stopIteration = (*doFunc)( curElement, userData);

											if ( !stopIteration )
												curElement = nextElement;
										}

										return curElement;
									}

		void					ForEach( void (*doFunc)( PLDoubleLinkedListNode<T> *element,  void *userData), void *userData )
								{
									PLDoubleLinkedListNode<T> *nextElement, *curElement;
									
									curElement = mHead;
									
									while ( curElement )
									{
										nextElement = curElement->mNext;	
									
										(*doFunc)( curElement, userData);
										
										curElement = nextElement;
									}

								}


		void					ClearList()
								{
									ForEach( DoClearList, this );
								}


		PLDoubleLinkedListNode<T>	*GetNthNode( int nodeIndex )

									{
										return ForEachUntil( CompareIndexToZero, &nodeIndex );
									}
		UInt32					GetNumElements() { return mNumElements; }
		
	protected:
		static bool		CompareIndexToZero( PLDoubleLinkedListNode<T> */*node*/, void * nodeIndex )
						{
							int 	val = *(int*)nodeIndex;
							
							if ( val == 0  )
								return true;
							
							*(int*)nodeIndex = val -1;
							
							return false;
						}		
						
		static void		DoClearList( PLDoubleLinkedListNode<T> *element, void * listPtr )
						{
							PLDoubleLinkedList<T> *list = (PLDoubleLinkedList<T> *)listPtr;
							
							list->RemoveElement( element );
							
							delete element;
							
						}
		PLDoubleLinkedListNode<T> *mHead;
		PLDoubleLinkedListNode<T> *mTail;
		UInt32					mNumElements;
		
	

};

#endif




