#include "config.h"
#include <string.h>
#include "songlyrics.moc"


SongLyrics::SongLyrics(KConfig *conf)
  : QSplitter(0, "Song Lyrics") {

  setCaption(i18n("Song Lyrics"));

  QString fileName("xmms-kde/lyrics");
  QString pathName(locateLocal("data", fileName));
  QDir dir(pathName);
  dir.mkdir(pathName);

  config = conf;
  entry = 0;

  QVBox *vbox = new QVBox(this);

  box = new QListBox(vbox);
  connect(box, SIGNAL(highlighted(int)),
	  this, SLOT(selected(int)));

  bar = new QProgressBar(vbox);
  bar->setTotalSteps(100);
  bar->setPercentageVisible(false);

  prog = 0;

  timer = new QTimer(this);
  connect(timer, SIGNAL(timeout()),
	  this, SLOT(progress()));

  vbox = new QVBox(this);

  edit = new QTextEdit(vbox);
  edit->setReadOnly(true);
  edit->setTextFormat(Qt::RichText);

  search = new QLineEdit(vbox);
  connect(search, SIGNAL(textChanged(const QString &)),
	  this, SLOT(markup(const QString &)));


  qInitNetworkProtocols();

  config->setGroup("LYRICS");

  framePos = QPoint(0, 0);
  framePos = config->readPointEntry("frameposition", &framePos);
  frameSize = QSize(200, 320);
  frameSize = config->readSizeEntry("framesize", &frameSize);

  resize(frameSize);
  move(framePos);

  currentText = NULL;
}

SongLyrics::~SongLyrics() {

}


QString SongLyrics::getPathName(QString artist, QString title) {

  artist = artist.replace(QRegExp("[\\s\\(\\)/]"), "_");

  QString fileName = QString("xmms-kde/lyrics/");
  QString pathName(locateLocal("data", fileName));
  pathName.append(artist);
  
  return pathName;
}

QString SongLyrics::getFileName(QString artist, QString title) {

  artist = artist.replace(QRegExp("[\\s\\(\\)/]"), "_");
  title = title.replace(QRegExp("[\\s\\(\\)/]"), "_");

  QString fileName = QString("xmms-kde/lyrics/");
  QString pathName(locateLocal("data", fileName));
  pathName.append(artist + "/" + title);

  return pathName;
}

void SongLyrics::writeConfig() {

  config->setGroup("LYRICS");
  config->writeEntry("frameposition", pos());
  config->writeEntry("framesize", size());
}

int SongLyrics::request(QString artist, QString title) {

  /*
   * delete requests
   */
  Request r(0);

  while (requests.find(&r) != -1) {
    Request *req = requests.current();
    requests.remove();
    delete req->parser;
    delete req->http;
    delete req;
  }

  qDebug("request: [%s] [%s]", artist.latin1(), title.latin1());

  // search file
  QString fileName = getFileName(artist, title);
  qDebug("looking up: [%s]", fileName.latin1());

  if (QFile::exists(fileName)) {

    if (entry)
      delete entry;

    entry = new Entry(artist, title);
    box->clear();
    box->insertItem(artist + ": " + title);
    box->setSelected(0, true);

  } else {
 
    qDebug("getting lyrics from web: [%s] [%s]", artist.latin1(), title.latin1());
    requestWeb(artist, title);
  }
}



void SongLyrics::requestWeb(QString artist, QString title) {

  qDebug("requestWeb: [%s] [%s]", artist.latin1(), title.latin1());

  ResultParser *resultParser = new ResultParser(artist, title);
  QUrl::encode(artist);
  QUrl::encode(title);

  QHttp *http = new QHttp();

  connect(http, SIGNAL(requestStarted(int)),
	  this, SLOT(searchRequestStarted(int)));
  connect(http, SIGNAL(requestFinished(int, bool)),
	  this, SLOT(searchRequestFinished(int, bool)));

  QString search = "auth=LeosLyrics5&artist=" + artist + 
    "&songtitle=" + title + "&search=false";
  QHttpRequestHeader header("POST", "/api_search.php");
  header.setValue("Host", "api.leoslyrics.com");
  header.setContentType("application/x-www-form-urlencoded");
  http->setHost("api.leoslyrics.com");
  int id = http->request(header, search.utf8());

  Request *r = new Request(id, (Parser *) resultParser, http);
  requests.append(r);
}

void SongLyrics::searchRequestStarted(int id) {

  prog = 0;
  timer->start(50);
}

void SongLyrics::getRequestStarted(int id) {

  prog = 0;
  timer->start(50);
}

void SongLyrics::searchRequestFinished(int id, bool error) {

  Request r(id);
  if (requests.find(&r) != -1) {

    qDebug("searchRequestFinished: %d", id);

    Request *req = requests.current();
    
    int size;

    if ((req->http->bytesAvailable()) != 0 ) {

      QByteArray data = req->http->readAll();
      req->contents.append(data);
    }

    QXmlInputSource source;
    source.setData(req->contents);
    QXmlSimpleReader reader;
    reader.setContentHandler(req->parser);
    reader.parse(source);

    if (entry)
      delete entry;

    entry = ((ResultParser *) req->parser)->getEntry();

    if (!entry) {
      box->clear();
      timer->stop();
      bar->setProgress(0);
      setText("No lyrics found.");
    } else {
      box->clear();
      box->insertItem(*(entry->dbArtist) + ": " + *(entry->dbTitle));
      box->setSelected(0, true);
    }

    req->id = 0;  // schedule for deletion
  }

  qDebug("!searchRequestFinished");
}


void SongLyrics::getRequestFinished(int id, bool error) {

  Request r(id);
  if (requests.find(&r) != -1) {

    Request *req = requests.current();

    int size;

    if ((req->http->bytesAvailable()) != 0 ) {

      QByteArray data = req->http->readAll();
      req->contents.append(data);
    }

    SongParser *parser = (SongParser *) req->parser;

    qDebug("getRequestFinished %d", id);
    QXmlInputSource source;
    source.setData(req->contents);
    QXmlSimpleReader reader;
    reader.setContentHandler(parser);
    reader.parse(source);
    
    QString text = *parser->getText();
    
    setText(text);
    
    QString artist = parser->getArtist();
    QString title = parser->getTitle();
    
    QString fileName = getFileName(artist, title);

    if (!QFile::exists(fileName)) {
      
      QString pathName = getPathName(artist, title);
      QDir dir(pathName);
      dir.mkdir(pathName);
      QFile file(fileName);
      qDebug("save: [%s]", fileName.latin1());
      file.open(IO_WriteOnly);
      file.writeBlock(text.latin1(), text.length());
      file.close();
    }
    timer->stop();
    bar->setProgress(0);

    req->id = 0;  // schedule for deletion
  } else {
    
  }

  qDebug("!getRequestFinished %d", id);
}


void SongLyrics::setText(QString text) {

  if (currentText != NULL)
    delete currentText;

  currentText = new QString(text);
  text = text.replace(QRegExp("\n"), "<br>");
  edit->setText(text);
}

int SongLyrics::selected(int index) {

  if (entry && !entry->hid) {

    QString artist(*entry->artist);
    QString title(*entry->title);
    qDebug("getting lyrics from filesystem: [%s] [%s]", artist.latin1(), title.latin1());

    QString fileName = getFileName(artist, title);

    QFile file(fileName);
    file.open(IO_ReadOnly);
    QString text("");
    char line[4096];

    int status = -2;
    while ((status != -1) && (status != 0)) {
      memset(line, 0, 4096);
      status = file.readBlock(line, 4095);
      text.append(line);
    }

    file.close();

    setText(text);
  } else {
    QString hid(*entry->hid);

    QUrl::encode(hid);
    hid = hid.replace("+", "%2B");
    
    qDebug("hid: %s", hid.latin1());

    QString artist(*entry->artist);
    QString title(*entry->title);

    QHttp *http = new QHttp();

    connect(http, SIGNAL(requestStarted(int)),
	    this, SLOT(getRequestStarted(int)));
    connect(http, SIGNAL(requestFinished(int, bool)),
	    this, SLOT(getRequestFinished(int, bool)));

    qDebug("songparser: [%s] [%s]", artist.latin1(), title.latin1());
    SongParser *songParser = new SongParser(artist, title);
    QString search = "auth=LeosLyrics5&hid=" + hid + "&file=NULL"; 
    QHttpRequestHeader header("POST", "/api_lyrics.php");
    header.setValue("Host", "api.leoslyrics.com");
    header.setContentType("application/x-www-form-urlencoded");
    http->setHost("api.leoslyrics.com");
    int id = http->request(header, search.utf8());

    Request *r = new Request(id, (Parser *) songParser, http);
    requests.append(r);
  }

  return 0;
}

void SongLyrics::progress() {

  prog += 5;
  prog %= 100;

  bar->setProgress(prog);
}

void SongLyrics::markup(const QString & text) {

  QString tmp = QString(currentText->latin1());
  QString replace;
  replace.sprintf("<strong><em><big>%s</big></em></strong>", text.latin1());
  tmp = tmp.replace(QRegExp(text, false), replace).replace(QRegExp("\n"), "<br>");
  edit->setText(tmp);
}
