/************************************************************************/
/*									*/
/*  Font administration for a document.					*/
/*									*/
/************************************************************************/

#   include	"appUtilConfig.h"

#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<string.h>

#   include	<appDebugon.h>

#   include	"utilPropMask.h"
#   include	"docFont.h"
#   include	"utilFontEncoding.h"

/************************************************************************/
/*									*/
/*  Initialise a Font List.						*/
/*  Clean a Font List.							*/
/*  Copy a Font List.							*/
/*									*/
/************************************************************************/

void docInitFontList(	DocumentFontList *	dfl )
    {
    dfl->dflCount= 0;
    dfl->dflFonts= (DocumentFont *)0;
    }

void docCleanFontList(	DocumentFontList *	dfl )
    {
    int		i;

    for ( i = 0; i < dfl->dflCount; i++ )
	{ docCleanFont( dfl->dflFonts+ i );	}

    if  ( dfl->dflFonts )
	{ free( dfl->dflFonts );	}
    }

int docCopyFontList(	DocumentFontList *		to,
			const DocumentFontList *	from )
    {
    int			i;
    DocumentFontList	copy;

    docInitFontList( &copy );

    copy.dflFonts= malloc( from->dflCount* sizeof(DocumentFont) );
    if  ( ! copy.dflFonts )
	{ XDEB(copy.dflFonts); docCleanFontList( &copy ); return -1;	}

    for ( i = 0; i < from->dflCount; i++ )
	{ docInitFont( copy.dflFonts+ i );	}
    copy.dflCount= from->dflCount;

    for ( i = 0; i < from->dflCount; i++ )
	{
	if  ( docCopyFont( copy.dflFonts+ i, from->dflFonts+ i ) )
	    { LDEB(i); docCleanFontList( &copy ); return -1;	}
	}

    docCleanFontList( to );

    *to= copy; return 0;
    }

/************************************************************************/
/*									*/
/*  Initialise a font.							*/
/*  Clean a font.							*/
/*  Copy a font.							*/
/*									*/
/************************************************************************/

void docInitFont(	DocumentFont *	df	)
    {
    df->dfFamilyStyle= (char *)0;
    df->dfName= (char *)0;
    df->dfDocFamilyNumber= -1;
    df->dfPsFamilyNumber= -1;
    df->dfInstanceCount= 0;
    df->dfInstances= (DocumentFontInstance *)0;
    df->dfCharset= FONTcharsetDEFAULT;
    df->dfEncodingUsed= -1;
    df->dfPitch= FONTpitchDEFAULT;

    df->dfPanose[0]= '\0';
    }

void docCleanFont(	DocumentFont *	df	)
    {
    if  ( df->dfFamilyStyle )
	{ free( df->dfFamilyStyle );	}
    if  ( df->dfName )
	{ free( df->dfName );	}
    if  ( df->dfInstances )
	{ free( df->dfInstances );	}
    }

int docCopyFont(	DocumentFont *		to,
			const DocumentFont *	from )
    {
    DocumentFont	copy;

    copy= *from;

    copy.dfFamilyStyle= (char *)0;
    copy.dfName= (char *)0;
    copy.dfInstances= (DocumentFontInstance *)0;

    if  ( from->dfFamilyStyle )
	{
	copy.dfFamilyStyle= strdup( from->dfFamilyStyle );
	if  ( ! copy.dfFamilyStyle )
	    { XDEB(copy.dfFamilyStyle); docCleanFont( &copy ); return -1; }
	}

    if  ( from->dfName )
	{
	copy.dfName= strdup( from->dfName );
	if  ( ! copy.dfName )
	    { XDEB(copy.dfName); docCleanFont( &copy ); return -1; }
	}

    if  ( from->dfInstanceCount > 0 )
	{
	int	i;

	copy.dfInstances= (DocumentFontInstance *)
	    malloc( from->dfInstanceCount* sizeof( DocumentFontInstance ) );

	for ( i= 0; i < from->dfInstanceCount; i++ )
	    { copy.dfInstances[i]= from->dfInstances[i];	}
	}

    if  ( to->dfFamilyStyle )
	{ free( to->dfFamilyStyle );	}
    if  ( to->dfName )
	{ free( to->dfName );	}
    if  ( to->dfInstances )
	{ free( to->dfInstances );	}

    *to= copy; return 0;
    }

/************************************************************************/
/*									*/
/*  Insert a font.							*/
/*									*/
/************************************************************************/

DocumentFont *	docInsertFont(	DocumentFontList *	dfl,
				int			n,
				const char *		familyStyle,
				const char *		fontName )
    {
    DocumentFont *	df;
    char *		s;

    char *		familyStyle_s;
    char *		fontName_s;

    if  ( n > dfl->dflCount )
	{
	df= (DocumentFont *)realloc( dfl->dflFonts,
		    ( n + 1 ) * sizeof( DocumentFont ) );
	}
    else{
	df= (DocumentFont *)realloc( dfl->dflFonts,
		    ( dfl->dflCount + 1 ) * sizeof( DocumentFont ) );
	}
    if  ( ! df )
	{ LLDEB(dfl->dflCount,df); return df; }
    dfl->dflFonts= df;

    if  ( familyStyle && familyStyle[0] )
	{
	s= (char *)malloc( strlen( familyStyle )+ 1 );
	if  ( ! s )
	    { LDEB(s); return (DocumentFont *)0;	}
	strcpy( s, familyStyle );
	familyStyle_s= s;
	}
    else{ familyStyle_s= (char *)0;	}

    if  ( fontName && fontName[0] )
	{
	s= (char *)malloc( strlen( fontName )+ 1 );
	if  ( ! s )
	    {
	    LDEB(s);
	    if  ( familyStyle_s )
		{ free( familyStyle_s );	}
	    return (DocumentFont *)0;
	    }
	strcpy( s, fontName );
	fontName_s= s;
	}
    else{ fontName_s= (char *)0;	}

    if  ( n == -1 )
	{
	for ( n= 0; n < dfl->dflCount; n++ )
	    {
	    if  (  df[n].dfDocFamilyNumber < 0 )
		{ break;	}
	    }
	}
    else{
	if  ( n < dfl->dflCount && df[n].dfDocFamilyNumber >= 0 )
	    { LLDEB(n,df[n].dfDocFamilyNumber);	}
	/* No!!
	for ( i= dfl->dflCount; i > n; i-- )
	    { df[i]= df[i-1];	}
	*/
	}

    while( dfl->dflCount < n )
	{ docInitFont( df+ dfl->dflCount ); dfl->dflCount++;	}

    df += n;
    docInitFont( df );

    df->dfDocFamilyNumber= n;
    df->dfFamilyStyle= familyStyle_s;
    df->dfName= fontName_s;

    if  ( n == dfl->dflCount )
	{ dfl->dflCount++;	}

    return df;
    }

/************************************************************************/
/*									*/
/*  Find or insert a font in a document font list.			*/
/*									*/
/*  In office documents, the same font in PostScript terms can occur	*/
/*  several times in the font list with a different character set and	*/
/*  a suffix to the family name to differentiate between the		*/
/*  occurences.								*/
/*									*/
/*  NOTE that the name of a font is the key field in this		*/
/*	administration. Character set and family style are dependent	*/
/*	fields.								*/
/*									*/
/************************************************************************/

int docGetFontByName(	DocumentFontList *	dfl,
			const char *		fontFamilyName,
			int			encoding,
			int			encodingIsDefault )
    {
    int			i;
    DocumentFont *	df;

    char *		allocated= (char *)0;
    int			charset= FONTcharsetDEFAULT;

    if  ( encoding >= 0 )
	{
	const FontCharset *	fc= PS_Encodings+ encoding;

	if  ( PS_Encodings[encoding].fcOfficeFontnameSuffix )
	    {
	    allocated= malloc( strlen( fontFamilyName )+ 
				strlen( fc->fcOfficeFontnameSuffix )+ 1 );
	    
	    if  ( ! allocated )
		{ XDEB(allocated); return -1;	}

	    strcpy( allocated, fontFamilyName );
	    strcat( allocated, fc->fcOfficeFontnameSuffix );

	    fontFamilyName= allocated;
	    }

	if  ( ! encodingIsDefault )
	    { charset= fc->fcOfficeCharset;	}
	}

    df= dfl->dflFonts;
    for ( i= 0; i < dfl->dflCount; df++, i++ )
	{
	if  ( df->dfDocFamilyNumber < 0	||
	      ! df->dfName		)
	    { continue;	}

	if  ( ! strcmp( df->dfName, fontFamilyName ) )
	    { break;	}
	}

    if  ( i >= dfl->dflCount )
	{
	const char *	fontFamilyStyle;

	fontFamilyStyle= utilFontFamilyStyle( fontFamilyName );

	df= docInsertFont( dfl, -1, fontFamilyStyle, fontFamilyName );
	if  ( ! df )
	    { LDEB(dfl->dflCount); return -1;	}

	df->dfCharset= charset;
	}

    if  ( allocated )
	{ free( allocated );	}

    return df->dfDocFamilyNumber;
    }
