/************************************************************************/
/*									*/
/*  Buffer administration routines. Functionality related to the item	*/
/*  tree.								*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>

#   include	<appDebugon.h>

#   include	"docBuf.h"

#   define	CHECK_ITEMS	0

/************************************************************************/
/*									*/
/*  Free a BufferItem.							*/
/*									*/
/************************************************************************/

void docCleanItem(	BufferDocument *	bd,
			BufferItem *		bi )
    {
    int			i;
    TextParticule *	tp;

    for ( i= bi->biChildCount- 1; i >= 0; i-- )
	{ docFreeItem( bd, bi->biChildren[i] ); }
    if  ( bi->biChildren )
	{ free( bi->biChildren );	}

    switch( bi->biLevel )
	{
	case DOClevDOC:
	    break;

	case DOClevSECT:
	    docCleanExternalItem( bd, &(bi->biSectHeader) );
	    docCleanExternalItem( bd, &(bi->biSectFirstPageHeader) );
	    docCleanExternalItem( bd, &(bi->biSectLeftPageHeader) );
	    docCleanExternalItem( bd, &(bi->biSectRightPageHeader) );

	    docCleanExternalItem( bd, &(bi->biSectFooter) );
	    docCleanExternalItem( bd, &(bi->biSectFirstPageFooter) );
	    docCleanExternalItem( bd, &(bi->biSectLeftPageFooter) );
	    docCleanExternalItem( bd, &(bi->biSectRightPageFooter) );

	    docCleanSectionProperties( &(bi->biSectProperties) );

	    break;

	case DOClevCELL:
	    break;

	case DOClevROW:
	    docCleanRowProperties( &(bi->biRowProperties) );
	    break;

	case DOClevPARA:
	    tp= bi->biParaParticules;
	    for ( i= 0; i < bi->biParaParticuleCount; tp++, i++ )
		{
		if  ( tp->tpKind == DOCkindFIELDSTART	||
		      tp->tpKind == DOCkindFIELDEND	)
		    {
		    if  ( tp->tpObjectNumber < 0			||
			  tp->tpObjectNumber >=
					bd->bdFieldList.dflFieldCount	)
			{
			LDEB(bi->biParaParticules[i].tpObjectNumber);
			LDEB(bd->bdFieldList.dflFieldCount);
			continue;
			}

		    docCleanField( bd->bdFieldList.dflFields+
							tp->tpObjectNumber );
		    docInitField( bd->bdFieldList.dflFields+
							tp->tpObjectNumber );
		    }

		if  ( tp->tpKind == DOCkindNOTE			&&
		      docDeleteNoteOfParticule( bd, bi, tp )	)
		    { LDEB(tp->tpStroff);	}
		}

	    for ( i= 0; i < bi->biParaObjectCount; i++ )
		{ docCleanObject( bi->biParaObjects+ i ); }

	    if  ( bi->biParaString )
		{ free( bi->biParaString );	}
	    if  ( bi->biParaParticules )
		{ free( bi->biParaParticules );	}
	    if  ( bi->biParaObjects )
		{ free( bi->biParaObjects );	}
	    if  ( bi->biParaLines )
		{ free( bi->biParaLines );	}
	    if  ( bi->biParaShapes )
		{ free( bi->biParaShapes );	}

	    docCleanParagraphProperties( &(bi->biParaProperties) );

	    break;

	default:
	    /*FALLTHROUGH*/
	case DOClevOUT:
	    LDEB(bi->biLevel);
	    break;
	}

    bi->biLevel= DOClevOUT;
    }

void docFreeItem(	BufferDocument *	bd,
			BufferItem *		bi )
    {
    docCleanItem( bd, bi );
    free( bi );
    }

/************************************************************************/
/*									*/
/*  Initialise a BufferItem.						*/
/*									*/
/************************************************************************/

void docInitItem(	BufferItem *		bi,
			BufferItem *		parent,
			const BufferDocument *	bd,
			int			numberInParent,
			int			level,
			int			inExternalItem )
    {
    bi->biChildren= (BufferItem **)0;
    bi->biChildCount= 0;
    bi->biLeftParagraphs= 0;

    switch( level )
	{
	case DOClevDOC:
	    break;

	case DOClevSECT:
	    docInitExternalItem( &(bi->biSectHeader) );
	    docInitExternalItem( &(bi->biSectFirstPageHeader) );
	    docInitExternalItem( &(bi->biSectLeftPageHeader) );
	    docInitExternalItem( &(bi->biSectRightPageHeader) );

	    docInitExternalItem( &(bi->biSectFooter) );
	    docInitExternalItem( &(bi->biSectFirstPageFooter) );
	    docInitExternalItem( &(bi->biSectLeftPageFooter) );
	    docInitExternalItem( &(bi->biSectRightPageFooter) );

	    docInitSectionProperties( &(bi->biSectProperties) );

	    if  ( bd )
		{ bi->biSectDocumentGeometry= bd->bdProperties.dpGeometry; }

	    docInitSelectionScope( &(bi->biSectSelectionScope) );

	    bi->biSectSelectionScope.ssInExternalItem= inExternalItem;

	    bi->biSectHeaderFooterUseForSectBi= (const BufferItem *)0;
	    bi->biSectHeaderFooterUseForPage= -1;

	    break;

	case DOClevCELL:
	    break;

	case DOClevROW:
	    docInitRowProperties( &(bi->biRowProperties) );
	    break;

	case DOClevPARA:
	    bi->biParaStrlen= 0;
	    bi->biParaString= (unsigned char *)0;

	    bi->biParaParticuleCount= 0;
	    bi->biParaParticules= (TextParticule *)0;

	    bi->biParaLineCount= 0;
	    bi->biParaLines= (TextLine *)0;

	    bi->biParaObjectCount= 0;
	    bi->biParaObjects= (InsertedObject *)0;

	    bi->biParaShapeCount= 0;
	    bi->biParaShapes= (DrawingShape *)0;

	    bi->biParaAscentTwips= 0;
	    bi->biParaDescentTwips= 0;
	    bi->biParaLeadingTwips= 0;

	    bi->biParaSpaceAboveLinesTwips= 0;
	    bi->biParaBorderAboveParagraph= (const BorderProperties *)0;
	    bi->biParaSpaceBelowLinesTwips= 0;
	    bi->biParaBorderBelowParagraph= (const BorderProperties *)0;

	    docInitParagraphProperties( &(bi->biParaProperties) );

	    break;
	default:
	    bi->biLevel= DOClevOUT;
	    bi->biParent= (BufferItem *)0;
	    LDEB(level); return;
	}

    bi->biLevel= level;
    bi->biInExternalItem= inExternalItem;
    bi->biParent= parent;
    bi->biNumberInParent= numberInParent;

    docInitLayoutPosition( &(bi->biTopPosition) );
    docInitLayoutPosition( &(bi->biBelowPosition) );

    return;
    }

/************************************************************************/
/*									*/
/*  1)  Delete a series of items.					*/
/*  2)  Delete an item from its parent.					*/
/*									*/
/************************************************************************/

static void docSetSectExternalParents(		BufferItem *	sectBi,
						int		n )
    {
    if  ( sectBi->biSectHeader.eiItem )
	{
	sectBi->biSectHeader.eiItem->
			    biSectSelectionScope.ssSectNrExternalTo= n;
	}
    if  ( sectBi->biSectFirstPageHeader.eiItem )
	{
	sectBi->biSectFirstPageHeader.eiItem->
			    biSectSelectionScope.ssSectNrExternalTo= n;
	}
    if  ( sectBi->biSectLeftPageHeader.eiItem )
	{
	sectBi->biSectLeftPageHeader.eiItem->
			    biSectSelectionScope.ssSectNrExternalTo= n;
	}
    if  ( sectBi->biSectRightPageHeader.eiItem )
	{
	sectBi->biSectRightPageHeader.eiItem->
			    biSectSelectionScope.ssSectNrExternalTo= n;
	}

    if  ( sectBi->biSectFooter.eiItem )
	{
	sectBi->biSectFooter.eiItem->
			    biSectSelectionScope.ssSectNrExternalTo= n;
	}
    if  ( sectBi->biSectFirstPageFooter.eiItem )
	{
	sectBi->biSectFirstPageFooter.eiItem->
			    biSectSelectionScope.ssSectNrExternalTo= n;
	}
    if  ( sectBi->biSectLeftPageFooter.eiItem )
	{
	sectBi->biSectLeftPageFooter.eiItem->
			    biSectSelectionScope.ssSectNrExternalTo= n;
	}
    if  ( sectBi->biSectRightPageFooter.eiItem )
	{
	sectBi->biSectLeftPageFooter.eiItem->
			    biSectSelectionScope.ssSectNrExternalTo= n;
	}

    return;
    }

/*  1  */
void docDeleteItems(	BufferDocument *	bd,
			BufferItem *		bi,
			int			first,
			int			count )
    {
    int		f;
    int		c;
    int		paragraphsDeleted= 0;

#   if CHECK_ITEMS
    if  ( docCheckItem( &(bd->bdItem) ) )
	{ LDEB(2); docListItem( 0, &(bd->bdItem) ); abort(); }
#   endif

    if  ( first > bi->biChildCount )
	{
	LLDEB(first,bi->biChildCount);
	first= bi->biChildCount;
	}

    if  ( first+ count > bi->biChildCount )
	{
	LLDEB(first+count,bi->biChildCount);
	count= bi->biChildCount- first;
	}

    if  ( count <= 0 )
	{ LDEB(count); return;	}

    bi->biChildCount -= count;

    if  ( count > 0 )
	{
	f= 0;
	if  ( first > 0 )
	    { f= bi->biChildren[first- 1]->biLeftParagraphs;	}

	c= bi->biChildren[first+ count- 1]->biLeftParagraphs;

	paragraphsDeleted= c- f;
	}

    f= first; c= count;
    while( c > 0 )
	{ docFreeItem( bd, bi->biChildren[f] ); f++; c--; }

    f= first;
    while( f < bi->biChildCount )
	{
	bi->biChildren[f]= bi->biChildren[f+ count];
	bi->biChildren[f]->biNumberInParent= f;
	bi->biChildren[f]->biLeftParagraphs -= paragraphsDeleted;

	if  ( bi->biLevel == DOClevSECT		&&
	      bi->biInExternalItem != DOCinBODY	)
	    { docSetSectExternalParents( bi, f ); }

	f++;
	}

    while( bi->biParent )
	{
	first= bi->biNumberInParent;
	bi= bi->biParent;

	for ( f= first; f < bi->biChildCount; f++ )
	    { bi->biChildren[f]->biLeftParagraphs -= paragraphsDeleted;	}
	}

    bi->biLeftParagraphs -= paragraphsDeleted;

#   if CHECK_ITEMS
    if  ( docCheckItem( &(bd->bdItem) ) )
	{ LDEB(2); docListItem( 0, &(bd->bdItem) ); abort(); }
#   endif

    return;
    }

/*  2  */
void docDeleteItem(	BufferDocument *	bd,
			BufferItem *		bi )
    {
    if  ( bi->biParent )
	{ docDeleteItems( bd, bi->biParent, bi->biNumberInParent, 1 );	}
    else{ docFreeItem( bd, bi );					}
    }

/************************************************************************/
/*									*/
/*  Add a new child to a parent.					*/
/*									*/
/************************************************************************/

BufferItem * docInsertItem(	const BufferDocument *	bd,
				BufferItem *		parent,
				int			n,
				ItemLevel		level	)
    {
    int			i;

    int			newSize;

    BufferItem **	freshChildren;
    BufferItem *	bi;

#   if CHECK_ITEMS
    if  ( docCheckItem( &(bd->bdItem) ) )
	{ LDEB(2); docListItem( 0, &(bd->bdItem) ); abort(); }
#   endif

    if  ( parent->biLevel != level- 1 )
	{
	SSDEB(docLevelStr(parent->biLevel),docLevelStr(level));
	return (BufferItem *)0;
	}

    switch( parent->biLevel )
	{
	case DOClevDOC:
	case DOClevSECT:
	case DOClevROW:
	case DOClevCELL:
	    break;

	default:
	    LDEB(parent->biLevel); return (BufferItem *)0;
	}

    newSize= parent->biChildCount;

    if  ( newSize % 10 )
	{ newSize ++;		}
    else{ newSize += 10;	}

    newSize *= sizeof(BufferItem *);

    freshChildren= (BufferItem **)realloc( parent->biChildren, newSize );
    if  ( ! freshChildren )
	{ LXDEB(newSize,freshChildren); return (BufferItem *)0; }
    parent->biChildren= freshChildren;

    bi= (BufferItem *)malloc( sizeof(BufferItem) );
    if  ( ! bi )
	{ XDEB(bi); return bi;	}

    if  ( n == -1 )
	{ n= parent->biChildCount; }

    docInitItem( bi, parent, bd, n, level, parent->biInExternalItem );

    if  ( n == 0 )
	{ bi->biTopPosition= parent->biTopPosition;			}
    else{ bi->biTopPosition= freshChildren[n-1]->biBelowPosition;	}

    for ( i= parent->biChildCount; i > n; i-- )
	{
	freshChildren[i]= freshChildren[i-1];

	freshChildren[i]->biNumberInParent= i;

	if  ( freshChildren[i]->biLevel == DOClevSECT		&&
	      freshChildren[i]->biInExternalItem != DOCinBODY	)
	    { docSetSectExternalParents( freshChildren[i], i ); }
	}

    freshChildren[n]= bi;
    parent->biChildCount++;

    if  ( level != DOClevPARA )
	{
	if  ( n > 0 )
	    { bi->biLeftParagraphs= freshChildren[n-1]->biLeftParagraphs; }

	return bi;
	}

    if  ( n > 0 )
	{ bi->biLeftParagraphs= freshChildren[n-1]->biLeftParagraphs; }

    for ( i= n; i < parent->biChildCount; i++ )
	{ parent->biChildren[i]->biLeftParagraphs++;	}

    while( parent->biParent )
	{
	n= parent->biNumberInParent;
	parent= parent->biParent;

	for ( i= n; i < parent->biChildCount; i++ )
	    { parent->biChildren[i]->biLeftParagraphs++;	}
	}

    parent->biLeftParagraphs++;

#   if CHECK_ITEMS
    if  ( docCheckItem( &(bd->bdItem) ) )
	{ LDEB(2); docListItem( 0, &(bd->bdItem) ); abort(); }
#   endif

    return bi;
    }

/************************************************************************/
/*									*/
/*  Make an empty paragraph: Needed at several locations.		*/
/*									*/
/************************************************************************/

BufferItem * docInsertEmptyParagraph(		BufferDocument *	bd,
						BufferItem *		bi,
						TextAttribute		ta )
    {
    if  ( bi->biLevel < DOClevSECT )
	{ LDEB(bi->biLevel); return (BufferItem *)0;	}

    if  ( bi->biLevel < DOClevROW )
	{
	bi= docInsertItem( bd, bi, -1, DOClevROW );
	if  ( ! bi )
	    { XDEB(bi); return (BufferItem *)0;   }
	}

    if  ( bi->biLevel < DOClevCELL )
	{
	bi= docInsertItem( bd, bi, -1, DOClevCELL );
	if  ( ! bi )
	    { XDEB(bi); return (BufferItem *)0;   }
	}

    if  ( bi->biLevel < DOClevPARA )
	{
	bi= docInsertItem( bd, bi, -1, DOClevPARA );
	if  ( ! bi )
	    { XDEB(bi); return (BufferItem *)0;   }
	}
    else{
	bi= docInsertItem( bd, bi->biParent, -1, DOClevPARA );
	if  ( ! bi )
	    { XDEB(bi); return (BufferItem *)0;   }
	}

    if  ( ! docInsertTextParticule( bi, 0, 0, 0, DOCkindTEXT, ta ) )
	{ LDEB(1); return (BufferItem *)0;	}

    return bi;
    }

/************************************************************************/
/*									*/
/*  Copy a paragraph to make a new one.					*/
/*									*/
/************************************************************************/

BufferItem * docCopyParaItem(	BufferDocument *	bdTo,
				const BufferDocument *	bdFrom,
				int *			fieldMap,
				unsigned int *		pFieldUpd,
				BufferItem *		biCellTo,
				int			n,
				BufferItem *		biParaFrom,
				int			inTable,
				const char *		refFileName )
    {
    BufferItem *	insBi;

    int			partTo= 0;
    int			partFrom= 0;
    int			particulesInserted= 0;
    int			charactersCopied= 0;

    PropertyMask	ppChgMask;
    PropertyMask	ppUpdMask;

    insBi= docInsertItem( bdTo, biCellTo, n, DOClevPARA );
    if  ( ! insBi )
	{ XDEB(insBi); return insBi;	}

    insBi->biParaInTable= inTable;

    PROPmaskCLEAR( &ppChgMask );

    PROPmaskCLEAR( &ppUpdMask );
    PROPmaskFILL( &ppUpdMask, PPprop_COUNT );
    PROPmaskUNSET( &ppUpdMask, PPpropIN_TABLE );

    if  ( docUpdParaProperties( &ppChgMask, &insBi->biParaProperties,
				&ppUpdMask, &biParaFrom->biParaProperties ) )
	{ LDEB(1); return (BufferItem *)0;	}

    if  ( docCopyParticules( bdTo, bdFrom, fieldMap, pFieldUpd,
			insBi, biParaFrom,
			partTo, partFrom, biParaFrom->biParaParticuleCount,
			&particulesInserted, &charactersCopied, refFileName ) )
	{
	LDEB(biParaFrom->biParaParticuleCount);
	docDeleteItem( bdTo, insBi ); return (BufferItem *)0;
	}

    return insBi;
    }

BufferItem * docCopyRowItem(	BufferDocument *	bdTo,
				const BufferDocument *	bdFrom,
				int *			fieldMap,
				unsigned int *		pFieldUpd,
				BufferItem *		sectBiTo,
				int			n,
				BufferItem *		rowBiFrom,
				int			inTable,
				const char *		refFileName )
    {
    BufferItem *	biRowTo;
    int			col;

    biRowTo= docInsertItem( bdTo, sectBiTo, n, DOClevROW );
    if  ( ! biRowTo )
	{ XDEB(biRowTo); return biRowTo;	}

    if  ( docCopyRowProperties( &biRowTo->biRowProperties,
						&rowBiFrom->biRowProperties ) )
	{ LDEB(1); return (BufferItem *)0;	}

    for ( col= 0; col < rowBiFrom->biChildCount; col++ )
	{
	BufferItem *	biCellFrom;
	BufferItem *	biCellTo;
	int		para;

	biCellFrom= rowBiFrom->biChildren[col];
	biCellTo= docInsertItem( bdTo, biRowTo, col, DOClevCELL );

	if  ( ! biCellTo )
	    { XDEB(biCellTo); return biCellTo;	}

	for ( para= 0; para < biCellFrom->biChildCount; para++ )
	    {
	    BufferItem *	biParaFrom= biCellFrom->biChildren[para];

	    if  ( ! docCopyParaItem( bdTo, bdFrom, fieldMap, pFieldUpd,
				    biCellTo, para, biParaFrom,
				    inTable, refFileName ) )
		{ LDEB(para); return (BufferItem *)0;	}
	    }
	}

    return biRowTo;
    }

BufferItem * docCopySectItem(	BufferDocument *	bdTo,
				const BufferDocument *	bdFrom,
				int *			fieldMap,
				unsigned int *		pFieldUpd,
				BufferItem *		parentBiTo,
				int			n,
				BufferItem *		biSectFrom,
				const SelectionScope *	ss,
				const char *		refFileName )
    {
    BufferItem *		sectBiTo;
    int				row;

    if  ( parentBiTo )
	{
	if  ( parentBiTo->biInExternalItem != ss->ssInExternalItem )
	    {
	    LDEB(parentBiTo->biInExternalItem);
	    LDEB(ss->ssInExternalItem);
	    return (BufferItem *)0;
	    }

	sectBiTo= docInsertItem( bdTo, parentBiTo, n, DOClevSECT );
	if  ( ! sectBiTo )
	    { XDEB(sectBiTo); return (BufferItem *)0;	}
	}
    else{
	n= 0;
	sectBiTo= (BufferItem *)malloc( sizeof(BufferItem) );

	if  ( ! sectBiTo )
	    { XDEB(sectBiTo); return (BufferItem *)0;	}

	docInitItem( sectBiTo, (BufferItem *)0, bdTo, n,
				DOClevSECT, ss->ssInExternalItem );
	}

    sectBiTo->biSectSelectionScope= *ss;

    if  ( docCopySectionProperties( &(sectBiTo->biSectProperties),
					&(biSectFrom->biSectProperties) ) )
	{
	LDEB(1);
	docDeleteItem( bdTo, sectBiTo );
	return (BufferItem *)0;
	}

    for ( row= 0; row < biSectFrom->biChildCount; row++ )
	{
	BufferItem *	rowBiFrom= biSectFrom->biChildren[row];
	BufferItem *	rowBiTo;

	rowBiTo= docCopyRowItem( bdTo, bdFrom, fieldMap, pFieldUpd,
					    sectBiTo, row, rowBiFrom,
					    rowBiFrom->biRowHasTableParagraphs,
					    refFileName );

	if  ( ! rowBiTo )
	    {
	    XDEB(rowBiTo);
	    docDeleteItem( bdTo, sectBiTo );
	    return (BufferItem *)0;
	    }
	}

    return sectBiTo;
    }

/************************************************************************/
/*									*/
/*  Insert a new row in a table.					*/
/*									*/
/************************************************************************/

BufferItem * docInsertRowItem(	BufferDocument *	bd,
				BufferItem *		sectBi,
				int			n,
				const RowProperties *	rp,
				TextAttribute		ta )
    {
    int				col;

    BufferItem *		rowBi= (BufferItem *)0;

    rowBi= docInsertItem( bd, sectBi, n, DOClevROW );
    if  ( ! rowBi )
	{ XDEB(rowBi); return (BufferItem *)0;	}

    for ( col= 0; col < rp->rpCellCount; col++ )
	{
	BufferItem *	paraBi;

	paraBi= docInsertEmptyParagraph( bd, rowBi, ta );
	if  ( ! paraBi )
	    {
	    XDEB(paraBi);
	    docDeleteItem( bd, rowBi );
	    return (BufferItem *)0;
	    }

	paraBi->biParaInTable= 1;
	rowBi->biRowHasTableParagraphs= 1;
	}

    if  ( docCopyRowProperties( &(rowBi->biRowProperties), rp ) )
	{
	LDEB(1);
	docDeleteItem( bd, rowBi );
	return (BufferItem *)0;
	}

    return rowBi;
    }

int docCopyRowColumnAttributes(	BufferItem *		rowBi,
				const BufferItem *	refRowBi )
    {
    int		colCount;
    int		col;

    if  ( rowBi->biChildCount < refRowBi->biChildCount )
	{ colCount= rowBi->biChildCount;	}
    else{ colCount= refRowBi->biChildCount;	}

    for ( col= 0; col < colCount; col++ )
	{
	DocumentPosition	dp;
	DocumentPosition	dpRef;

	TextParticule *		tp;
	const TextParticule *	tpRef;

	PropertyMask		ppChgMask;
	PropertyMask		ppUpdMask;

	if  ( docFirstPosition( &dp, rowBi->biChildren[col] ) )
	    { continue;	}
	if  ( docFirstPosition( &dpRef, refRowBi->biChildren[col] ) )
	    { continue;	}

	PROPmaskCLEAR( &ppChgMask );

	PROPmaskCLEAR( &ppUpdMask );
	PROPmaskFILL( &ppUpdMask, PPprop_COUNT );

	if  ( docUpdParaProperties( &ppChgMask, &(dp.dpBi->biParaProperties),
				&ppUpdMask, &(dpRef.dpBi->biParaProperties) ) )
	    { LDEB(1); return -1;	}

	tp= dp.dpBi->biParaParticules;
	tpRef= dpRef.dpBi->biParaParticules;

	tp->tpTextAttribute= tpRef->tpTextAttribute;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Split an item with childen.						*/
/*									*/
/*  1)  See whether there is a parent of the given level that can be	*/
/*	split.								*/
/*									*/
/************************************************************************/

static int docSplitGroupItemLow(	const BufferDocument *	bd,
					BufferItem **		pNewBi,
					BufferItem *		oldBi,
					int			n )
    {
    BufferItem *	newBi;
    int			i;
    int			prev;

    newBi= docInsertItem( bd, oldBi->biParent,
				oldBi->biNumberInParent, oldBi->biLevel );
    if  ( ! newBi )
	{ XDEB(newBi); return -1;	}

    switch( oldBi->biLevel )
	{
	case DOClevSECT:
	    if  ( docCopySectionProperties( &newBi->biSectProperties,
					    &oldBi->biSectProperties ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevCELL:
	    break;

	case DOClevROW:
	    if  ( oldBi->biRowHasTableParagraphs )
		{
		LLDEB(oldBi->biLevel,oldBi->biRowHasTableParagraphs);
		return -1;
		}

	    if  ( docCopyRowProperties( &newBi->biRowProperties,
						&oldBi->biRowProperties ) )
		{ LDEB(1); return -1;	}
	    break;

	default:
	    LDEB(oldBi->biLevel); return -1;
	}

    newBi->biChildren= (BufferItem **)malloc( n* sizeof(BufferItem *) );
    if  ( ! newBi->biChildren )
	{ XDEB(newBi->biChildren); return -1;	}

    for ( i= 0; i < n; i++ )
	{
	newBi->biChildren[i]= oldBi->biChildren[i];
	newBi->biChildren[i]->biParent= newBi;
	}

    prev= 0;
    if  ( newBi->biNumberInParent > 0 )
	{
	prev= newBi->biParent->biChildren[newBi->biNumberInParent-1]->
							biLeftParagraphs;
	}

    if  ( n == 0 )
	{ newBi->biLeftParagraphs= prev; }
    else{
	newBi->biLeftParagraphs= prev+ newBi->biChildren[n-1]->biLeftParagraphs;
	}

    newBi->biChildCount= n;
    oldBi->biChildCount -= n;

    prev= 0;
    for ( i= 0; i < oldBi->biChildCount; i++ )
	{
	BufferItem *	child= oldBi->biChildren[i+ n];;

	oldBi->biChildren[i]= child;
	child->biNumberInParent -= n;

	if  ( oldBi->biChildren[i]->biLevel == DOClevPARA )
	    { prev++;	}
	else{
	    if  ( child->biChildCount > 0 )
		{
		prev += child->biChildren[child->biChildCount-1]->
							    biLeftParagraphs;
		}
	    }

	oldBi->biChildren[i]->biLeftParagraphs= prev;
	}

    *pNewBi= newBi; return 0;
    }

int docSplitGroupItem(	const BufferDocument *	bd,
			BufferItem **		pNewBi,
			BufferItem **		pParentBi,
			BufferItem *		parentBi,
			int			n,
			int			level )
    {
    BufferItem *	newBi= (BufferItem *)0;
    BufferItem *	splitBi;

#   if CHECK_ITEMS
    SDEB(docLevelStr(parentBi->biLevel));
    if  ( docCheckItem( &(bd->bdItem) ) )
	{ LDEB(2); docListItem( 0, &(bd->bdItem) ); abort(); }
#   endif

    splitBi= parentBi;
    while( splitBi && splitBi->biLevel != level- 1 )
	{ splitBi= splitBi->biParent; }
    if  ( ! splitBi )
	{
	SSXDEB(docLevelStr(parentBi->biLevel),docLevelStr(level),splitBi);
	return -1;
	}

    if  ( n < 0 || n >= parentBi->biChildCount )
	{ LLDEB(n,parentBi->biChildCount); return -1;	}

    for (;;)
	{
	/*
	oldBi= parentBi->biChildren[n];
	*/

	if  ( n > 0 || parentBi->biLevel == level )
	    {
	    if  ( docSplitGroupItemLow( bd, &newBi, parentBi, n ) )
		{ LDEB(n); return -1;	}
	    }

	if  ( parentBi->biLevel == level )
	    { break;	}

	n= parentBi->biNumberInParent;
	parentBi= parentBi->biParent;
	}

#   if CHECK_ITEMS
    SDEB(docLevelStr(parentBi->biLevel));
    if  ( docCheckItem( &(bd->bdItem) ) )
	{ LDEB(2); docListItem( 0, &(bd->bdItem) ); abort(); }
#   endif

    *pNewBi= newBi;
    *pParentBi= parentBi;
    return 0;
    }

/************************************************************************/
/*									*/
/*  Return the number of a paragraph.					*/
/*									*/
/************************************************************************/

int docNumberOfParagraph(	const BufferItem *	bi )
    {
    int		n= 0;

    if  ( bi->biLevel != DOClevPARA )
	{ SDEB(docLevelStr(bi->biLevel)); return -1;	}

    n= bi->biLeftParagraphs;

    while( bi->biParent )
	{
	bi= bi->biParent;

	if  ( bi->biParent && bi->biNumberInParent > 0 )
	    {
	    n += bi->biParent->biChildren[
				bi->biNumberInParent- 1]->biLeftParagraphs;
	    }
	}

    return n;
    }

BufferItem * docGetParagraphByNumber(	BufferItem *	bi,
					int		n )
    {
    if  ( bi->biParent )
	{ XDEB(bi->biParent); return (BufferItem *)0;	}

    while( bi->biChildCount > 0 )
	{
	int		i;

	for ( i= 0; i < bi->biChildCount; i++ )
	    {
	    if  ( bi->biChildren[i]->biLeftParagraphs >= n )
		{ break;	}
	    }

	if  ( i >= bi->biChildCount )
	    {
	    LLSDEB(n,bi->biChildCount,docLevelStr(bi->biLevel));
	    return (BufferItem *)0;
	    }

	if  ( i > 0 )
	    { n -= bi->biChildren[i-1]->biLeftParagraphs;	}

	bi= bi->biChildren[i];
	}

    if  ( bi->biLevel != DOClevPARA )
	{ SDEB(docLevelStr(bi->biLevel)); return (BufferItem *)0;	}

    if  ( n != 1 )
	{ LDEB(n); return (BufferItem *)0; }

    return bi;
    }

/************************************************************************/
/*									*/
/*  Change the kind of external item for a node and all its children.	*/
/*									*/
/************************************************************************/

void docSetExternalItemKind(	BufferItem *		bi,
				int			extItKind )
    {
    int		i;

    bi->biInExternalItem= extItKind;

    if  ( bi->biLevel == DOClevSECT )
	{ bi->biSectSelectionScope.ssInExternalItem= extItKind; }

    for ( i= 0; i < bi->biChildCount; i++ )
	{ docSetExternalItemKind( bi->biChildren[i], extItKind );	}

    return;
    }

