/************************************************************************/
/*									*/
/*  Save a BufferDocument into an RTF file.				*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<bitmap.h>

#   include	<appSystem.h>
#   include	<appWinMeta.h>
#   include	<appImage.h>

#   include	<sioGeneral.h>
#   include	<sioStdio.h>
#   include	<sioMemory.h>
#   include	<sioHex.h>
#   include	<sioBase64.h>

#   include	"docBuf.h"
#   include	"docLayout.h"

#   include	<appDebugon.h>

#   include	<charnames.h>

#   define	TWIPS_TO_SIZE(x)	(((x)+9)/18)

#   define	IDlenTAIL		200

#   define	USE_PNG			0
#   define	USE_GIF			1

/************************************************************************/
/*									*/
/*  Information used when writing HTML.					*/
/*									*/
/************************************************************************/

#   define	HTMLmaskFONT		0x01
#   define	HTMLmaskSUPER		0x02
#   define	HTMLmaskSUB		0x04
#   define	HTMLmaskITALIC		0x08
#   define	HTMLmaskBOLD		0x10
#   define	HTMLmaskUNDERLINE	0x20

typedef struct HtmlPushedAttribute
    {
    struct HtmlPushedAttribute *	hpaPrevious;
    unsigned int			hpaChangeMask;
    TextAttribute			hpaAttributeNew;
    } HtmlPushedAttribute;

typedef struct HtmlWritingContext
    {
    int			hwcAsMimeAggregate;

    const char *	hwcFilename;
    const char *	hwcMimeBoundary;

    int			hwcBaselength;
    int			hwcRelativeOffset;
    int			hwcUseTableForFirstIndent;
    int			hwcInHyperlink;
    int			hwcInBookmark;
    int			hwcInPageref;
    int			hwcBytesInLink;
    TextAttribute	hwcDefaultAttribute;
    ParagraphProperties	hwcParagraphProperties;

    int			hwcColumn;

    int			hwcMakeTransparentGif;
    int			hwcImageCount;
    int			hwcNoteRefCount;
    int			hwcNoteDefCount;

    char *		hwcNameScratch;
    bmWriteBitmap	hwcWriteThisBitmap;

    char		hwcImageEncoding[40+1];
    char		hwcContentIdTail[IDlenTAIL+1];
    } HtmlWritingContext;

static void docInitHtmlWritingContext(	HtmlWritingContext *	hwc )
    {
    hwc->hwcAsMimeAggregate= 0;

    hwc->hwcFilename= (const char *)0;
    hwc->hwcMimeBoundary= (const char *)0;

    hwc->hwcBaselength= 0;
    hwc->hwcRelativeOffset= 0;
    hwc->hwcUseTableForFirstIndent= 0;
    hwc->hwcInHyperlink= 0;
    hwc->hwcInBookmark= 0;
    hwc->hwcInPageref= 0;
    hwc->hwcBytesInLink= 0;

    docInitTextAttribute( &(hwc->hwcDefaultAttribute) );
    hwc->hwcDefaultAttribute.taFontSizeHalfPoints= 24;
    hwc->hwcDefaultAttribute.taFontNumber= -1;

    docInitParagraphProperties( &(hwc->hwcParagraphProperties) );

    hwc->hwcColumn= 0;

    hwc->hwcMakeTransparentGif= 0;
    hwc->hwcImageCount= 0;
    hwc->hwcNoteRefCount= 0;
    hwc->hwcNoteDefCount= 0;

    hwc->hwcNameScratch= (char *)0;
    hwc->hwcWriteThisBitmap= (bmWriteBitmap)0;

    hwc->hwcImageEncoding[0]= '\0';
    hwc->hwcContentIdTail[0]= '\0';

    return;
    }

static void docCleanHtmlWritingContext(	HtmlWritingContext *	hwc )
    {
    docCleanParagraphProperties( &(hwc->hwcParagraphProperties) );

    if  ( hwc->hwcNameScratch )
	{ free( hwc->hwcNameScratch );	}

    return;
    }

/************************************************************************/
/*									*/
/*  Make the bitmap for an image.					*/
/*  Make a name for an image.						*/
/*									*/
/************************************************************************/

static int	docHtmlMakeNameForIndent(	HtmlWritingContext *	hwc )
    {
    char *		fresh;

    fresh= (char *)realloc( hwc->hwcNameScratch, hwc->hwcBaselength+ 50 );
    if  ( ! fresh )
	{ XDEB(fresh); return -1;  }
    hwc->hwcNameScratch= fresh;

#   if  USE_PNG
    if  ( hwc->hwcAsMimeAggregate || hwc->hwcBaselength == 0 )
	{ sprintf( hwc->hwcNameScratch, "transp.png" );	}
    else{
	sprintf( hwc->hwcNameScratch, "%.*s.img/transp.png",
				    hwc->hwcBaselength, hwc->hwcFilename );
	}

    strcpy( hwc->hwcImageEncoding, "image/png" );
    hwc->hwcWriteThisBitmap= bmPngWritePng;

    return 0;
#   endif

#   if  USE_GIF
    if  ( hwc->hwcAsMimeAggregate || hwc->hwcBaselength == 0 )
	{ sprintf( hwc->hwcNameScratch, "transp.gif" );	}
    else{
	sprintf( hwc->hwcNameScratch, "%.*s.img/transp.gif",
				    hwc->hwcBaselength, hwc->hwcFilename );
	}

    strcpy( hwc->hwcImageEncoding, "image/gif" );
    hwc->hwcWriteThisBitmap= bmGifWriteGif;
    return 0;
#   endif

#   if ! USE_PNG && ! USE_GIF
    LDEB(1); return -1;
#   endif
    }

static int docHtmlMakeNameForImage(	HtmlWritingContext *	hwc,
					InsertedObject *	io )
    {
    char *		fresh;
    AppBitmapImage *	abi;
    int			siz;

    abi= (AppBitmapImage *)io->ioPrivate;
    if  ( ! abi )
	{ return 1;	}

    if  ( bmCanWriteGifFile( &abi->abiBitmap, 89, 10.0 )	&&
	  bmCanWriteJpegFile( &abi->abiBitmap, 0, 10.0 )	)
	{ return 1;	}

    siz= hwc->hwcBaselength+ 50;
    fresh= (char *)realloc( hwc->hwcNameScratch, siz+ 1 );
    if  ( ! fresh )
	{ LXDEB(siz,fresh); return -1;  }
    hwc->hwcNameScratch= fresh;

    if  ( io->ioBliptag == 0 )
	{ io->ioBliptag= appGetTimestamp();	}

#   if  USE_PNG
    if  ( abi->abiBitmap.bdColorEncoding == BMcoRGB8PALETTE )
	{
	if  ( hwc->hwcAsMimeAggregate || hwc->hwcBaselength == 0 )
	    { sprintf( hwc->hwcNameScratch, "%08x.png", io->ioBliptag ); }
	else{
	    sprintf( hwc->hwcNameScratch, "%.*s.img/%08x.png",
			hwc->hwcBaselength, hwc->hwcFilename, io->ioBliptag );
	    }

	strcpy( hwc->hwcImageEncoding, "image/png" );
	hwc->hwcWriteThisBitmap= bmPngWritePng;
	return 0;
	}
#   endif

#   if  USE_GIF
    if  ( ! bmCanWriteGifFile( &abi->abiBitmap, 89, 10.0 ) )
	{
	if  ( hwc->hwcAsMimeAggregate || hwc->hwcBaselength == 0 )
	    { sprintf( hwc->hwcNameScratch, "%08x.gif", io->ioBliptag ); }
	else{
	    sprintf( hwc->hwcNameScratch, "%.*s.img/%08x.gif",
			hwc->hwcBaselength, hwc->hwcFilename, io->ioBliptag );
	    }

	strcpy( hwc->hwcImageEncoding, "image/gif" );
	hwc->hwcWriteThisBitmap= bmGifWriteGif;
	return 0;
	}
#   endif

    if  ( ! bmCanWriteJpegFile( &abi->abiBitmap, 89, 10.0 ) )
	{
	if  ( hwc->hwcAsMimeAggregate || hwc->hwcBaselength == 0 )
	    { sprintf( hwc->hwcNameScratch, "%08x.jpg", io->ioBliptag ); }
	else{
	    sprintf( hwc->hwcNameScratch, "%.*s.img/%08x.jpg",
			hwc->hwcBaselength, hwc->hwcFilename, io->ioBliptag );
	    }

	strcpy( hwc->hwcImageEncoding, "image/jpeg" );
	hwc->hwcWriteThisBitmap= bmJpegWriteJfif;
	return 0;
	}

    return 1;
    }

/************************************************************************/
/*									*/
/*  Save a tag with an argument.					*/
/*									*/
/************************************************************************/

static void docHtmlPutString(		const char *		s,
					HtmlWritingContext *	hwc,
					SimpleOutputStream *	sos )
    {
    int		col= hwc->hwcColumn;

    /*  mime  */
    if  ( col == 0 && s[0] == '-' )
	{ sioOutPutCharacter( ' ', sos ); col += 1; }

    sioOutPutString( s, sos ); col += strlen( s );

    hwc->hwcColumn= col; return;
    }

static void docHtmlNewLine(	HtmlWritingContext *	hwc,
				SimpleOutputStream *	sos )
    {
    if  ( hwc->hwcAsMimeAggregate )
	{ sioOutPutCharacter( '\r', sos );	}

    sioOutPutCharacter( '\n', sos );
    hwc->hwcColumn= 0;
    }

static void docHtmlWriteTagArg(	const char *		tag,
				int			arg,
				HtmlWritingContext *	hwc,
				SimpleOutputStream *	sos )
    {
    char	scratch[20];

    if  ( hwc->hwcColumn > 1				&&
	  hwc->hwcColumn+ 1+ strlen( tag )+ 1+ 3 > 76	)
	{ docHtmlNewLine( hwc, sos );		}
    else{ docHtmlPutString( " ", hwc, sos );	}

    docHtmlPutString( tag, hwc, sos );

    sprintf( scratch, "=%d", arg );
    docHtmlPutString( scratch, hwc, sos );

    return;
    }

static void docHtmlEscapeCharacters(	const unsigned char *	s,
					int			len,
					HtmlWritingContext *	hwc,
					SimpleOutputStream *	sos )
    {
    while( len > 0 )
	{
	switch( *s )
	    {
	    case '"':
		docHtmlPutString( "&quot;", hwc, sos );
		break;

	    case '&':
		docHtmlPutString( "&amp;", hwc, sos );
		break;

	    case '>':
		docHtmlPutString( "&gt;", hwc, sos );
		break;

	    case '<':
		docHtmlPutString( "&lt;", hwc, sos );
		break;

	    case '-':
		/*  mime  */
		if  ( hwc->hwcColumn == 0 )
		    { sioOutPutCharacter( ' ', sos ); hwc->hwcColumn += 1; }

		sioOutPutCharacter( *s, sos ); hwc->hwcColumn += 1;
		break;

	    default:
		sioOutPutCharacter( *s, sos ); hwc->hwcColumn += 1;
		break;
	    }

	s++; len--;
	}

    return;
    }

static void docHtmlEscapeString(	const unsigned char *	s,
					HtmlWritingContext *	hwc,
					SimpleOutputStream *	sos )
    {
    docHtmlEscapeCharacters( s, strlen( (const char *)s ), hwc, sos );

    return;
    }

static int docHtmlFontSize(	int	halfPoints )
    {
    if  ( halfPoints/2 < 6 )	{ return 1;	}
    if  ( halfPoints/2 < 9 )	{ return 2;	}

    if  ( halfPoints/2 > 24 )	{ return 7;	}
    if  ( halfPoints/2 > 19 )	{ return 6;	}
    if  ( halfPoints/2 > 15 )	{ return 5;	}
    if  ( halfPoints/2 > 11 )	{ return 4;	}

    return 3;
    }

static void docHtmlFinishGroup(		SimpleOutputStream *	sos,
					HtmlWritingContext *	hwc )
    {
    if  ( hwc->hwcUseTableForFirstIndent )
	{
	docHtmlPutString( "</TABLE>", hwc, sos );
	docHtmlNewLine( hwc, sos );

	hwc->hwcUseTableForFirstIndent= 0;
	}
    }

static void docHtmlPopAttributes(	HtmlWritingContext *	hwc,
					SimpleOutputStream *	sos,
					HtmlPushedAttribute *	hpaOld,
					HtmlPushedAttribute **	pHpaNew )
    {
    HtmlPushedAttribute *	hpaNew;

    if  ( ! hpaOld )
	{ XDEB(hpaOld); *pHpaNew= hpaOld; return;		}

    if  ( ( hpaOld->hpaChangeMask & HTMLmaskUNDERLINE ) )
	{ docHtmlPutString( "</U>", hwc, sos );	}

    if  ( ( hpaOld->hpaChangeMask & HTMLmaskBOLD ) )
	{ docHtmlPutString( "</B>", hwc, sos );	}

    if  ( ( hpaOld->hpaChangeMask & HTMLmaskITALIC ) )
	{ docHtmlPutString( "</I>", hwc, sos );	}

    if  ( ( hpaOld->hpaChangeMask & HTMLmaskSUB ) )
	{ docHtmlPutString( "</SMALL></SUB>", hwc, sos );	}

    if  ( ( hpaOld->hpaChangeMask & HTMLmaskSUPER ) )
	{ docHtmlPutString( "</SMALL></SUP>", hwc, sos );	}

    if  ( ( hpaOld->hpaChangeMask & HTMLmaskFONT ) )
	{ docHtmlPutString( "</FONT>", hwc, sos );	}

    hpaNew= hpaOld->hpaPrevious;

    free( hpaOld );

    *pHpaNew= hpaNew; return;
    }

/************************************************************************/
/*									*/
/*  Change attributes.							*/
/*									*/
/*  1)  Pop all attributes until nothing is to be turned off compared	*/
/*	to the old attributes. (And the font remains the same)		*/
/*									*/
/************************************************************************/

static int docHtmlChangeAttributes(	HtmlWritingContext *		hwc,
					SimpleOutputStream *		sos,
					const BufferDocument *		bd,
					HtmlPushedAttribute *		hpaOld,
					HtmlPushedAttribute **		pHpaNew,
					TextAttribute			taDflt,
					TextAttribute			taNew )
    {
    HtmlPushedAttribute *	hpaNew;
    TextAttribute		taOld;

    /*  1  */
    while( hpaOld )
	{
	taOld= hpaOld->hpaAttributeNew;

	if  ( ( ! taOld.taFontIsSlanted			||
		taNew.taFontIsSlanted			)	&&

	      ( ! taOld.taFontIsBold			||
		taNew.taFontIsBold			)	&&

	      ( ! taOld.taIsUnderlined			||
		taNew.taIsUnderlined			)	&&

	      ( taOld.taSuperSub == DOCfontREGULAR	||
		taNew.taSuperSub != DOCfontREGULAR )		&&

	      taOld.taFontNumber ==
				taNew.taFontNumber		&&

	      taOld.taFontSizeHalfPoints ==
				taNew.taFontSizeHalfPoints	)
	    { break;	}

	docHtmlPopAttributes( hwc, sos, hpaOld, &hpaOld );
	}

    if  ( hpaOld						&&
	  docEqualFont( &(hpaOld->hpaAttributeNew), &taNew )	)
	{ *pHpaNew= hpaOld; return 0; }

    hpaNew= (HtmlPushedAttribute *)malloc( sizeof(HtmlPushedAttribute) );
    if  ( ! hpaNew )
	{ XDEB(hpaNew); return -1;	}

    hpaNew->hpaPrevious= hpaOld;
    hpaNew->hpaChangeMask= 0;
    if  ( hpaOld )
	{ taOld= hpaOld->hpaAttributeNew;	}
    else{ taOld= taDflt;			}
    hpaNew->hpaAttributeNew= taNew;

    if  ( taNew.taFontNumber != taOld.taFontNumber			||
	  taNew.taFontSizeHalfPoints != taOld.taFontSizeHalfPoints	)
	{
	const DocumentProperties *	dp= &(bd->bdProperties);
	const DocumentFontList *	dfl= &dp->dpFontList;
	const DocumentFont *		df;
	char *				font= (char *)0;

	if  ( taNew.taFontNumber < 0			||
	      taNew.taFontNumber >= dfl->dflCount	)
	    {
	    LLDEB(taNew.taFontNumber,dfl->dflCount);
	    df= dfl->dflFonts;
	    }
	else{
	    df= dfl->dflFonts+ taNew.taFontNumber;
	    }

	docHtmlPutString( "<FONT", hwc, sos );

	if  ( taNew.taFontNumber != taOld.taFontNumber )
	    {
	    if  ( ! strcmp( df->dfFamilyStyle, "fswiss" ) )
		{ font= "Helvetica,Arial";	}
	    if  ( ! strcmp( df->dfFamilyStyle, "froman" ) )
		{ font= "Times,Times New Roman";	}
	    if  ( ! strcmp( df->dfFamilyStyle, "fmodern" ) )
		{ font= "Courier";	}
	    if  ( ! strcmp( df->dfFamilyStyle, "ftech" ) )
		{ font= "Symbol";	}

	    if  ( font )
		{
		if  ( hwc->hwcColumn+ 6+ strlen( font )+ 1 > 76 )
		    { docHtmlNewLine( hwc, sos );		}
		else{ docHtmlPutString( " ", hwc, sos );	}

		docHtmlPutString( "FACE=\"", hwc, sos );
		docHtmlPutString( font, hwc, sos );
		docHtmlPutString( "\"", hwc, sos );
		}
	    }

	if  ( taNew.taFontSizeHalfPoints != taOld.taFontSizeHalfPoints )
	    {
	    int		oldSize= docHtmlFontSize( taOld.taFontSizeHalfPoints );
	    int		newSize= docHtmlFontSize( taNew.taFontSizeHalfPoints );

	    if  ( newSize != oldSize )
		{ docHtmlWriteTagArg( "SIZE", newSize, hwc, sos ); }
	    }

	hpaNew->hpaChangeMask |= HTMLmaskFONT;
	docHtmlPutString( ">", hwc, sos );
	}

    if  ( taNew.taSuperSub == DOCfontSUPERSCRIPT	&&
	  taOld.taSuperSub == DOCfontREGULAR		)
	{
	hpaNew->hpaChangeMask |= HTMLmaskSUPER;
	docHtmlPutString( "<SUP><SMALL>", hwc, sos );
	}

    if  ( taNew.taSuperSub == DOCfontSUBSCRIPT	&&
	  taOld.taSuperSub == DOCfontREGULAR	)
	{
	hpaNew->hpaChangeMask |= HTMLmaskSUB;
	docHtmlPutString( "<SUB><SMALL>", hwc, sos );
	}

    if  ( taNew.taFontIsSlanted && ! taOld.taFontIsSlanted )
	{
	hpaNew->hpaChangeMask |= HTMLmaskITALIC;
	docHtmlPutString( "<I>", hwc, sos );
	}

    if  ( taNew.taFontIsBold && ! taOld.taFontIsBold )
	{
	hpaNew->hpaChangeMask |= HTMLmaskBOLD;
	docHtmlPutString( "<B>", hwc, sos );
	}

    if  ( taNew.taIsUnderlined && ! taOld.taIsUnderlined )
	{
	hpaNew->hpaChangeMask |= HTMLmaskUNDERLINE;
	docHtmlPutString( "<U>", hwc, sos );
	}

    *pHpaNew= hpaNew; return 0;
    }

static int docHtmlSaveImgTag(	int				w,
				int				h,
				HtmlWritingContext *		hwc,
				SimpleOutputStream *		sos )
    {
    docHtmlPutString( "<IMG", hwc, sos );

    if  ( hwc->hwcColumn > 10					&&
	  hwc->hwcColumn+ 6+ strlen( hwc->hwcNameScratch ) > 76	)
	{ docHtmlNewLine( hwc, sos );		}
    else{ docHtmlPutString( " ", hwc, sos );	}

    docHtmlPutString( "SRC=\"", hwc, sos );

    if  ( hwc->hwcAsMimeAggregate )
	{
	docHtmlPutString( "cid:", hwc, sos );
	sioOutPutString( hwc->hwcNameScratch, sos );
	sioOutPutCharacter( '.', sos );
	sioOutPutString( hwc->hwcContentIdTail, sos );
	}
    else{
	docHtmlPutString( hwc->hwcNameScratch+ hwc->hwcRelativeOffset,
								hwc, sos );
	}

    docHtmlPutString( "\"", hwc, sos );

    docHtmlWriteTagArg( "WIDTH", w, hwc, sos );
    docHtmlWriteTagArg( "HEIGHT", h, hwc, sos );

    docHtmlPutString( " ALT=\"&lt;IMG&gt;\">", hwc, sos );
    docHtmlNewLine( hwc, sos );

    return 0;
    }

static int docHtmlSavePicture(	InsertedObject *	io,
				SimpleOutputStream *	sos,
				int *			pDone,
				HtmlWritingContext *	hwc )
    {
    int				res;

    const AppBitmapImage *	abi;
    const BitmapDescription *	bd;

    int				w;
    int				h;
    int				d;

    if  ( ! io->ioPrivate )
	{
	if  ( docGetBitmapForObject( io ) )
	    { LDEB(1); return 1;	}
	}

    abi= (AppBitmapImage *)io->ioPrivate;
    if  ( ! abi )
	{ return 0;	}
    bd= &(abi->abiBitmap);

    res= docHtmlMakeNameForImage( hwc, io );
    if  ( res < 0 )
	{ LDEB(res); return -1;	}
    if  ( res > 0 )
	{ return 0;	}

    w= TWIPS_TO_SIZE( ( io->ioScaleX* io->ioTwipsWide )/100 );
    h= TWIPS_TO_SIZE( ( io->ioScaleY* io->ioTwipsHigh )/100 );

    d= ( 100* bd->bdPixelsWide- 100* w )/ bd->bdPixelsWide;
    if  ( d < 0 )
	{ d= -d;	}
    if  ( d <= 15 )
	{ w= bd->bdPixelsWide;	}

    d= ( 100* bd->bdPixelsHigh- 100* h )/ bd->bdPixelsHigh;
    if  ( d < 0 )
	{ d= -d;	}
    if  ( d <= 15 )
	{ h= bd->bdPixelsHigh;	}

    if  ( docHtmlSaveImgTag( w, h, hwc, sos ) )
	{ SDEB(hwc->hwcNameScratch); return -1;	}

    hwc->hwcImageCount++;

    *pDone= 1; return 0;
    }

static int docHtmlStartAnchor(	HtmlWritingContext *		hwc,
				SimpleOutputStream *		sos,
				const char *			fileName,
				int				fileSize,
				const char *			markName,
				int				markSize,
				const char *			refName,
				int				refSize )
    {
    int		afterSpace;
    int		needed= 0;

    docHtmlPutString( "<A", hwc, sos );
    afterSpace= 0;

    if  ( fileSize > 0 || markSize > 0 )
	{
	needed += 1+ 6+ fileSize+ markSize+ 1;
	
	if  ( markSize > 0 )
	    { needed += 1;	}
	}

    if  ( refSize > 0 )
	{ needed += 1+ 6+ refSize+ 1;	}

    if  ( hwc->hwcColumn > 5		&&
	  hwc->hwcColumn+ needed > 76	)
	{ docHtmlNewLine( hwc, sos ); afterSpace= 1;		}

    if  ( fileSize > 0 || markSize > 0 )
	{
	if  ( ! afterSpace )
	    { docHtmlPutString( " ", hwc, sos ); }

	docHtmlPutString( "HREF=\"", hwc, sos );

	if  ( fileSize > 0 )
	    {
	    while( fileSize > 0 )
		{
		sioOutPutCharacter( *fileName, sos );
		fileName++; fileSize--; hwc->hwcColumn++;
		}
	    }

	if  ( markName && markSize > 0 )
	    {
	    sioOutPutCharacter( '#', sos );

	    while( markSize > 0 )
		{
		sioOutPutCharacter( *markName, sos );
		markName++; markSize--; hwc->hwcColumn++;
		}
	    }

	docHtmlPutString( "\"", hwc, sos );
	afterSpace= 0;
	}

    if  ( refSize > 0 )
	{
	if  ( ! afterSpace )
	    { docHtmlPutString( " ", hwc, sos ); }

	docHtmlPutString( "NAME=\"", hwc, sos );

	while( refSize > 0 )
	    {
	    sioOutPutCharacter( *refName, sos );
	    refName++; refSize--; hwc->hwcColumn++;
	    }

	docHtmlPutString( "\"", hwc, sos );
	afterSpace= 0;
	}

    docHtmlPutString( ">", hwc, sos );

    return 0;
    }

static int docHtmlStartField(	SimpleOutputStream *		sos,
				const DocumentField *		df,
				HtmlWritingContext *		hwc,
				const BufferDocument *		bd,
				const BufferItem *		bi,
				HtmlPushedAttribute *		hpaOld,
				HtmlPushedAttribute **		pHpaNew )
    {
    const char *	fileName= (const char *)0;
    int			fileSize= 0;
    const char *	markName= (const char *)0;
    int			markSize= 0;
    const char *	refName= (const char *)0;
    int			refSize= 0;

    switch( df->dfKind )
	{
	case DOCfkCHFTN:
	    hwc->hwcInHyperlink++;
	    if  ( ! hwc->hwcInBookmark && hwc->hwcInHyperlink == 1 )
		{
		TextAttribute	ta= hwc->hwcDefaultAttribute;

		char		ref[25+1];
		char		def[25+1];

		if  ( bi->biInExternalItem == DOCinBODY )
		    {
		    sprintf( ref, "_NREF_%d", hwc->hwcNoteRefCount+ 1 );
		    sprintf( def, "_NDEF_%d", hwc->hwcNoteRefCount+ 1 );

		    markName= def;
		    refName=  ref;
		    }
		else{
		    sprintf( ref, "_NREF_%d", hwc->hwcNoteDefCount+ 1 );
		    sprintf( def, "_NDEF_%d", hwc->hwcNoteDefCount+ 1 );

		    markName= ref;
		    refName=  def;
		    }

		markSize= strlen( markName );
		refSize= strlen( refName );

		if  ( docHtmlChangeAttributes( hwc, sos, bd,
					    hpaOld, &hpaOld,
					    hwc->hwcDefaultAttribute,
					    hwc->hwcDefaultAttribute ) )
		    { LDEB(1); return -1;	}

		docHtmlStartAnchor( hwc, sos,
					fileName, fileSize,
					markName, markSize,
					refName, refSize );

		ta= hwc->hwcDefaultAttribute;
		ta.taSuperSub= DOCfontSUPERSCRIPT;

		if  ( docHtmlChangeAttributes( hwc, sos, bd,
					    hpaOld, pHpaNew,
					    hwc->hwcDefaultAttribute, ta ) )
		    { LDEB(1); return -1;	}

		hwc->hwcBytesInLink= 0;
		}

	    break;

	case DOCfkHYPERLINK:
	    hwc->hwcInHyperlink++;
	    if  ( ! hwc->hwcInBookmark && hwc->hwcInHyperlink == 1 )
		{
		if  ( ! docFieldGetHyperlink( df,
				&fileName, &fileSize, &markName, &markSize ) )
		    {
		    if  ( docHtmlChangeAttributes( hwc, sos, bd,
					    hpaOld, pHpaNew,
					    hwc->hwcDefaultAttribute,
					    hwc->hwcDefaultAttribute ) )
			{ LDEB(1); return -1;	}

		    docHtmlStartAnchor( hwc, sos,
					    fileName, fileSize,
					    markName, markSize,
					    refName, refSize );

		    hwc->hwcBytesInLink= 0;
		    }
		}
	    break;

	case DOCfkBOOKMARK:
	    hwc->hwcInBookmark++;
	    if  ( ! hwc->hwcInHyperlink && hwc->hwcInBookmark == 1 )
		{
		if  ( ! docFieldGetBookmark( df, &refName, &refSize ) )
		    {
		    if  ( docHtmlChangeAttributes( hwc, sos, bd,
					    hpaOld, pHpaNew,
					    hwc->hwcDefaultAttribute,
					    hwc->hwcDefaultAttribute ) )
			{ LDEB(1); return -1;	}

		    docHtmlStartAnchor( hwc, sos,
					    fileName, fileSize,
					    markName, markSize,
					    refName, refSize );
		    }
		}
	    break;

	case DOCfkPAGEREF:
	    hwc->hwcInPageref++;
	    break;

	default:
	    break;
	}

    return 0;
    }

static int docHtmlFinishField(	SimpleOutputStream *		sos,
				const DocumentField *		df,
				HtmlWritingContext *		hwc,
				const BufferDocument *		bd,
				HtmlPushedAttribute *		hpaOld,
				HtmlPushedAttribute **		pHpaNew )
    {
    switch( df->dfKind )
	{
	case DOCfkCHFTN:
	case DOCfkHYPERLINK:
	    if  ( ! hwc->hwcInBookmark && hwc->hwcInHyperlink == 1 )
		{
		if  ( docHtmlChangeAttributes( hwc, sos, bd,
					    hpaOld, pHpaNew,
					    hwc->hwcDefaultAttribute,
					    hwc->hwcDefaultAttribute ) )
		    { LDEB(1); return -1;	}

		docHtmlPutString( "</A>", hwc, sos );
		}
	    hwc->hwcInHyperlink--;
	    if  ( hwc->hwcInHyperlink < 0 )
		{ hwc->hwcInHyperlink= 0;	}
	    break;

	case DOCfkBOOKMARK:
	    if  ( ! hwc->hwcInHyperlink && hwc->hwcInBookmark == 1 )
		{
		if  ( docHtmlChangeAttributes( hwc, sos, bd,
					    hpaOld, pHpaNew,
					    hwc->hwcDefaultAttribute,
					    hwc->hwcDefaultAttribute ) )
		    { LDEB(1); return -1;	}

		docHtmlPutString( "</A>", hwc, sos );
		}
	    hwc->hwcInBookmark--;
	    if  ( hwc->hwcInBookmark < 0 )
		{ hwc->hwcInBookmark= 0;	}
	    break;

	case DOCfkPAGEREF:
	    hwc->hwcInPageref--;
	    break;

	default:
	    break;
	}
    
    return 0;
    }

static int docHtmlSaveParticules( SimpleOutputStream *		sos,
				const BufferDocument *		bd,
				const BufferItem *		bi,
				int				part,
				int				partUpto,
				HtmlWritingContext *		hwc )
    {
    int					done= 0;

    int					afterSpace= 0;

    TextParticule *			tp= bi->biParaParticules+ part;
    int					stroff= tp->tpStroff;
    unsigned char *			s= bi->biParaString+ stroff;

    int					pictureDone;
    InsertedObject *			io;

    HtmlPushedAttribute *		hpa= (HtmlPushedAttribute *)0;

    const DocumentField *		df;
    const FieldKindInformation *	fki;

    int					len;

    while( part < partUpto )
	{
	switch( tp->tpKind )
	    {
	    case DOCkindTAB:
		docHtmlPutString( " ", hwc, sos );
		afterSpace= 1;
		s++; stroff++;
		break;

	    case DOCkindTEXT:
		if  ( docHtmlChangeAttributes( hwc, sos, bd, hpa, &hpa,
			    hwc->hwcDefaultAttribute, tp->tpTextAttribute ) )
		    { LDEB(1); return -1;	}

		if  ( afterSpace && hwc->hwcColumn+ tp->tpStrlen > 76 )
		    { docHtmlNewLine( hwc, sos );	}

		len= tp->tpStroff+ tp->tpStrlen- stroff;

		if  ( ! hwc->hwcInHyperlink	||
		      ! hwc->hwcInPageref	||
		      hwc->hwcBytesInLink == 0	)
		    {
		    docHtmlEscapeCharacters( s, len, hwc, sos );

		    if  ( hwc->hwcInHyperlink )
			{ hwc->hwcBytesInLink += len;	}
		    }

		s += len;
		stroff += len;

		afterSpace= 0;
		if  ( tp->tpStrlen > 0 && s[-1] == ' ' )
		    { afterSpace= 1;	}
		break;

	    case DOCkindOBJECT:
		pictureDone= 0;
		io= bi->biParaObjects+ tp->tpObjectNumber;

		if  (   io->ioKind == DOCokPICTWMETAFILE		||
			io->ioKind == DOCokPICTPNGBLIP			||
			io->ioKind == DOCokPICTJPEGBLIP			||
			io->ioKind == DOCokMACPICT			||
		      ( io->ioKind == DOCokOLEOBJECT 		&&
		        io->ioResultKind == DOCokPICTWMETAFILE	)	)
		    {
		    if  ( docHtmlSavePicture( io, sos, &pictureDone, hwc ) )
			{ XDEB(io);	}
		    }

		if  ( ! pictureDone )
		    { docHtmlPutString( " ", hwc, sos );	}

		afterSpace= 0; s++; stroff++;
		break;

	    case DOCkindFIELDSTART:
		df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;
		fki= DOC_FieldKinds+ df->dfKind;

		if  ( fki->fkiIsFieldInRtf		&&
		      fki->fkiLevel == DOClevTEXT	)
		    { docHtmlStartField( sos, df, hwc, bd, bi, hpa, &hpa ); }

		if  ( df->dfKind == DOCfkBOOKMARK )
		    { docHtmlStartField( sos, df, hwc, bd, bi, hpa, &hpa ); }

		if  ( df->dfKind == DOCfkCHFTN )
		    {
		    int		count;
		    char	scratch[20+1];

		    count= docCountParticulesInField( bi, part, partUpto );

		    docHtmlStartField( sos, df, hwc, bd, bi, hpa, &hpa );

		    if  ( bi->biInExternalItem == DOCinBODY )
			{
			sprintf( scratch, "%d", hwc->hwcNoteRefCount+ 1 );
			hwc->hwcNoteRefCount++;
			}
		    else{
			sprintf( scratch, "%d", hwc->hwcNoteDefCount+ 1 );
			hwc->hwcNoteDefCount++;
			}

		    docHtmlPutString( scratch, hwc, sos );

		    if  ( hwc->hwcInHyperlink )
			{ hwc->hwcBytesInLink += strlen( scratch );	}

		    done= count+ 1;
		    stroff= tp[done].tpStroff; s= bi->biParaString+ stroff;
		    break;
		    }

		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		done= 1;
		break;

	    case DOCkindFIELDEND:
		df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;
		fki= DOC_FieldKinds+ df->dfKind;

		if  ( df->dfKind == DOCfkCHFTN )
		    { docHtmlFinishField( sos, df, hwc, bd, hpa, &hpa ); }

		if  ( df->dfKind == DOCfkBOOKMARK )
		    { docHtmlFinishField( sos, df, hwc, bd, hpa, &hpa ); }

		if  ( fki->fkiIsFieldInRtf		&&
		      fki->fkiLevel == DOClevTEXT	)
		    { docHtmlFinishField( sos, df, hwc, bd, hpa, &hpa ); }

		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		done= 1;
		break;

	    case DOCkindLINEBREAK:
	    case DOCkindPAGEBREAK:
		docHtmlPutString( "<BR>", hwc, sos );
		docHtmlNewLine( hwc, sos );
		afterSpace= 0;
		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 1 */
		break;

	    case DOCkindXE:
	    case DOCkindTC:
		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		break;

	    case DOCkindNOTE:
		s += tp->tpStrlen; stroff += tp->tpStrlen;
		break;

	    default:
		LDEB(tp->tpKind);
		s += tp->tpStrlen; stroff += tp->tpStrlen;
		break;
	    }

	done++; part++; tp++;
	}

    while( hpa )
	{ docHtmlPopAttributes( hwc, sos, hpa, &hpa ); }

    return done;
    }

/************************************************************************/
/*									*/
/*  The body of a paragraph is written as a <DIV> or <TD> division.	*/
/*									*/
/************************************************************************/

static void docHtmlStartParagraphBody(	const BufferItem *	bi,
					const char *		tag,
					int			fontHeight,
					HtmlWritingContext *	hwc,
					SimpleOutputStream *	sos )
    {
    sioOutPutCharacter( '<', sos ); hwc->hwcColumn++;
    docHtmlPutString( tag, hwc, sos );

    switch( bi->biParaAlignment )
	{
	case DOCiaLEFT:
	    break;
	case DOCiaRIGHT:
	    docHtmlPutString( " ALIGN=\"RIGHT\"", hwc, sos );
	    break;

	case DOCiaCENTERED:
	    docHtmlPutString( " ALIGN=\"CENTER\"", hwc, sos );
	    break;

	case DOCiaJUSTIFIED:
	    docHtmlPutString( " ALIGN=\"JUSTIFY\"", hwc, sos );
	    break;

	default:
	    LDEB(bi->biParaAlignment);
	    break;
	}

    docHtmlPutString( ">", hwc, sos );

    if  ( bi->biParaSpaceBeforeTwips > fontHeight/ 2 )
	{
	docHtmlPutString( "&nbsp;<BR>", hwc, sos );
	docHtmlNewLine( hwc, sos );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Use a HTML table for the implementation of the 'First Indent' of an	*/
/*  RTF paragraph?							*/
/*									*/
/************************************************************************/

static int docHtmlUseTableForIndent(	int *			pTabParticule,
					const BufferDocument *	bd,
					const BufferItem *	bi )
    {
    int			stroff;
    int			part;
    TextParticule *	tp;

    if  ( bi->biParaAlignment != DOCiaLEFT	&&
	  bi->biParaAlignment != DOCiaJUSTIFIED	)
	{ return 0;	}

    if  ( bi->biParaLeftIndentTwips <= 0 )
	{ return 0;	}
    if  ( bi->biParaFirstIndentTwips >= 0 )
	{ return 0;	}
    if  ( bi->biParaFirstIndentTwips < -bi->biParaLeftIndentTwips )
	{ return 0;	}

    tp= bi->biParaParticules;
    for ( part= 0; part < bi->biParaParticuleCount; tp++, part++ )
	{
	switch( tp->tpKind )
	    {
	    const DocumentField *		df;
	    const FieldKindInformation *	fki;

	    case DOCkindTAB:
		*pTabParticule= part;
		return 1;

	    case DOCkindFIELDSTART:
	    case DOCkindFIELDEND:
		df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;
		fki= DOC_FieldKinds+ df->dfKind;

		if  ( fki->fkiIsFieldInRtf		&&
		      fki->fkiLevel == DOClevTEXT	)
		    { return 0;	}

		if  ( df->dfKind == DOCfkBOOKMARK )
		    { return 0;	}

		break;

	    default:
		break;
	    }

	stroff= tp->tpStroff+ tp->tpStrlen;
	if  ( stroff > 10 )
	    { return 0;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save a 'Paragraph'							*/
/*									*/
/*  But not as a <P>...</P>.						*/
/*									*/
/************************************************************************/

static int docHtmlSaveParaItem(	SimpleOutputStream *		sos,
				const BufferDocument *		bd,
				const BufferItem *		bi,
				HtmlWritingContext *		hwc )
    {
    TextParticule *			tp;
    unsigned char *			s;

    int					useTABLE= 0;

    int					part= 0;
    int					stroff= 0;
    int					tabParticule;

    int					fontHeight;

    PropertyMask			ppChgMask;
    PropertyMask			ppUpdMask;

    if  ( bi->biParaParticuleCount == 0		||
	  bi->biParaStrlen == 0			)
	{
	if  ( hwc->hwcUseTableForFirstIndent )
	    {
	    docHtmlPutString( "</TABLE>&nbsp;<BR>", hwc, sos );
	    hwc->hwcUseTableForFirstIndent= 0;
	    }
	else{ docHtmlPutString( "<DIV>&nbsp;</DIV>", hwc, sos ); }

	docHtmlNewLine( hwc, sos );

	return 0;
	}

    useTABLE= docHtmlUseTableForIndent( &tabParticule, bd, bi );

    part= 0;
    stroff= 0;
    tp= bi->biParaParticules+ part;
    s= bi->biParaString+ stroff;
    fontHeight= 10* tp->tpTextAttribute.taFontSizeHalfPoints;

    if  ( ! useTABLE && hwc->hwcUseTableForFirstIndent )
	{
	hwc->hwcUseTableForFirstIndent= useTABLE;
	docHtmlPutString( "</TABLE>", hwc, sos );
	docHtmlNewLine( hwc, sos );
	}

    if  ( useTABLE )
	{
	int	left= TWIPS_TO_SIZE( bi->biParaLeftIndentTwips );

	if  ( ! hwc->hwcUseTableForFirstIndent )
	    {
	    docHtmlPutString(
		"<TABLE CELLPADDING=0 CELLSPACING=0><TR VALIGN=\"TOP\">",
								hwc, sos );
	    }
	else{
	    docHtmlPutString( "<TR VALIGN=\"TOP\">", hwc, sos );
	    }

	docHtmlPutString( "<TD", hwc, sos );
	docHtmlWriteTagArg( "WIDTH", left, hwc, sos );

	docHtmlPutString( ">", hwc, sos );
	if  ( ! hwc->hwcUseTableForFirstIndent )
	    { docHtmlNewLine( hwc, sos );	}

	if  ( bi->biParaSpaceBeforeTwips > fontHeight/ 2 )
	    {
	    docHtmlPutString( "&nbsp;<BR>", hwc, sos );
	    docHtmlNewLine( hwc, sos );
	    }

	if  ( docHtmlSaveParticules( sos, bd, bi,
					    part, tabParticule, hwc ) < 0 )
	    { LDEB(part); return -1;	}

	docHtmlPutString( "</TD>", hwc, sos );

	hwc->hwcUseTableForFirstIndent= useTABLE;

	docHtmlStartParagraphBody( bi, "TD", fontHeight, hwc, sos );

	part= tabParticule+ 1;
	}
    else{
	docHtmlStartParagraphBody( bi, "DIV", fontHeight, hwc, sos );
	}

    if  ( bi->biParaTopBorder.bpStyle != DOCbsNONE )
	{
	docHtmlPutString( "<HR NOSHADE WIDTH=100%>", hwc, sos );
	docHtmlNewLine( hwc, sos );
	}

    if  ( ! useTABLE							&&
	  bi->biParaFirstIndentTwips- bi->biParaLeftIndentTwips > 80	)
	{
	hwc->hwcMakeTransparentGif= 1;

	docHtmlMakeNameForIndent( hwc );

	docHtmlSaveImgTag( TWIPS_TO_SIZE( bi->biParaFirstIndentTwips ), 1,
								    hwc, sos );
	}

    if  ( docHtmlSaveParticules( sos, bd, bi,
				part, bi->biParaParticuleCount, hwc ) < 0 )
	{ LDEB(part); return -1;	}

    if  ( bi->biParaBottomBorder.bpStyle != DOCbsNONE )
	{
	docHtmlPutString( "<HR NOSHADE WIDTH=100%>", hwc, sos );
	docHtmlNewLine( hwc, sos );
	}

    if  ( hwc->hwcUseTableForFirstIndent )
	{
	docHtmlPutString( "</TD></TR>", hwc, sos );
	docHtmlNewLine( hwc, sos );
	}
    else{
	if  ( bi->biParaSpaceAfterTwips > fontHeight/ 2 )
	    { docHtmlPutString( "<BR>&nbsp;</DIV>", hwc, sos );	}
	else{ docHtmlPutString( "</DIV>", hwc, sos );			}
	docHtmlNewLine( hwc, sos );
	}

    PROPmaskCLEAR( &ppChgMask );

    PROPmaskCLEAR( &ppUpdMask );
    PROPmaskFILL( &ppUpdMask, PPprop_COUNT );

    if  ( docUpdParaProperties( &ppChgMask, &(hwc->hwcParagraphProperties),
					&ppUpdMask, &(bi->biParaProperties) ) )
	{ LDEB(1);	}

    return 0;
    }

static int docHtmlSaveRowItem(	SimpleOutputStream *		sos,
				BufferDocument *		bd,
				const BufferItem *		row,
				HtmlWritingContext *		hwc )
    {
    int				i;

    if  ( ! hwc->hwcParagraphProperties.ppInTable )
	{
	int			cellPadding;
	int			useBorder= 0;
	CellProperties *	cp;

	cp= row->biRowCells;
	for ( i= 0; i < row->biChildCount; cp++, i++ )
	    {
	    if  ( cp->cpTopBorder.bpStyle != DOCbsNONE		||
		  cp->cpLeftBorder.bpStyle != DOCbsNONE		||
		  cp->cpRightBorder.bpStyle != DOCbsNONE	||
		  cp->cpBottomBorder.bpStyle != DOCbsNONE	)
		{ useBorder= 1; break;	}
	    }

	docHtmlPutString( "<TABLE CELLSPACING=0", hwc, sos );

	if  ( useBorder )
	    { docHtmlPutString( " BORDER=\"1\"", hwc, sos ); }

	cellPadding= TWIPS_TO_SIZE( row->biRowHalfGapWidthTwips )- 4;
	if  ( cellPadding < 1 )
	    { docHtmlWriteTagArg( "CELLPADDING", 0, hwc, sos ); }
	if  ( cellPadding > 1 )
	    { docHtmlWriteTagArg( "CELLPADDING", cellPadding, hwc, sos ); }

	docHtmlPutString( "><TR VALIGN=\"TOP\">", hwc, sos );
	hwc->hwcParagraphProperties.ppInTable= 1;
	}
    else{
	docHtmlPutString( "<TR VALIGN=\"TOP\">", hwc, sos );
	}

    for ( i= 0; i < row->biChildCount; i++ )
	{
	int			j;
	BufferItem *		cellBi= row->biChildren[i];
	int			wide;

	int			bottomTwips= -1;
	int			stripHigh= -1;
	const int		page= 0;
	const int		column= 0;

	BlockFrame		bf;
	ParagraphFrame		pf;

	docBlockFrameTwips( &bf, cellBi->biChildren[0], bd, page, column );

	docParagraphFrameTwips( &pf, &bf, bottomTwips, stripHigh,
						    cellBi->biChildren[0] );
	wide= TWIPS_TO_SIZE( pf.pfX1GeometryTwips- pf.pfX0GeometryTwips );

	docHtmlPutString( "<TD", hwc, sos );
	docHtmlWriteTagArg( "WIDTH", wide, hwc, sos );
	docHtmlPutString( ">", hwc, sos );
	docHtmlNewLine( hwc, sos );

	for ( j= 0; j < cellBi->biChildCount; j++ )
	    {
	    BufferItem *	para= cellBi->biChildren[j];

	    if  ( docHtmlSaveParaItem( sos, bd, para, hwc ) )
		{ LDEB(1); return -1;	}
	    }

	docHtmlFinishGroup( sos, hwc );

	docHtmlPutString( "</TD>", hwc, sos );
	if  ( i < row->biChildCount- 1 )
	    { docHtmlNewLine( hwc, sos );	}
	}

    docHtmlPutString( "</TR>", hwc, sos );
    docHtmlNewLine( hwc, sos );

    return 0;
    }

static int docHtmlSaveItem(	SimpleOutputStream *		sos,
				BufferDocument *		bd,
				const BufferItem *		bi,
				HtmlWritingContext *		hwc )
    {
    int			i;

    switch( bi->biLevel )
	{
	case DOClevSECT:
	case DOClevDOC:
	case DOClevCELL:
	rowAsGroup:
	    for ( i= 0; i < bi->biChildCount; i++ )
		{
		if  ( docHtmlSaveItem( sos, bd,
					    bi->biChildren[i], hwc ) )
		    { LDEB(i); return -1;	}
		}
	    break;

	case DOClevROW:
	    if  ( ! bi->biRowHasTableParagraphs )
		{
		if  ( hwc->hwcParagraphProperties.ppInTable )
		    {
		    docHtmlPutString( "</TABLE>", hwc, sos );
		    docHtmlNewLine( hwc, sos );
		    hwc->hwcParagraphProperties.ppInTable= 0;
		    }

		goto rowAsGroup;
		}

	    if  ( docHtmlSaveRowItem( sos, bd, bi, hwc ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevPARA:
	    if  ( docHtmlSaveParaItem( sos, bd, bi, hwc ) )
		{ LDEB(1); return -1;	}
	    break;

	default:
	    LDEB(bi->biLevel); return -1;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Start to output an image						*/
/*									*/
/************************************************************************/

static SimpleOutputStream * docHtmlStartImage(	HtmlWritingContext *	hwc,
						SimpleOutputStream *	sos )
    {
    SimpleOutputStream *	sosImage;

    if  ( hwc->hwcAsMimeAggregate )
	{
	sioOutPutCharacter( '-', sos ); sioOutPutCharacter( '-', sos );
	sioOutPutString( hwc->hwcMimeBoundary, sos );
	sioOutPutCharacter( '\r', sos ); sioOutPutCharacter( '\n', sos );

	sioOutPutString( "Content-Type: ", sos );
	sioOutPutString( hwc->hwcImageEncoding, sos );
	sioOutPutCharacter( '\r', sos ); sioOutPutCharacter( '\n', sos );

	sioOutPutString( "Content-Id: <", sos );
	sioOutPutString( hwc->hwcNameScratch, sos );
	sioOutPutCharacter( '.', sos );
	sioOutPutString( hwc->hwcContentIdTail, sos );
	sioOutPutCharacter( '>', sos );
	sioOutPutCharacter( '\r', sos ); sioOutPutCharacter( '\n', sos );

	sioOutPutString( "Content-Transfer-Encoding: ", sos );
	sioOutPutString( "base64", sos );
	sioOutPutCharacter( '\r', sos ); sioOutPutCharacter( '\n', sos );

	sioOutPutString( "Content-Disposition: inline;", sos );
	sioOutPutCharacter( '\r', sos ); sioOutPutCharacter( '\n', sos );
	sioOutPutString( " filename=\"", sos );
	sioOutPutString( hwc->hwcNameScratch, sos );
	sioOutPutCharacter( '"', sos );
	sioOutPutCharacter( '\r', sos ); sioOutPutCharacter( '\n', sos );

	sioOutPutCharacter( '\r', sos ); sioOutPutCharacter( '\n', sos );

	sosImage= sioOutBase64Open( sos );
	if  ( ! sosImage )
	    { SXDEB(hwc->hwcNameScratch,sosImage); return sosImage; }
	}
    else{
	sosImage= sioOutStdioOpen( hwc->hwcNameScratch );
	if  ( ! sosImage )
	    { SXDEB(hwc->hwcNameScratch,sosImage); return sosImage; }
	}

    return sosImage;
    }

/************************************************************************/
/*									*/
/*  Save a completely transparent gif to use for positive First Indents	*/
/*									*/
/************************************************************************/

static int docHtmlSaveTransparentGif(	HtmlWritingContext *	hwc,
					SimpleOutputStream *	sos )
    {
    SimpleOutputStream *	sosImage;
    AppBitmapImage *		abi;

    docHtmlMakeNameForIndent( hwc );

    abi= (AppBitmapImage *)malloc( sizeof(AppBitmapImage) );
    if  ( ! abi )
	{ XDEB(abi); return -1;	}
    appInitBitmapImage( abi );

    if  ( bmTransparentImage( &(abi->abiBitmap), &(abi->abiBuffer),
						    BMcoRGB8PALETTE, 1, 1 ) )
	{ free(abi); return -1;	}

    sosImage= docHtmlStartImage( hwc, sos );
    if  ( ! sosImage )
	{ XDEB(sosImage); return -1; }

    if  ( (*hwc->hwcWriteThisBitmap)( &(abi->abiBitmap), abi->abiBuffer,
								sosImage ) )
	{
	LDEB(1);
	sioOutClose( sosImage );
	appCleanBitmapImage( abi ); free( abi );
	return -1;
	}

    sioOutClose( sosImage );

    appCleanBitmapImage( abi ); free( abi ); return 0;
    }

/************************************************************************/
/*									*/
/*  Save the images in the document.					*/
/*									*/
/************************************************************************/

static int docHtmlSaveImageBytes(	HtmlWritingContext *	hwc,
					SimpleOutputStream *	sos,
					InsertedObject *	io )
    {
    int				res;
    SimpleOutputStream *	sosImage;
    AppBitmapImage *		abi;

    res= docHtmlMakeNameForImage( hwc, io );
    if  ( res < 0 )
	{ LDEB(res); return -1;	}
    if  ( res > 0 )
	{ return 0;	}

    abi= (AppBitmapImage *)io->ioPrivate;

    sosImage= docHtmlStartImage( hwc, sos );
    if  ( ! sosImage )
	{ XDEB(sosImage); return -1; }

    if  ( (*hwc->hwcWriteThisBitmap)( &(abi->abiBitmap), abi->abiBuffer,
								sosImage ) )
	{ LDEB(1); sioOutClose( sosImage ); return -1;	}

    sioOutClose( sosImage );

    return 0;
    }

static int docHtmlSaveImages(	SimpleOutputStream *		sos,
				const BufferDocument *		bd,
				const BufferItem *		bi,
				HtmlWritingContext *		hwc )
    {
    int			i;
    TextParticule *	tp;

    switch( bi->biLevel )
	{
	case DOClevSECT:
	case DOClevDOC:
	case DOClevCELL:
	case DOClevROW:
	    for ( i= 0; i < bi->biChildCount; i++ )
		{
		if  ( docHtmlSaveImages( sos, bd, bi->biChildren[i], hwc ) )
		    { LDEB(i); return -1;	}
		}

	    return 0;

	case DOClevPARA:
	    break;

	default:
	    LDEB(bi->biLevel); return -1;
	}

    tp= bi->biParaParticules;
    for ( i= 0; i < bi->biParaParticuleCount; tp++, i++ )
	{
	InsertedObject *	io;

	if  ( tp->tpKind != DOCkindOBJECT )
	    { continue;	}

	io= bi->biParaObjects+ tp->tpObjectNumber;

	if  (   io->ioKind == DOCokPICTWMETAFILE		||
		io->ioKind == DOCokPICTPNGBLIP			||
		io->ioKind == DOCokPICTJPEGBLIP			||
		io->ioKind == DOCokMACPICT			||
	      ( io->ioKind == DOCokOLEOBJECT 		&&
		io->ioResultKind == DOCokPICTWMETAFILE	)	)
	    {
	    if  ( docHtmlSaveImageBytes( hwc, sos, io ) )
		{ LDEB(1); return -1;	}
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save a document to HTML, or to a html mail mime aggregate.		*/
/*									*/
/************************************************************************/

int docHtmlSaveDocument(	SimpleOutputStream *	sos,
				BufferDocument *	bd,
				int			asMimeAggr,
				const char *		mimeBoundary,
				const char *		filename )
    {
    HtmlWritingContext		hwc;

    const BufferItem *		bi= &bd->bdItem;
    const DocumentProperties *	dp= &(bd->bdProperties);
    const DocumentGeometry *	dg= &(dp->dpGeometry);

#   if 0
    const DocumentFontList *	dfl= &bd->bdFontList;
    const DocumentFont *	df;
    int				i;
#   endif

    docInitHtmlWritingContext( &hwc );

    hwc.hwcAsMimeAggregate= asMimeAggr;
    hwc.hwcMimeBoundary= mimeBoundary;

    if  ( filename )
	{
	const char *	s;

	hwc.hwcFilename= filename;
	s= strrchr( filename, '.' );
	if  ( s )
	    { hwc.hwcBaselength= s- filename;		}
	else{ hwc.hwcBaselength= strlen( filename );	}

	s= strrchr( filename, '/' );
	if  ( s )
	    { hwc.hwcRelativeOffset= s- filename+ 1;	}
	else{ hwc.hwcRelativeOffset= 0;			}
	}

    if  ( hwc.hwcAsMimeAggregate )
	{
	if  ( appMakeUniqueString( hwc.hwcContentIdTail, IDlenTAIL ) )
	    { LDEB(IDlenTAIL); return -1;	}

	sioOutPutCharacter( '-', sos ); sioOutPutCharacter( '-', sos );
	sioOutPutString( hwc.hwcMimeBoundary, sos );
	sioOutPutCharacter( '\r', sos ); sioOutPutCharacter( '\n', sos );
	sioOutPutString( "Content-Type: text/html", sos );
	sioOutPutCharacter( '\r', sos ); sioOutPutCharacter( '\n', sos );
	sioOutPutString( "Content-Transfer-Encoding: 8bit", sos );
	sioOutPutCharacter( '\r', sos ); sioOutPutCharacter( '\n', sos );

	sioOutPutCharacter( '\r', sos ); sioOutPutCharacter( '\n', sos );
	}

    docHtmlPutString( "<HTML>", &hwc, sos );
    docHtmlNewLine( &hwc, sos );

#   if 0
    docHtmlPutString( "<STYLE>", &hwc, sos );
    docHtmlNewLine( &hwc, sos );

    df= dfl->dflFonts;
    for ( i= 0; i < dfl->dflCount; df++, i++ )
	{
	char *			font= (char *)0;
	char			scratch[15];

	sprintf( scratch, "%d", i );

	if  ( ! strcmp( df->dfFamilyStyle, "fswiss" ) )
	    { font= "Helvetica,Arial,sans-serif";	}
	if  ( ! strcmp( df->dfFamilyStyle, "froman" ) )
	    { font= "Times,Times New Roman,serif";	}
	if  ( ! strcmp( df->dfFamilyStyle, "fmodern" ) )
	    { font= "Courier,monospace";	}
	if  ( ! strcmp( df->dfFamilyStyle, "ftech" ) )
	    { font= "Symbol";	}

	docHtmlPutString( "FONT.f", &hwc, sos );
	docHtmlPutString( scratch, &hwc, sos );
	docHtmlPutString( " {", &hwc, sos );
	if  ( font )
	    {
	    docHtmlPutString( "font-family: ", &hwc, sos );
	    docHtmlPutString( font, &hwc, sos );
	    }

	docHtmlPutString( " }", &hwc, sos );
	docHtmlNewLine( &hwc, sos );
	}

    docHtmlPutString( "</STYLE>", &hwc, sos );
    docHtmlNewLine( &hwc, sos );
#   endif

    if  ( docHasDocumentInfo( dp ) )
	{
	docHtmlPutString( "<HEAD>", &hwc, sos );
	docHtmlNewLine( &hwc, sos );

	if  ( dp->dpTitle )
	    {
	    docHtmlPutString( "<TITLE>", &hwc, sos );
	    docHtmlEscapeString( dp->dpTitle, &hwc, sos );
	    docHtmlPutString( "</TITLE>", &hwc, sos );
	    docHtmlNewLine( &hwc, sos );
	    }

	if  ( dp->dpSubject )
	    {
	    docHtmlPutString(
			"<META NAME=\"description\" CONTENT=\"", &hwc, sos );
	    docHtmlEscapeString( dp->dpSubject, &hwc, sos );
	    docHtmlPutString( "\">", &hwc, sos );
	    docHtmlNewLine( &hwc, sos );
	    }

	if  ( dp->dpKeywords )
	    {
	    docHtmlPutString(
			"<META NAME=\"keywords\" CONTENT=\"", &hwc, sos );
	    docHtmlEscapeString( dp->dpKeywords, &hwc, sos );
	    docHtmlPutString( "\">", &hwc, sos );
	    docHtmlNewLine( &hwc, sos );
	    }

	if  ( dp->dpComment )
	    {
	    docHtmlPutString(
			"<META NAME=\"comment\" CONTENT=\"", &hwc, sos );
	    docHtmlEscapeString( dp->dpComment, &hwc, sos );
	    docHtmlPutString( "\">", &hwc, sos );
	    docHtmlNewLine( &hwc, sos );
	    }

	if  ( dp->dpAuthor )
	    {
	    docHtmlPutString(
			"<META NAME=\"author\" CONTENT=\"", &hwc, sos );
	    docHtmlEscapeString( dp->dpAuthor, &hwc, sos );
	    docHtmlPutString( "\">", &hwc, sos );
	    docHtmlNewLine( &hwc, sos );
	    }

	if  ( dp->dpHlinkbase )
	    {
	    docHtmlPutString( "<BASE HREF=\"", &hwc, sos );
	    docHtmlEscapeString( dp->dpHlinkbase, &hwc, sos );
	    docHtmlPutString( "\">", &hwc, sos );
	    docHtmlNewLine( &hwc, sos );
	    }

	docHtmlPutString( "</HEAD>", &hwc, sos );
	docHtmlNewLine( &hwc, sos );
	}

    docHtmlPutString( "<BODY BGCOLOR=\"#ffffff\" TEXT=\"#000000\">",
								&hwc, sos );
    docHtmlNewLine( &hwc, sos );

    if  ( dg->dgLeftMarginTwips > 300		||
	  dg->dgRightMarginTwips > 300		||
	  dg->dgTopMarginTwips > 300		||
	  dg->dgBottomMarginTwips > 300		)
	{
	docHtmlPutString( "<TABLE>", &hwc, sos );
	docHtmlNewLine( &hwc, sos );

	if  ( dg->dgTopMarginTwips > 300 )
	    {
	    docHtmlPutString( "<TR><TD", &hwc, sos );

	    docHtmlWriteTagArg( "HEIGHT",
		TWIPS_TO_SIZE( dg->dgTopMarginTwips ), &hwc, sos );

	    docHtmlPutString( ">&nbsp;</TD></TR>", &hwc, sos );
	    }

	docHtmlPutString( "<TR>", &hwc, sos );

	if  ( dg->dgLeftMarginTwips > 300 )
	    {
	    docHtmlPutString( "<TD", &hwc, sos );

	    docHtmlWriteTagArg( "WIDTH",
		TWIPS_TO_SIZE( dg->dgLeftMarginTwips ), &hwc, sos );

	    docHtmlPutString( ">&nbsp;</TD>", &hwc, sos );
	    }

	docHtmlPutString( "<TD>", &hwc, sos );
	docHtmlNewLine( &hwc, sos );
	}

    if  ( docHtmlSaveItem( sos, bd, bi, &hwc ) )
	{ LDEB(bi->biLevel); return -1; }

    docHtmlFinishGroup( sos, &hwc );

    if  ( hwc.hwcNoteRefCount > 0 )
	{
	int		i;
	DocumentNote *	dn;

	docHtmlPutString( "<HR>", &hwc, sos );

	dn= bd->bdNotes;
	for ( i= 0; i < bd->bdNoteCount; dn++, i++ )
	    {
	    ExternalItem *	ei;

	    if  ( dn->dnParaNr < 0 )
		{ continue;	}

	    ei= &(dn->dnExternalItem);;
	    if  ( ! ei->eiItem )
		{ XDEB(ei->eiItem); continue;	}

	    if  ( docHtmlSaveItem( sos, bd, ei->eiItem, &hwc ) )
		{ LDEB(bi->biLevel); return -1; }

	    docHtmlFinishGroup( sos, &hwc );
	    }

	if  ( hwc.hwcNoteDefCount != hwc.hwcNoteRefCount )
	    { LLDEB(hwc.hwcNoteDefCount,hwc.hwcNoteRefCount);	}
	}

    if  ( dg->dgLeftMarginTwips > 300		||
	  dg->dgRightMarginTwips > 300		||
	  dg->dgTopMarginTwips > 300		||
	  dg->dgBottomMarginTwips > 300		)
	{
	docHtmlPutString( "</TD>", &hwc, sos );

	if  ( dg->dgRightMarginTwips > 300 )
	    {
	    docHtmlPutString( "<TD", &hwc, sos );

	    docHtmlWriteTagArg( "WIDTH",
		TWIPS_TO_SIZE( dg->dgRightMarginTwips ), &hwc, sos );

	    docHtmlPutString( ">&nbsp;</TD>", &hwc, sos );
	    }

	docHtmlPutString( "</TR>", &hwc, sos );

	if  ( dg->dgBottomMarginTwips > 300 )
	    {
	    docHtmlPutString( "<TR><TD", &hwc, sos );

	    docHtmlWriteTagArg( "HEIGHT",
		TWIPS_TO_SIZE( dg->dgBottomMarginTwips ), &hwc, sos );

	    docHtmlPutString( ">&nbsp;</TD></TR>", &hwc, sos );
	    }

	docHtmlPutString( "</TABLE>", &hwc, sos );
	docHtmlNewLine( &hwc, sos );
	}

    docHtmlPutString( "</BODY></HTML>", &hwc, sos );
    docHtmlNewLine( &hwc, sos );

    if  ( ! hwc.hwcAsMimeAggregate		&&
	  ( hwc.hwcMakeTransparentGif	||
	    hwc.hwcImageCount > 0	)	)
	{
	strncpy( hwc.hwcNameScratch, hwc.hwcFilename, hwc.hwcBaselength );
	strcpy( hwc.hwcNameScratch+ hwc.hwcBaselength, ".img" );

	if  ( appTestDirectory( hwc.hwcNameScratch )	&&
	      appMakeDirectory( hwc.hwcNameScratch )	)
	    { SDEB(hwc.hwcNameScratch); return -1;	}
	}

    if  ( hwc.hwcMakeTransparentGif			&&
	  docHtmlSaveTransparentGif( &hwc, sos )	)
	{ LDEB(hwc.hwcMakeTransparentGif); return -1;	}

    if  ( hwc.hwcImageCount > 0				&&
	  docHtmlSaveImages( sos, bd, bi, &hwc )	)
	{ LDEB(hwc.hwcImageCount); return -1;	}

    if  ( hwc.hwcAsMimeAggregate )
	{
	sioOutPutCharacter( '-', sos ); sioOutPutCharacter( '-', sos );
	sioOutPutString( hwc.hwcMimeBoundary, sos );
	sioOutPutCharacter( '-', sos ); sioOutPutCharacter( '-', sos );
	sioOutPutCharacter( '\r', sos ); sioOutPutCharacter( '\n', sos );
	}

    docCleanHtmlWritingContext( &hwc );

    return 0;
    }
