/************************************************************************/
/*									*/
/*  Text Editor Buffer structure.					*/
/*									*/
/************************************************************************/

#   ifndef	DOC_BUF_H
#   define	DOC_BUF_H

#   include	<time.h>

#   include	<bitmap.h>
#   include	<docFont.h>
#   include	"docShape.h"
#   include	<appGeo.h>
#   include	<sioGeneral.h>
#   include	<utilMemoryBuffer.h>

#   include	"docParaProperties.h"
#   include	"docRowProperties.h"
#   include	"docSectProperties.h"
#   include	"docStyleSheet.h"
#   include	"docDocumentProperties.h"
#   include	"docShapeProperties.h"
#   include	"docField.h"
#   include	"docObject.h"
#   include	"docSelect.h"
#   include	"docExternalItem.h"

/************************************************************************/
/*									*/
/*  Forward declarations of structs.					*/
/*									*/
/************************************************************************/

struct BufferDocument;
struct BufferItem;
struct DocumentField;

/************************************************************************/
/*									*/
/*  An homogenous piece of text. Without hyphenation, it will not be	*/
/*  divided over lines.							*/
/*  It consists of some characters and then some spaces.		*/
/*									*/
/************************************************************************/

typedef enum DocParticuleKind
    {
    DOCkindUNKNOWN= 0,

    DOCkindTEXT,
    DOCkindTAB,
    DOCkindOBJECT,
    DOCkindFIELDSTART,
    DOCkindFIELDEND,
    DOCkindNOTE,
    DOCkindTC,
    DOCkindXE,
    DOCkindLINEBREAK,
    DOCkindPAGEBREAK,
    DOCkindCHFTNSEP,

    DOCkind_COUNT
    } DocParticuleKind;

typedef struct TextParticule
    {
    unsigned int	tpStroff:16;
    unsigned int	tpStrlen:12;
    unsigned int	tpX0:12;
    unsigned int	tpPixelsWide:12;
    unsigned int	tpKind:6;
    unsigned int	tpInField:1;
    short int		tpPhysicalFont;
    short int		tpObjectNumber;
    TextAttribute	tpTextAttribute;
    } TextParticule;

/************************************************************************/
/*									*/
/*  Used to lay out the text on the page.				*/
/*  A text line consists of a series of particules.			*/
/*									*/
/*  1)  What part of the string in the paragraph.			*/
/*  2)  What particules.						*/
/*  3)  Where is it geo/typographically?				*/
/*  4)  Postscript geometry information.				*/
/*	lineSpacing includes the leading that is irrelevant for the	*/
/*	last line of a page.						*/
/*  5)  Screen geometry.						*/
/*									*/
/************************************************************************/

typedef struct TextLine
    {
								/*  1,2	*/
    int			tlStroff;
    int			tlFirstParticule;
    short int		tlStrlen;
    short int		tlParticuleCount;
    short int		tlWordCount;
								/*  3  */
    LayoutPosition	tlTopPosition;
								/*  4  */
    short int		tlLineAscentTwips;
    short int		tlLineHeightTwips;
    short int		tlLineSpacingTwips;
								/*  5  */
    short int		tlHasPageBreak;

    short int		tlX0Pixels;
    short int		tlX1Pixels;
    } TextLine;

/************************************************************************/
/*									*/
/*  Information on Lists.						*/
/*									*/
/************************************************************************/

typedef struct DocumentList
    {
    int			dlListId;
    int			dlListTemplateId;
    int			dlListIsSimple;
    int			dlRestartForEverySection;
    char *		dlListName;
    } DocumentList;

typedef struct BufferPara
    {
    unsigned char *		btString;
    unsigned int		btStrlen;

    int				btParticuleCount;
    TextParticule *		btParticules;

    int				btObjectCount;
    InsertedObject *		btObjects;

    int				btLineCount;
    TextLine *			btLines;

    int				btShapeCount;
    DrawingShape *		btShapes;

    ParagraphProperties		btProperties;

    int				btAscentTwips;
    int				btDescentTwips;
    int				btLeadingTwips;

    int				btSpaceAboveLinesTwips;
    const BorderProperties *	btBorderAboveParagraph;

    int				btSpaceBelowLinesTwips;
    const BorderProperties *	btBorderBelowParagraph;
    } BufferPara;

/************************************************************************/
/*									*/
/*  Type dependent part of a BufferItem of 'Section' level.		*/
/*									*/
/*  Section level BufferItems are the sections of the document and the	*/
/*  'External Items' that are not part of the regular tree of items	*/
/*  that forms the document. Examples of 'External Items' are the page	*/
/*  headers and footers and the footnotes and endnotes in the document.	*/
/*									*/
/*  The root of external items are recognised by the fact that		*/
/*  bi->biInExternalItem != DOCinBODY. In practice there are no parents	*/
/*  to the section level items in external items. This is accidental	*/
/*  and should not be used as a recognition criterion.			*/
/*									*/
/*  1)  Formatting properties of the section.				*/
/*  2)  Page Headers.							*/
/*  3)  Page Footers.							*/
/*  5)  Number of the section in the document tree that an external	*/
/*	item belongs to.						*/
/*									*/
/************************************************************************/

typedef struct BufferSect
    {
				/*  1  */
    SectionProperties		bsProperties;

				/*  2  */
    ExternalItem		bsHeader;
    ExternalItem		bsFirstPageHeader;
    ExternalItem		bsLeftPageHeader;
    ExternalItem		bsRightPageHeader;

				/*  3  */
    ExternalItem		bsFooter;
    ExternalItem		bsFirstPageFooter;
    ExternalItem		bsLeftPageFooter;
    ExternalItem		bsRightPageFooter;

				/*  5  */
    SelectionScope		bsSelectionScope;

				/****************************************/
				/*					*/
				/*  Not used in the body of a document:	*/
				/*					*/
				/*  A)  Normally not set. Is set by the	*/
				/*	formatter to pass information	*/
				/*	to calculate page numbers.	*/
				/*  B)  Normally not set. In headers	*/
				/*	and footers it is the number of	*/
				/*	the page it is formatted for.	*/
				/*	[ Either during the formatting	*/
				/*	process or after formatting has	*/
				/*	been finished.]			*/
				/*					*/
				/****************************************/
    const struct BufferItem *	bsUseForSectBi;			/*  A	*/
    int				bsUseForPage;			/*  B	*/

				/*  A  */
#   define			biSectHeaderFooterUseForSectBi \
					    BIU.biuSect.bsUseForSectBi
				/*  B	*/
#   define			biSectHeaderFooterUseForPage \
					    BIU.biuSect.bsUseForPage
    } BufferSect;

typedef struct BufferRow
    {
    RowProperties		bguRowProperties;
    } BufferRow;

/************************************************************************/
/*									*/
/*  Levels of nesting.							*/
/*									*/
/************************************************************************/

typedef enum ItemLevel
    {
    DOClevANY,
    DOClevOUT,
			/****************************************/
			/*  Ignore; Garbage values.		*/
			/****************************************/
    DOClevDOC,
			/****************************************/
			/*					*/
			/*  Document.				*/
			/*					*/
			/*  * biGroupChildren			*/
			/*  * biGroupChildCount			*/
			/****************************************/
    DOClevSECT,
			/****************************************/
			/*					*/
			/*  Section.				*/
			/*					*/
			/*  * biGroupChildren			*/
			/*  * biGroupChildCount			*/
			/*					*/
			/****************************************/

#	define	biSectHeader		BIU.biuSect.bsHeader
#	define	biSectFirstPageHeader	BIU.biuSect.bsFirstPageHeader
#	define	biSectLeftPageHeader	BIU.biuSect.bsLeftPageHeader
#	define	biSectRightPageHeader	BIU.biuSect.bsRightPageHeader

#	define	biSectFooter		BIU.biuSect.bsFooter
#	define	biSectFirstPageFooter	BIU.biuSect.bsFirstPageFooter
#	define	biSectLeftPageFooter	BIU.biuSect.bsLeftPageFooter
#	define	biSectRightPageFooter	BIU.biuSect.bsRightPageFooter

#	define	biSectSelectionScope	BIU.biuSect.bsSelectionScope

#	define	biSectProperties	BIU.biuSect.bsProperties

#	define	biSectDocumentGeometry	biSectProperties.spDocumentGeometry
#	define	biSectPageWideTwips	biSectProperties.spPageWideTwips
#	define	biSectPageHighTwips	biSectProperties.spPageHighTwips
#	define	biSectLeftMarginTwips	biSectProperties.spLeftMarginTwips
#	define	biSectTopMarginTwips	biSectProperties.spTopMarginTwips
#	define	biSectRightMarginTwips	biSectProperties.spRightMarginTwips
#	define	biSectBottomMarginTwips	biSectProperties.spBottomMarginTwips
#	define	biSectHeaderYTwips	biSectProperties.spHeaderYTwips
#	define	biSectFooterYTwips	biSectProperties.spFooterYTwips

#	define	biSectHasTitlePage	biSectProperties.spHasTitlePage
#	define	biSectBreakKind		biSectProperties.spBreakKind
#	define	biSectPageNumberStyle	biSectProperties.spPageNumberStyle
#	define	biSectRestartPageNumbers \
					biSectProperties.spRestartPageNumbers
#	define	biSectStartPageNumber	biSectProperties.spStartPageNumber

#	define	biSectParagraphNumbers	biSectProperties.spParagraphNumbers
#	define	biSectParagraphNumberCount \
					biSectProperties.spParagraphNumberCount

			/****************************************/

    DOClevROW,
			/****************************************/
			/*  Row.				*/
			/*  * biGroupChildren			*/
			/*  * biGroupChildCount			*/
			/****************************************/
#	define	biRowProperties		BIU.biuRow.bguRowProperties
#	define	biRowHasTableParagraphs	biRowProperties.rpHasTableParagraphs
#	define	biRowHalfGapWidthTwips	biRowProperties.rpHalfGapWidthTwips
#	define	biRowHalfGapWidthPixels \
					biRowProperties.rpHalfGapWidthPixels
#	define	biRowLeftIndentTwips	biRowProperties.rpLeftIndentTwips
#	define	biRowLeftIndentPixels	biRowProperties.rpLeftIndentPixels
#	define	biRowHeightTwips	biRowProperties.rpHeightTwips
#	define	biRowIsTableHeader	biRowProperties.rpIsTableHeader
#	define	biRowKeepOnOnePage	biRowProperties.rpKeepOnOnePage
#	define	biRowHasHorizontalBorders \
					biRowProperties.rpHasHorizontalBorders
#	define	biRowHasVerticalBorders \
					biRowProperties.rpHasVerticalBorders
#	define	biRowCells		biRowProperties.rpCells
#	define	biRowCellCount		biRowProperties.rpCellCount
#	define	biRowAlignment		biRowProperties.rpAlignment

#	define	biRowTopBorder		biRowProperties.rpTopBorder
#	define	biRowBottomBorder	biRowProperties.rpBottomBorder
#	define	biRowLeftBorder		biRowProperties.rpLeftBorder
#	define	biRowRightBorder	biRowProperties.rpRightBorder
#	define	biRowVerticalBorder	biRowProperties.rpVerticalBorder
#	define	biRowHorizontalBorder	biRowProperties.rpHorizontalBorder

    DOClevCELL,
			/****************************************/
			/*  Cell.				*/
			/*  * biGroupChildren			*/
			/*  * biGroupChildCount			*/
			/****************************************/
	/* No.. Information is stored with the row.
	define	biCellProperties	BIU.biuGroup.BGU.bguCellProperties
	*/

    DOClevPARA,
			/********************************/
			/*  Paragraph.			*/
			/********************************/
#	define	biParaString		BIU.biuPara.btString
#	define	biParaStrlen		BIU.biuPara.btStrlen
#	define	biParaParticuleCount	BIU.biuPara.btParticuleCount
#	define	biParaParticules	BIU.biuPara.btParticules
#	define	biParaObjectCount	BIU.biuPara.btObjectCount
#	define	biParaObjects		BIU.biuPara.btObjects
#	define	biParaLineCount		BIU.biuPara.btLineCount
#	define	biParaLines		BIU.biuPara.btLines
#	define	biParaShapeCount	BIU.biuPara.btShapeCount
#	define	biParaShapes		BIU.biuPara.btShapes
#	define	biParaProperties	BIU.biuPara.btProperties
#	define	biParaInTable		biParaProperties.ppInTable
#	define	biParaLineSpacingIsMultiple \
					biParaProperties.ppLineSpacingIsMultiple
#	define	biParaStartsOnNewPage	biParaProperties.ppStartsOnNewPage
#	define	biParaFirstIndentTwips	biParaProperties.ppFirstIndentTwips
#	define	biParaLeftIndentTwips	biParaProperties.ppLeftIndentTwips
#	define	biParaRightIndentTwips	biParaProperties.ppRightIndentTwips
#	define	biParaSpaceBeforeTwips	biParaProperties.ppSpaceBeforeTwips
#	define	biParaSpaceAfterTwips	biParaProperties.ppSpaceAfterTwips
#	define	biParaLineSpacingTwips	biParaProperties.ppLineSpacingTwips
#	define	biParaAlignment		biParaProperties.ppAlignment
#	define	biParaTabStops		biParaProperties.ppTabStops
#	define	biParaTabCount		biParaProperties.ppTabCount
#	define	biParaKeepOnPage	biParaProperties.ppKeepOnPage
#	define	biParaKeepWithNext	biParaProperties.ppKeepWithNext
#	define	biParaShadingLevel	biParaProperties.ppShadingLevel
#	define	biParaShadingPattern	biParaProperties.ppShadingPattern
#	define	biParaOutlineLevel	biParaProperties.ppOutlineLevel
#	define	biParaTopBorder		biParaProperties.ppTopBorder
#	define	biParaBottomBorder	biParaProperties.ppBottomBorder
#	define	biParaLeftBorder	biParaProperties.ppLeftBorder
#	define	biParaRightBorder	biParaProperties.ppRightBorder
#	define	biParaBoxBorder		biParaProperties.ppBoxBorder
#	define	biParaBetweenBorder	biParaProperties.ppBetweenBorder
#	define	biParaBar		biParaProperties.ppBar
#	define	biParaStyle		biParaProperties.ppStyle
#	define	biParaWidowControl	biParaProperties.ppWidowControl

#	define	biParaAscentTwips	BIU.biuPara.btAscentTwips
#	define	biParaDescentTwips	BIU.biuPara.btDescentTwips
#	define	biParaLeadingTwips	BIU.biuPara.btLeadingTwips

#	define	biParaSpaceAboveLinesTwips \
					BIU.biuPara.btSpaceAboveLinesTwips
#	define	biParaBorderAboveParagraph \
					BIU.biuPara.btBorderAboveParagraph
#	define	biParaSpaceBelowLinesTwips \
					BIU.biuPara.btSpaceBelowLinesTwips
#	define	biParaBorderBelowParagraph \
					BIU.biuPara.btBorderBelowParagraph

    DOClevTEXT
			/********************************/
			/*  Handeled inside TEXT.	*/
			/********************************/
    } ItemLevel;

typedef struct BufferItem
    {
    short int			biLevel;
    unsigned char		biInExternalItem;
    struct BufferItem *		biParent;
    struct BufferItem **	biChildren;
    short int			biChildCount;
    short int			biNumberInParent;
    int				biLeftParagraphs;

    LayoutPosition		biTopPosition;
    LayoutPosition		biBelowPosition;

    union	BIU
	{
	BufferPara	biuPara;
	BufferSect	biuSect;
	BufferRow	biuRow;
	} BIU;

    } BufferItem;

/************************************************************************/
/*									*/
/*  The document as a whole.						*/
/*									*/
/************************************************************************/

typedef struct BufferDocument
    {
    BufferItem		bdItem;
    DocumentProperties	bdProperties;

    DocumentStyleSheet	bdStyleSheet;
    DocumentFieldList	bdFieldList;

    DocumentNote *	bdNotes;
    int			bdNoteCount;

    ExternalItem	bdEiFtnsep;
    ExternalItem	bdEiFtnsepc;
    ExternalItem	bdEiFtncn;

    ExternalItem	bdEiAftnsep;
    ExternalItem	bdEiAftnsepc;
    ExternalItem	bdEiAftncn;
    } BufferDocument;

/************************************************************************/
/*									*/
/*  Statistics about a document. Used in the 'Document Properties'	*/
/*  dialog.								*/
/*									*/
/************************************************************************/

typedef struct DocumentStatistics
    {
    int		dsPageCount;
    int		dsParagraphCount;
    int		dsLineCount;
    int		dsWordCount;
    int		dsCharacterCount;
    } DocumentStatistics;

/************************************************************************/
/*									*/
/*  Routine declarations.						*/
/*									*/
/************************************************************************/

extern BufferDocument *	docPlainReadFile( SimpleInputStream *	sis,
				    int *			pMxL,
				    const DocumentGeometry *	dg );

extern BufferDocument *	docRtfReadFile(	SimpleInputStream *	sis,
					int			defAnsigpg );

extern BufferDocument *	docLeafReadFile(	SimpleInputStream *	sis );
extern BufferDocument *	docWordReadFile(	SimpleInputStream *	sis );

extern BufferDocument * docNewFile(	const char *		fontName,
					int			fontSize,
					const DocumentGeometry * dg );

extern int docRtfSaveDocument(	SimpleOutputStream *		sos,
				BufferDocument *		bd,
				const DocumentSelection *	bs,
				int				saveBookmarks );

extern int docRtfSaveSelectionAsLink(
				SimpleOutputStream *		sos,
				const BufferDocument *		bd,
				const DocumentSelection *	ds,
				int				asRef,
				int				asPageref,
				const unsigned char *		fileName,
				int				fileSize,
				const unsigned char *		markName,
				int				markSize );

extern int docHtmlSaveDocument(		SimpleOutputStream *	sos,
					BufferDocument *	bd,
					int			asMimeAggr,
					const char *		mimeBoundary,
					const char *		filename );

extern int docPlainSaveDocument(SimpleOutputStream *		sos,
				BufferDocument *		bd,
				const DocumentSelection *	ds,
				int				fold,
				int				closed );

extern int docRtfSaveRuler(	SimpleOutputStream *		sos,
				const ParagraphProperties *	pp );

extern int docRtfReadRuler(	SimpleInputStream *	sis,
				ParagraphProperties *	pp );

extern int docInflateTextString(	BufferItem *    bi,
					int		by	);

extern void docInitTextLine(		TextLine *	tl );

extern TextLine * docInsertTextLine(	BufferItem *	bi,
					int		line );

extern void docCleanItem(	BufferDocument *	bd,
				BufferItem *		bi );

extern void docFreeItem(	BufferDocument *	bd,
				BufferItem *		bi );

extern void docFreeDocument( BufferDocument *	bd );

extern void docInitDocument(	BufferDocument *	bd	);

extern BufferItem * docInsertItem(	const BufferDocument *	bd,
					BufferItem *		parent,
					int			n,
					ItemLevel		level );

extern BufferItem * docCopyParaItem(	BufferDocument *	bdTo,
					const BufferDocument *	bdFrom,
					int *			fieldMap,
					unsigned int *		pFieldUpd,
					BufferItem *		biCellTo,
					int			n,
					BufferItem *		biParaFrom,
					int			inTable,
					const char *		refFilename );

extern BufferItem * docCopyRowItem(	BufferDocument *	bdTo,
					const BufferDocument *	bdFrom,
					int *			fieldMap,
					unsigned int *		pFieldUpd,
					BufferItem *		biSectTo,
					int			n,
					BufferItem *		biRowFrom,
					int			inTable,
					const char *		refFilename );

extern BufferItem * docCopySectItem(	BufferDocument *	bdTo,
					const BufferDocument *	bdFrom,
					int *			fieldMap,
					unsigned int *		pFieldUpd,
					BufferItem *		biParentTo,
					int			n,
					BufferItem *		biSectFrom,
					const SelectionScope *	ss,
					const char *		refFilename );

extern void docDeleteItems(		BufferDocument *	bd,
					BufferItem *		bi,
					int			first,
					int			count );

extern TextParticule *	docInsertTextParticule(	BufferItem *	bi,
						int		n,
						int		off,
						int		len,
						int		kind,
						TextAttribute	ta	);

extern TextParticule * docCopyParticule(	BufferItem *		bi,
						int			n,
						int			off,
						int			len,
						int			kind,
						const TextParticule *	from );

extern void docDeleteParticules(	BufferItem *	bi,
					int		first,
					int		count );

extern void docDeleteLines(		BufferItem *	bi,
					int		first,
					int		count );

extern void docInitSelectionScope(	SelectionScope *	ss );
extern void docInitTableRectangle(	TableRectangle *	tr );

extern int docSplitGroupItem(	const BufferDocument *	bd,
				BufferItem **		pNewBi,
				BufferItem **		pOldBi,
				BufferItem *		oldBi,
				int			n,
				int			level );

extern BufferItem * docNextParagraph(	BufferItem *	bi );
extern BufferItem * docPrevParagraph(	BufferItem *	bi );

extern BufferItem * docNextSection(	BufferItem *	bi );
extern BufferItem * docPrevSection(	BufferItem *	bi );

extern int docNextPosition(	DocumentPosition *	dp );

extern int docPrevPosition(	DocumentPosition *	dp,
				int			lastOne );

extern int docNextWord(		DocumentPosition *	dp,
				int			lastOne );
extern int docPrevWord(		DocumentPosition *	dp,
				int			lastOne );

extern int docParaBegin(	DocumentPosition *	dp,
				int			lastOne );
extern int docParaEnd(		DocumentPosition *	dp,
				int			lastOne );

extern int docNextLine(		TextParticule **	pTp,
				TextLine **		pTl,
				DocumentPosition *	dp );
extern int docPrevLine(		TextParticule **	pTp,
				TextLine **		pTl,
				DocumentPosition *	dp );

extern int docFirstPosition(	DocumentPosition *	dp,
				BufferItem *		bi );

extern int docLastPosition(	DocumentPosition *	dp,
				BufferItem *		bi );

extern int docFirstDocumentPosition(	DocumentPosition *	dp,
					BufferDocument *	bd );

extern int docLastDocumentPosition(	DocumentPosition *	dp,
					BufferDocument *	bd );

extern void docSetParaSelection( DocumentSelection *	ds,
				BufferItem *		bi,
				int			direction,
				int			start,
				int			length );

extern int docComparePositions(	const DocumentPosition *	dp1,
				const DocumentPosition *	dp2,
				int				mindPart );

extern int docCompareItemPositions(	const BufferItem *	bi1,
					const BufferItem *	bi2 );

extern int docSetDocumentPosition(	DocumentPosition *	dp,
					BufferItem *		bi,
					int			stroff,
					int			lastOne );

extern int docFindParticule(		int *			pPart,
					const BufferItem *	bi,
					int			stroff,
					int			lastOne );

extern int docFindLineOfParticule(	int *			pLine,
					const BufferItem *	bi,
					int			part );

extern void docDeleteItem(		BufferDocument *	bd,
					BufferItem *		bi );

extern void docListItem(	int			indent,
				const BufferItem *	bi );

extern int docCheckItem(	const BufferItem *	bi );

extern void docListParticule(	int			indent,
				const char *		label,
				int			n,
				const BufferItem *	bi,
				const TextParticule *	tp );

extern void docListTextLine(	int			indent,
				const char *		label,
				int			n,
				const BufferItem *	bi,
				const TextLine *	tl );

extern void docInitItem(	BufferItem *		bi,
				BufferItem *		parent,
				const BufferDocument *	bd,
				int			numberInParent,
				int			level,
				int			inExternalItem );

extern const char * docKindStr(		int			kind );
extern const char * docLevelStr(	int			level );
extern const char * docExternalKindStr(	int			level );
extern const char * docFieldKindStr(	int			kind );
extern const char * docAttributeStr(	TextAttribute		ta );

extern TextParticule *	docInsertObject(	BufferItem *		bi,
						InsertedObject **	pIo,
						int			n,
						int			off,
						TextAttribute		ta );

extern void docCleanParticuleObject(	BufferItem *	bi,
					TextParticule *	tp );

extern InsertedObject *	docClaimObject(	int *			pNr,
					BufferItem *		bi );

extern InsertedObject * docClaimObjectCopy( BufferItem *	bi,
					int *			pNr,
					const InsertedObject *	ioFrom );

extern DocumentField * docClaimField(	int *			pNr,
					DocumentFieldList *	dfl );

extern int docSetFieldInst(	DocumentField *		df,
				const unsigned char *	bytes,
				int			size );

extern int docAddToFieldInst(	DocumentField *		df,
				const unsigned char *	bytes,
				int			size );

extern void docInitCellProperties(	CellProperties *	cp );

extern int docDelimitTable(	BufferItem *		paraBi,
				BufferItem **		pSectBi,
				int *			pCol,
				int *			pRow0,
				int *			pRow,
				int *			pRow1 );

extern int docGetBookmarkForPosition(
				const BufferDocument *		bd,
				const DocumentPosition *	dp,
				int *				pPart,
				int *				pEndPart,
				const char **			pMarkName,
				int *				pMarkSize );

extern int docGetHyperlinkForPosition(
				const BufferDocument *		bd,
				const DocumentPosition *	dp,
				int *				pStartPart,
				int *				pEndPart,
				const char **			pFileName,
				int *				pFileSize,
				const char **			pMarkName,
				int *				pMarkSize );

extern DocumentField *	docFindBookmarkField( const DocumentFieldList * dfl,
					const char *		markName,
					int			markSize );

extern int docCopyParticuleData(	BufferDocument *	bdTo,
					const BufferDocument *	bdFrom,
					int *			fieldMap,
					unsigned int *		pFieldUpd,
					BufferItem *		biTo,
					const BufferItem *	biFrom,
					TextParticule *		tpTo,
					const TextParticule *	tpFrom );

extern int docCopyParticules(	BufferDocument *	bdTo,
				const BufferDocument *	bdFrom,
				int *			fieldMap,
				unsigned int *		pFieldUpd,
				BufferItem *		biTo,
				const BufferItem *	biFrom,
				int			partTo,
				int			partFrom,
				int			countFrom,
				int *			pParticulesInserted,
				int *			pCharactersCopied,
				const char *		refFileName );

extern void docInitShape(	DrawingShape *	ds );
extern void docCleanShape(	DrawingShape *	ds );

extern DrawingShape * docClaimShape(	int *			pNr,
					BufferItem *		bi );

extern int docParticuleInField(		BufferItem *	bi,
					int		part );

extern int docSaveParticules(	BufferItem *		bi,
				TextAttribute		ta,
				const unsigned char *	text,
				int			len );

extern int docSaveSpecialParticule(	BufferItem *		bi,
					TextAttribute		ta,
					int			kind );

extern void docLogRectangle(	const char *			label,
				const DocumentRectangle *	dr );

extern void docLogRectangles(	const char *			label1,
				const DocumentRectangle *	dr1,
				const char *			label2,
				const DocumentRectangle *	dr2 );

extern void docInitLayoutPosition(	LayoutPosition *	lp );

extern void docLineSelection(	DocumentSelection *	dsLine,
				const BufferItem *	bi,
				int			line );

extern void docWordSelection(
			DocumentSelection *		dsWord,
			int *				pIsObject,
			const DocumentPosition *	dpAround );

extern void docInitDocumentStatistics(		DocumentStatistics *	ds );

extern void docCollectDocumentStatistics(	DocumentStatistics *	ds,
						const BufferDocument *	bd );

extern int docWhatPageHeader(	ExternalItem **			pHf,
				BufferItem *			sectBi,
				int				page,
				const DocumentProperties *	dp );

extern int docWhatPageFooter(	ExternalItem **			pHf,
				BufferItem *			sectBi,
				int				page,
				const DocumentProperties *	dp );

extern BufferItem * docGetSelectionRoot(
				ExternalItem **			pEi,
				BufferItem **			pBodySectBi,
				BufferDocument *		bd,
				const DocumentSelection *	ds );

extern void docInitParagraphNumber(	ParagraphNumber *	pn );

extern TextParticule *	docInsertParticule(	BufferItem *	bi,
						int		n,
						int		off,
						int		len,
						int		kind );

extern int docCopyFieldRelative(	BufferDocument *	bdTo,
					const BufferDocument *	bdFrom,
					int *			fieldMap,
					unsigned int *		pFieldUpd,
					BufferItem *		biTo,
					const BufferItem *	biFrom,
					TextParticule *		tpTo,
					const TextParticule *	tpFrom,
					const char *		refFileName,
					int			refFileSize );

extern void docFieldRefreshFlags(	BufferItem *		bi,
					const BufferDocument *	bd );

extern void docAdaptBookmarkName(	int *		pLength,
					char *		mark );

extern void docInitDocumentList(	DocumentList *	dl );
extern void docCleanDocumentList(	DocumentList *	dl );

extern int docRecalculateTextLevelFields(	RecalculateFields *	rf,
						BufferItem *		bi );

extern int docRecalculateTextLevelFieldsInExternalItem(
					RecalculateFields *	rf,
					ExternalItem *		ei,
					const BufferItem *	sectBi,
					int			page );

extern int docRedivideStringInParticules(	BufferItem *	bi,
						int		strOff,
						int		strLen,
						int		part,
						int		partsFree,
						TextAttribute	ta );

extern int docBalanceFieldSelection(	int *			pBeginMoved,
					int *			pEndMoved,
					const BufferDocument *	bd,
					DocumentPosition *	dpBegin,
					DocumentPosition *	dpEnd );

extern int docSplitFieldInstructions(	const MemoryBuffer *	mb,
				FieldInstructionsComponent *	fic,
				int				count );

extern int docGetHeaderFooter(	ExternalItem **			pHf,
				BufferItem **			pBodySectBi,
				const DocumentSelection *	ds,
				BufferDocument *		bd,
				int				which );

extern BufferItem * docMakeExternalItem(
				BufferDocument *		bd,
				const SelectionScope *		ss,
				const SectionProperties *	sp );

extern BufferItem * docMakeExternalparagraph(
				BufferDocument *		bd,
				ExternalItem *			ei,
				const BufferItem *		bodyBi,
				TextAttribute			ta,
				int				noteIndex,
				int				inExtIt );

extern BufferItem * docInsertEmptyParagraph(	BufferDocument *	bd,
						BufferItem *		bi,
						TextAttribute		ta );

extern int docIntersectTableRectangle(	TableRectangle *		tr,
					const TableRectangle *		tr1,
					const TableRectangle *		tr2 );

extern void docExpandTableRectangleToWholeTable( TableRectangle *	tr );
extern void docExpandTableRectangleToWholeRows(	TableRectangle *	tr );
extern void docExpandTableRectangleToWholeColumns( TableRectangle *	tr );

extern int docShiftTableRectangleByRows( TableRectangle *	tr,
					int			rows );
extern int docShiftTableRectangleByColumns( TableRectangle *	tr,
					int			columns );

extern int docTableRectangleSelection(	DocumentSelection *	ds,
					BufferItem **		pSelSectBi,
					BufferDocument *	bd,
					const TableRectangle *	tr );

extern int docMergeDocumentFontsIntoList( DocumentFontList *	dflTo,
					BufferDocument *	bdFrom );

extern int docFindBookmarkInDocument(	DocumentSelection *	ds,
					BufferDocument *	bd,
					const char *		markName,
					int			markSize );

extern int docSurroundTextSelectionByField(
				DocumentSelection *		dsField,
				const DocumentSelection *	ds,
				int				fieldNumber,
				const PropertyMask *		taSetMask,
				const TextAttribute *		taSet );

extern void docConstrainSelectionToOneParagraph( int *		pBeginMoved,
					int *			pEndMoved,
					DocumentSelection *	ds );

extern void docUnionParaSelections(	DocumentSelection *		ds,
					const DocumentSelection *	ds1,
					const DocumentSelection *	ds2 );

extern int *	docAllocateFieldMap(	const BufferDocument *	bdFrom );

extern int docParaStringReplace(	int *			pStroffShift,
					BufferItem *		bi,
					int			stroffBegin,
					int			stroffEnd,
					const unsigned char *	addedString,
					int			addedStrlen );

extern int docInsertNote(		DocumentNote **		pDn,
					BufferDocument *	bd,
					BufferItem *		paraBi,
					int			stroff );

extern int docGetNote(			DocumentNote **		pDn,
					const BufferDocument *	bd,
					const BufferItem *	paraBi,
					int			stroff );

extern int docMakeNote(		DocumentNote **			pDn,
				BufferDocument *		bd,
				const DocumentPosition *	dp,
				int				extItKind );

int docGetSelectedNote(		DocumentNote **			pDn,
				BufferDocument *		bd,
				const DocumentSelection *	ds );

extern void docRenumberNotes(		int *			pChanged,
					BufferDocument *	bd );

extern ExternalItem *	docSectionHeaderFooter(	BufferItem *	sectBi,
						int		which );

ExternalItem *	docDocumentNoteSeparator(
					BufferDocument *	bd,
					int			which );

extern int docGetExternalItem(		ExternalItem **		pEi,
					BufferItem **		pBodySectBi,
					BufferDocument *	bd,
					BufferItem *		bi );

extern int docGetRootOfSelectionScope(	ExternalItem **		pEi,
					BufferItem **		pSelRootBi,
					BufferItem **		pBodySectBi,
					BufferDocument *	bd,
					const SelectionScope *	ss );

extern int docGetTopOfColumn(	DocumentPosition *		dp,
				BufferDocument *		bd,
				int				page,
				int				column );

extern int docGetBottomOfColumn(DocumentPosition *		dp,
				BufferDocument *		bd,
				int				page,
				int				column );

extern BufferItem * docInsertRowItem(	BufferDocument *	bd,
					BufferItem *		sectBi,
					int			n,
					const RowProperties *	rp,
					TextAttribute		ta );

extern int docCopyRowColumnAttributes(	BufferItem *		rowBi,
					const BufferItem *	refRowBi );

extern void docSetSelectionScope(	DocumentSelection *	ds,
					const BufferItem *	bi );

extern int docSelectionDifferentRoot(
				const DocumentSelection *	dsOld,
				const BufferItem *		biSet );

extern int docSectionHeaderFooterFirstPage(
				int *				pExists,
				const BufferItem *		bodySectBi,
				int				which,
				const DocumentProperties *	dp );

extern int docSelectWholeSection(	DocumentSelection *	bs,
					BufferDocument *	bd,
					int			direction );

extern int docSelectAll(		DocumentSelection *	ds,
					BufferDocument *	bd );

extern int docNumberOfParagraph(	const BufferItem *	paraBi );

extern BufferItem * docGetParagraphByNumber(	BufferItem *	bi,
						int		n );

extern int docShiftParticuleOffsets(	BufferDocument *	bd,
					BufferItem *		paraBi,
					int			partFrom,
					int			partUpto,
					int			stroffShift );

extern void docShiftNoteReferences(	BufferDocument *	bd,
					int			paraFrom,
					int			stroffFrom,
					int			sectShift,
					int			paraShift,
					int			stroffShift );

extern void docInitNote(		DocumentNote *		dn );

extern void docCleanNote(		BufferDocument *	bd,
					DocumentNote *		dn );

extern int docDeleteNoteOfParticule(	BufferDocument *	bd,
					const BufferItem *	paraBi,
					const TextParticule *	tp );

extern int docCountNotes(		const BufferDocument *	bd );

extern int docGetFirstNoteFromPage(	DocumentNote **		pDn,
					const BufferDocument *	bd,
					int			page,
					int			extItKind );

extern int docGetFirstNoteOnPage(	DocumentNote **		pDn,
					const BufferDocument *	bd,
					int			page,
					int			extItKind );

extern int docGetFirstNoteOfSection(	DocumentNote **		pDn,
					const BufferDocument *	bd,
					int			sect,
					int			extItKind );

extern int docGetFirstNoteOfDocument(	DocumentNote **		pDn,
					const BufferDocument *	bd,
					int			extItKind );

extern void docSetExternalItemKind(	BufferItem *		bi,
					int			extItKind );

extern int docCheckNoteSeparatorItem(	BufferDocument *	bd,
					int			extItKind );

extern int docCountParticulesInField(	const BufferItem *	bi,
					int			part,
					int			partUpto );

#   endif
