/*
 * synaptiks -- a touchpad control tool
 *
 *
 * Copyright (C) 2009 Sebastian Wiesner <basti.wiesner@gmx.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef MOUSEDEVICESMODEL_H
#define MOUSEDEVICESMODEL_H

/**
 * @file
 *
 * Provide a Qt model class to manage mouse devices through HAL.
 */

#include <QtCore/QAbstractListModel>

namespace synaptiks {

    class MouseDevicesModelPrivate;

    /**
     * @brief A Qt model class to manager mouse devices through HAL.
     */
    class MouseDevicesModel: public QAbstractListModel {
        Q_OBJECT
        /**
         * @brief Whether touchpads are ignored or not.
         *
         * @sa setTouchpadsIgnored(bool)
         * @sa touchpadsIgnored()
         */
        Q_PROPERTY(bool touchpadsIgnored
                   READ touchpadsIgnored
                   WRITE setTouchpadsIgnored)

        /**
         * @brief A list of devices checked by the user.
         *
         * @sa setCheckedDevices(const QStringList&)
         * @sa checkedDevices()
         */
        Q_PROPERTY(QStringList checkedDevices
                   READ checkedDevices
                   WRITE setCheckedDevices)

    public:
        /**
         * Create a new object.
         *
         * @param parent same as for QObject(QObject*)
         */
            MouseDevicesModel(QObject *parent=0);

        /**
         * Destroy this object.
         */
        virtual ~MouseDevicesModel();

        /**
         * Return the number of rows in this model.
         *
         * @param parent ignored, required by base class
         * @return the number of rows
         */
        int rowCount(const QModelIndex &parent=QModelIndex()) const;

        /**
         * Returns the item flags for the given @p index.
         *
         * Items of this model are enabled and checkable, but @e not
         * selectable.
         *
         * @param index the item to return flags for
         */
        Qt::ItemFlags flags(const QModelIndex &index) const;

        /**
         * Return the data for the given @p index and the given @p role.
         *
         * @param index points to the data to fetch
         * @param role the role that data is needed for
         * @return the data
         */
        QVariant data(const QModelIndex &index, int role) const;

        /**
         * Sets the data for the given @p index and the given @p role to the
         * given @p value.  If data was changed, @c true is returned and the
         * itemChanged(const QModelIndex&, const QModelIndex&) signal is
         * emitted.
         *
         * The only data, that can be changed for this model, is the check
         * state.
         *
         * @param index the index to set data for
         * @param value the value
         * @param role the role to set data for
         * @return @c true, if data was changed, @c false otherwise
         */
        bool setData(const QModelIndex &index, const QVariant &value,
                     int role);

        /**
         * Are touchpads ignored by this model?
         *
         * @return @c true, if this model ignores touchpads, @c false
         *         otherwise
         */
        bool touchpadsIgnored() const;

        /**
         * Ignore touchpads.  Generally touchpads are mouses too and
         * displayed by this model.  However, there are cases, in which
         * touchpads should not be considered as mouse devices.  For such
         * cases, you can have the model ignore touchpads with this
         * function.
         *
         * Setting this will reset the model.
         *
         * By default, mouse devices are not ignored.
         *
         * @param ignore if @c true, ignore touchpads
         */
        void setTouchpadsIgnored(bool ignore);

        /**
         * Query the checked devices.
         *
         * @return a list of all checked devices
         */
        QStringList checkedDevices() const;

        /**
         * Mark all given @p devices as checked.
         *
         * Emits dataChanged(const QModelIndex&, const QModelIndex&) and
         * checkedDevicesChanged(const QStringList&).
         *
         * @param devices the devices to mark as checked
         */
        void setCheckedDevices(const QStringList &devices);

    Q_SIGNALS:
        /**
         * Emitted, if the the list of checked devices changes.
         *
         * @param devices the new list of checked devices
         */
        void checkedDevicesChanged(const QStringList &devices);

    private:
        Q_DECLARE_PRIVATE(MouseDevicesModel)
        Q_DISABLE_COPY(MouseDevicesModel)

        Q_PRIVATE_SLOT(
            d_func(), void _k_mousePlugged(const QString &udi))
        Q_PRIVATE_SLOT(
            d_func(), void _k_mouseUnplugged(const QString &udi))

        MouseDevicesModelPrivate * const d_ptr;
    };

}

#endif /* MOUSEDEVICESMODEL_H */
