/*
 * synaptiks -- a touchpad control tool
 *
 *
 * Copyright (C) 2009, 2010 Sebastian Wiesner <basti.wiesner@gmx.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef QXDEVICE_H
#define QXDEVICE_H

/**
 * @file
 *
 * Provides the QXDevice class to deal with XInput device properties.
 */

#include "config-synaptiks.h"
#include <QtCore/QList>
#include <QtCore/QSharedPointer>
#include <KDebug>
#include <KLocalizedString>


/**
 * @brief A class for errors resulting from QXDevice.
 */
class QXDeviceError {
public:
    /**
     * @brief Create a new error.
     *
     * @param name the name of the device.
     * @param message a human-readable error message
     */
    QXDeviceError(const QByteArray &name,
                  const QString &message);

    /**
     * @brief Return the name of the device, by which this error was caused.
     *
     * @return the name of the device.
     */
    QByteArray name() const;

    /**
     * @brief Return the error message as given to the constructor.
     *
     * @return the error message
     */
    QString message() const;

    /**
     * @brief Convert this error to a string.
     *
     * @return a human readable representation of this error.
     */
    virtual QString toString() const;

    virtual ~QXDeviceError();

private:
    QByteArray m_name;
    QString m_message;
};


/**
 * @brief A class for errors resulting from property access.
 */
class QXDevicePropertyError: public QXDeviceError {

public:
    /**
     * @brief Create a new error.
     *
     * @param name the device name
     * @param property the property name
     * @param message the error message
     */
    QXDevicePropertyError(const QByteArray &name,
                          const QByteArray &property,
                          const QString &message);

    /**
     * @brief Return the name of the property, by which this error was
     * caused.
     *
     * @return the name of the property.
     */
    QByteArray property() const;

    /**
     * @brief Convert this error to a string.
     *
     * @return a human readable representation of this error.
     */
    virtual QString toString() const;

    virtual ~QXDevicePropertyError();

private:
    QByteArray m_property;
};


/**
 * @brief If a specified property was not defined on a QXDevice.
 */
class QXNoSuchPropertyError: public QXDevicePropertyError {
public:
    /**
     * @brief Create a new error.
     *
     * @param name the name of the device.
     * @param property the affected property
     */
    QXNoSuchPropertyError(const QByteArray &name,
                          const QByteArray &property);
};


class QXDevicePrivate;


/**
 * @brief
 *
 * The QXDevice class provides convenient access to properties of XInput
 * devices.
 *
 * Since Xorg 1.6 XInput supports device-specific properties.  This class
 * provides a simple, Qt-compatible interface to get and set these
 * properties.  Use isSupported() and isPropertyDefined(const QByteArray&)
 * to check, whether the current X server supports properties and knows
 * about specific properties.
 */
class QXDevice {
public:

    /**
     * @brief A device ID.
     */
#ifdef HAVE_XINPUT2
    typedef int ID;
#else /* HAVE_XINPUT2 */
    typedef unsigned long ID;
#endif /* HAVE_XINPUT2 */

    /**
     * A list of devices.
     */
    typedef QList<QSharedPointer<QXDevice> > List;

    /**
     * @brief The property data type.
     *
     * The numeric values correspond to the atom types.
     */
    enum PropertyType {
        /** An integral property. */
        IntegerType,
        /** A float property */
        FloatType
    };

    /**
     * @brief The format of a single property item defined as its size in
     * bytes.
     */
    enum PropertyFormat {
        ByteFormat = 8, /** An item consists of one single byte */
        ShortFormat = 16, /** An item consists of two bytes */
        WordFormat = 32 /** An item consists for four bytes */
    };

    /**
     * @brief Check for XInput support on the server side.
     *
     * This function checks the XInput version reported by the server.  If
     * no display is available or XInput is too old, it returns @c false.
     *
     * @return @c true, if input device properties are supported, @c false
     *         otherwise
     */
    static bool isSupported();

    /**
     * @brief Check, if the given property is defined.
     *
     * This function tries to construct an X11 Atom for the given property.
     * If this fails, the property is most likely not supported by the
     * server.
     *
     * @param name the property name
     * @return @c true, if the property is defined, @c false otherwise
     */
    static bool isPropertyDefined(const QByteArray &name);

    /**
     * @brief List devices which have the given @p property.
     *
     * This function enumerates all server-side devices and returns a list
     * of devices with the given @p property.
     *
     * @param property the property name
     * @return a list of devices
     */
    static List findDevicesWithProperty(const QByteArray &property);

    /**
     * @brief Create a device from the given device @p id.
     *
     * @param name the device name
     * @param id a X11 device id
     */
    QXDevice(const QByteArray &name, const ID id);

    /**
     * Destroy this instance.
     */
    ~QXDevice();

    /**
     * @brief Return the name of this device.
     *
     * @return the device name.
     */
    QByteArray name() const;

    /**
     * @brief Return the raw values of the given property.
     *
     * The returned list is empty, if the actual property type or format do
     * not match @p type and @p format or if this device is invalid.
     *
     * @param name the property name
     * @param type expected property type
     * @param format expected property format
     * @return the list of items as byte arrays
     * @exception NoSuchPropertyError if the given property is not defined
     * @exception QXDeviceError if the property could not be retrieved
     * @sa PropertyType
     * @sa PropertyFormat
     * @sa setProperty(const char*, PropertyType, PropertyFormat,
     *                 const QList<QByteArray>&)
     */
    QList<QByteArray> property(const QByteArray &name, PropertyType type,
                               PropertyFormat format) const;

    /**
     * @brief Set the raw values of the given property.
     *
     * The returned list is empty, if the actual property type or format do
     * not match @p type and @p format or if this device is invalid.
     *
     * @param name the property name
     * @param type the property type
     * @param format the property format
     * @param values the list of items
     * @exception NoSuchPropertyError if the given property is not defined
     * @exception QXDeviceError if the property could not be retrieved
     * @sa PropertyType
     * @sa PropertyFormat
     * @sa property(const char*, PropertyType, PropertyFormat)
     */
    void setProperty(const QByteArray &name, PropertyType type,
                     PropertyFormat format,
                     const QList<QByteArray> &values);

    /**
     * @brief Query the values of a property.
     *
     * The server-side property with the given @p name is queried.  The
     * return value is interpreted according to the template argument.
     *
     * If the property is not defined for this device or cannot be
     * interpreted correctly, an empty list is returned.
     *
     * This method is implemented for these types:
     *
     *  @li @c bool
     *  @li @c char
     *
     * @param name the property
     * @param type the type of the property value
     * @return a list of all values, or empty, if an error occurred
     * @exception NoSuchPropertyError if the given property is not defined
     * @exception QXDeviceError if the property could not be retrieved
     */
    template <class T>
    QList<T> property(const QByteArray &name) const;

    /**
     * @brief Query a single value.
     *
     * This is an overloaded member function.  It simply returns the value
     * at the given @p index (0-based).  If the property contains too few
     * values, the default-constructed value is returned.
     *
     * @param name the property name
     * @param index the index of the value
     * @return the value
     * @exception PropertyIndexError if there is no item at the given
     *            @p index
     * @sa property(const QByteArray &)
     */
    template <class T>
    T property(const QByteArray &name, int index) {
        QList<T> values = this->property<T>(name);
        if (values.length() <= index) {
            kWarning() << "No item at index" << index << "of property"
                       << name;
            throw QXDevicePropertyError(
                this->name(), name, i18nc("device error message",
                                          "no item at index %1", index));
        }
        return values.at(index);
    }

    /**
     * @brief Set the values of a property.
     *
     * The server-side property with the given @p name is set to the given
     * list of @p values.
     *
     * This method is implemented for these types:
     *
     *  @li @c bool
     *  @li @c char
     *
     * @param name the property name
     * @param values the property values
     * @return @c true, if the property was set, @c false otherwise
     */
    template <class T>
    void setProperty(const QByteArray &name, const QList<T> &values);

    /**
     * @brief Set a single value.
     *
     * This is an overloaded member function.  It simply sets the value at
     * the given @p index (0-based).  If the property contains too few
     * values, @c false is returned.
     *
     * @param name the property name
     * @param value the value
     * @param index the index at which to set the value
     * @throw PropertyIndexError if there is no item at the given @p index
     * @sa setProperty(const char*, const QList<T> &)
     */
    template <class T>
    void setProperty(const QByteArray &name, T value, int index=0) {
        QList<T> values = this->property<T>(name);
        if (values.length() <= index) {
            kWarning() << "No item at index" << index << "of property"
                       << name;
            throw QXDevicePropertyError(
                this->name(), name, i18nc("device error message",
                                          "no item at index %1", index));
        }
        values[index] = value;
        return this->setProperty(name, values);
    }

    /**
     * @brief Check, if a property exists.
     *
     * @param name the property name
     * @return @c true, if the property is defined for this device, @c
     *         false otherwise.
     */
    bool hasProperty(const QByteArray &name) const;

private:
    Q_DISABLE_COPY(QXDevice)
    Q_DECLARE_PRIVATE(QXDevice)

    QXDevicePrivate *d_ptr;
};
#endif /* QXDEVICE_H */
