// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

/*
  convert.cc
  Contains the main() function for the starconvert utility and all the
  functions allowing it to convert a star data file to StarPlot format.
*/

#include "convert.h"

// defined in names.cc:
extern void get_names(const char *, namedata, StringList *, StringList *);

Star parse_star_record(const char *, parsedata);
void get_coordinates(const char *, coordinates, StringList *);
void get_mag_distance(const char *, const char *, characteristics,
		      double *, double *);
void check_for_double_stars(const char *, systems, Star *, Star *);


void parse_star_file(istream & infile, ostream & outfile, parsedata format,
		     bool using_stdout)
{
  int lineno = 0;
  bool sun_found = false, moredata = true;
  char record[1000];
  Star tempstar, currentstar = BADSTAR, previousstar;

  // boring legal disclaimer:
  outfile << "This StarPlot data file was automatically generated by "
          << "starconvert v. " << STARPLOT_VERSION << "." << endl
	  << "If this file was derived from a copyrighted source, you "
	  << "may not redistribute" << endl << "it without the permission of "
	  << "the original author." << endl << endl 
	  << "If you obtained this data file by installing one of the data "
	  << "packages available" << endl << "on the StarPlot web page, you "
	  << "should see the file copyright notice in that" << endl
	  << "package for more information." << endl << endl;

  for (unsigned int i = 0; i < 1000; i++)
    record[i] = 0;

  do {
    moredata = infile.getline(record, 999, '\n');
    // $ ; and , have special meanings to StarPlot, so purge them:
    for (unsigned int i = 0; i < strlen(record); i++)
      if (record[i] == '$' || record[i] == ';' || record[i] == ',')
        record[i] = ' ';

    tempstar = parse_star_record(record, format);
    if (strcmp(tempstar.GetStarNames()[0], BADSTARNAME) != 0 || !moredata) {
      previousstar = currentstar;
      currentstar = tempstar;
    }
    else continue;

    if (strcmp(previousstar.GetStarNames()[0], BADSTARNAME) != 0) {
      Rules r;
      r.CelestialCoords = true;
      check_for_double_stars(record, format.Systems, 
			     &previousstar, &currentstar);
      StringList starinfo = previousstar.GetInfo(r, false, ',');
      
      // write star information to output
      outfile << "$ " << starinfo[1];
      if (starinfo.size() >= 11) {
	for (unsigned int i = 10; i < starinfo.size(); i++)
	  outfile << ", " << starinfo[i];
      }
      outfile << ";" << endl << "  ";
      
      outfile << starinfo[2] << "; " << starinfo[3] << "; " << starinfo[4] 
	      << "; 0;" << endl << "  " << starinfo[5] << "; ";
      outfile << ((strcasecmp(starinfo[1], "Sun") == 0) 
		  ? (sun_found = true, "4.85") : starinfo[6]);
      outfile << "; " << starinfo[9] << "; " << starinfo[7] << ";" << endl;
      
      if (starinfo.strlen(8))
	outfile << "  " << starinfo[8] << endl;
      outfile << endl;
    }
    for (unsigned int i = 0; i < 1000; i++) record[i] = 0;

    lineno++;
    if (!using_stdout && !(lineno % 1000))
      cout << lineno << " records converted." << endl;

  } while (moredata) ;

  // if the Sun is not in the data file, append it at the end!
  if (!sun_found)
    outfile << "$ Sun, Sol;" << endl << "  ; ; 0; 0;" << endl 
    	    << "  G2 V; 4.85; 0; ;" << endl << endl;
  return;
}


// parse_star_record(): converts the current record to a Star

Star parse_star_record(const char *record, parsedata format)
{
  if (isempty(record)) return BADSTAR;

  Star result;
  StringList starnames = StringList(), starcoord = StringList();
  StringList starcomments = StringList(), starmembership = StringList();
  double starmag, stardist;
  char *spectrum = 0;

  // find the first non-empty spectral class datum out of those specified
  //  in the specification file
  unsigned int i = 0;
  do {
    delete [] spectrum; // it's safe to delete a NULL pointer
    spectrum = new char[format.Charact.specclass_len[i] + 1];
    strncpy(spectrum, record + format.Charact.specclass_start[i],
	    format.Charact.specclass_len[i]);
    spectrum[format.Charact.specclass_len[i]] = 0;
    stripspace(spectrum);
    i++;
  } while (i < format.Charact.specclass_len.size() && !spectrum[0]) ;
  if (spectrum && !spectrum[0]) { delete [] spectrum; spectrum = 0; }

  // convert comments (if any) to a StringList
  if (format.Comments.comments_len > 0) {
    char * commentstring = new char[format.Comments.comments_len + 1];
    strncpy(commentstring, record + format.Comments.comments_start,
	    format.Comments.comments_len);
    commentstring[format.Comments.comments_len] = 0;
    starcomments = StringList(commentstring, ' ');
    starcomments.stripspace();
    for (unsigned int i = 0; i < starcomments.size(); i++)
      if (!starcomments[i][0]) {
	starcomments.remove(i);
	i--;
      }
    delete [] commentstring;
  }

  get_coordinates(record, format.Coord, &starcoord);
  get_mag_distance(record, spectrum, format.Charact, &stardist, &starmag);
  get_names(record, format.Names, &starnames, &starcomments);

  if (starnames.size() && (strcasecmp(starnames[0], "Sun") == 0))
    stardist = 0.0;

  if (stardist >= 0.0) {
    // see note regarding <sstream> in convert.h
    ostringstream starstring;

    for (unsigned int i = 0; i + 1 < starnames.size(); i++)
      starstring << starnames[i] << ", ";
    if (starnames.size())
      starstring << starnames[starnames.size() - 1];
    else
      starstring << "<no name>";
    starstring << "; ";

    for (unsigned int i = 0; i < 6; i++)
      starstring << starcoord[i] << (((i + 1) % 3) ? ", " : "; ");
    starstring << stardist << "; 0; " << (spectrum ? spectrum : "") << "; "
	       << starmag << "; " << "0; ; ";

    if (starcomments.size())
      for (unsigned int i = 0; i < starcomments.size(); i++)
	starstring << starcomments[i] << " ";

    starstring << '\0';
    result = Star(starstring.str().c_str(),
		  false /* no fast conversion */,
		  false /* no name translation */);
  }

  else {
    cerr << "*** Cannot obtain distance to star " << starnames[0] << "; "
	 << "leaving it out." << endl;
    result = BADSTAR;
  }

  if (spectrum) delete [] spectrum;
  return result;
}


void get_coordinates(const char * record, coordinates c, 
		     StringList * starcoords)
{
  char *tempstrings[NUM_COORD_OPTIONS];
  for (unsigned int i = 0; i < NUM_COORD_OPTIONS; i++) {
    if (c.coord_len[i] > 0) {
      tempstrings[i] = new char[c.coord_len[i] + 1];
      strncpy(tempstrings[i], record + c.coord_start[i], c.coord_len[i]);
      tempstrings[i][c.coord_len[i]] = 0;
      stripspace(tempstrings[i]);
    }
    else {
      tempstrings[i] = new char[1];
      tempstrings[i][0] = 0;
    }
  }

  char *coordstrings[6];
  unsigned int j = 0, len;
  for (unsigned int i = 0; i < NUM_COORD_OPTIONS; (i != 2) ? i++ : i += 3) {
    len = strlen(tempstrings[i]) + 1;
    coordstrings[j] = new char[(len > 5) ? len : 5];
    strcpy(coordstrings[j], tempstrings[i]);
    if (++j == 3) j++;
  }
  len = strlen(tempstrings[3]) + strlen(tempstrings[4]) + 1;
  coordstrings[3] = new char[(len > 5) ? len : 5];
  strcpy(coordstrings[3], tempstrings[3]);
  strcat(coordstrings[3], tempstrings[4]);

  for (unsigned int i = 0; i < NUM_COORD_OPTIONS; i++)
    delete [] tempstrings[i];

  double theta, phi;
  SolidAngle omega;

  phi = RAStringsToRadians(coordstrings[0], coordstrings[1],
			   coordstrings[2], c.isCelestial);
  theta = DecStringsToRadians(coordstrings[3], coordstrings[4],
			      coordstrings[5]);
  omega = SolidAngle(phi, theta);

  if (!c.isCelestial) omega = omega.toCelestial();
  RadiansToRAStrings(omega.getPhi(), coordstrings[0],
		     coordstrings[1], coordstrings[2]);
  RadiansToDecStrings(omega.getTheta(), coordstrings[3],
		      coordstrings[4], coordstrings[5]);

  for (unsigned int i = 0; i < 6; i++) {
    starcoords->append(coordstrings[i]);
    delete [] coordstrings[i];
  }
  return;
}


// get_mag_distance(): This function obtains the distance and magnitude of
//  the star.  It goes through the distance/magnitude formatting pairs 
//  specified in the config file, and uses the first pair which gives an 
//  acceptable result.

void get_mag_distance(const char * record, const char *spectrum,
		      characteristics c, double *dist, double *mag)
{
  double tempmag, tempdist;

  for (unsigned int i = 0; i < c.distarray.size(); i++) {
    if (c.magarray[i].len <= 0) continue;
    char *endptr, *magstring = new char[c.magarray[i].len + 1];
    strncpy(magstring, record + c.magarray[i].start, c.magarray[i].len);
    magstring[c.magarray[i].len] = 0;
    stripspace(magstring);
    tempmag = strtod(magstring, &endptr);
    if (!magstring[0] || (*endptr != 0 && *endptr != '.')) {
      delete [] magstring;
      continue;
    }
    delete [] magstring;

    if (c.distarray[i].type == SPECCLASS) {
      if (!spectrum || !spectrum[0] || c.magarray[i].type == ABSOLUTE) 
	continue;
      double absmag = findabsmag(spectrum);
      if (absmag < -25 || absmag > 25) continue;
      
      // Technically this should also account for the effect of 
      //  interstellar dust on the visual magnitude...
      tempdist = 10.0 * LY_PER_PC * pow(10.0, (tempmag - absmag) / 5.0);

      // should NOT be using spectral class distance estimate for nearby stars:
      if (tempdist < 20 /* LY */) continue;

      *dist = sigdigits(tempdist, 3);
      *mag = absmag;
      return;
    }

    else if (c.distarray[i].len > 0) {
      char *endptr, *diststring = new char[c.distarray[i].len + 1];
      strncpy(diststring, record + c.distarray[i].start, c.distarray[i].len);
      diststring[c.distarray[i].len] = 0;
      stripspace(diststring);
      tempdist = strtod(diststring, &endptr);
      if (!diststring[0] || tempdist < 0 
	  || (*endptr != 0 && *endptr != '.')) {
	delete [] diststring;
	continue;
      }
      delete [] diststring;

      char *errstring;
      double parallaxerr = 0.0;
      if ((c.distarray[i].type == ARCSEC || c.distarray[i].type == MILLIARCSEC)
	  && c.distarray[i].errorlen > 0) {
	errstring = new char[c.distarray[i].errorlen + 1];
	strncpy(errstring, record + c.distarray[i].errorstart,
		c.distarray[i].errorlen);
	errstring[c.distarray[i].errorlen] = 0;
	stripspace(errstring);
	parallaxerr = strtod(errstring, &endptr);
	if (!errstring[0] || parallaxerr < 0 
	    || (*endptr != 0 && *endptr != '.')) {
	  delete [] errstring;
	  continue;
	}
	delete [] errstring;
      }

      switch (c.distarray[i].type) {
      case MILLIARCSEC: tempdist /= 1000.0; parallaxerr /= 1000.0;
      case ARCSEC:
	{
	  // if parallax is < min. acceptable parallax, go to next rule
	  if (tempdist < c.distarray[i].minparallax) break;
	  // if relative error is > max. acceptable error, go to next rule
	  else if (c.distarray[i].errorlen > 0 
		   && parallaxerr > c.distarray[i].maxerror * tempdist) break;
	  else tempdist = 1.0 / tempdist;
	}
      case PC:          tempdist *= LY_PER_PC;
      case LY:
	{
	  *dist = sigdigits(tempdist, 3);
	  if (c.magarray[i].type == ABSOLUTE)
	    *mag = tempmag;
	  else {
	    tempmag -= (5.0 * (log10(tempdist / LY_PER_PC) - 1.0));
	    *mag = roundoff(tempmag, 1);
	  }
	  return;
	}
      case SPECCLASS: break; // do nothing
      } // end switch statement
    } 
  } // end for loop
  
  // if we haven't returned yet, something is wrong; flag the values.
  *mag = *dist = -9999;
  return;
}


// Check for double systems.  
// Not yet implemented.

void check_for_double_stars(const char *record, systems s,
			    Star *previous, Star *current)
{
  double distdiff 
    = (previous->GetStarXYZ() - current->GetStarXYZ()).magnitude();
  if (distdiff < 0.2 /* light-years */) {
    if (previous->GetStarMembership().size() == 0) {
	  




    }
    else {
	
    }
  }
  else {
    
  }
}


void printusage()
{
  cout << "Usage: " << PROGNAME << " specfile infile [outfile]" << endl
       << "Uses specifications in `specfile' to convert `infile' to StarPlot "
       << "data format" << endl << "and writes the results to `outfile' if "
       << "specified, or standard output if not." << endl << "The first two "
       << "arguments are required.  Exactly ONE may be replaced with" << endl
       << "a dash `-' for reading from standard input." << endl << endl;
  cout << "starconvert: version " << STARPLOT_VERSION
       << " (C) 2000, 2001 Kevin B. McCarty" << endl
       << "starconvert comes with ABSOLUTELY NO WARRANTY; for details, see"
       << endl << "the file " << DOCDIR << "/COPYING." << endl;
  return;
}


// main(): This doesn't do much besides error checking, and calling the two
//  top-level parsing functions (one for the specification file, and one for
//  the data file).

int main(int argc, char **argv)
{
  if (argc != 3 && argc != 4) {
    printusage();
    return 0;
  }

  parsedata specdata;
  ifstream infile, specfile;
  ofstream outfile;

  if (strcmp(argv[1], "-") == 0) {
    if (strcmp(argv[2], "-") == 0) {
      printusage();
      return 0;
    }
    else {
      if (! cin.good()) {
	cerr << "Can't read specifications from stdin - not a tty?" << endl;
	return EXIT_FAILURE;
      }
      parse_config_file(cin, &specdata);
    }
  }
  else {
    specfile.open(argv[1]);
    if (! specfile.good()) {
      cerr << "Can't open specification file `" << argv[1]
	   << "' for input; exiting." << endl;
      return EXIT_FAILURE;
    }
    parse_config_file(specfile, &specdata);
    specfile.close();
  }

  if (argc == 4) {
    outfile.open(argv[3]);
    if (! outfile.good()) {
      cerr << "Can't open output file `" << argv[3] 
	   << "' for output; exiting." << endl;
      return EXIT_FAILURE;
    }
    else
      cout << "Attempting to convert " 
           << ((strcmp(argv[2], "-") == 0) ? "standard input" : argv[2])
	   << " to StarPlot format..." << endl;
  }

  if (strcmp(argv[2], "-") == 0) {
    if (! cin.good()) {
      cerr << "Can't read star data from stdin - not a tty?" << endl;
      return 1;
    }
    if (argc == 4) parse_star_file(cin, outfile, specdata, false);
    else           parse_star_file(cin, cout, specdata, true);
  }

  else {
    infile.open(argv[2]);
    if (! infile.good()) {
      cerr << "Can't open star data file `" << argv[2]
	   << "' for input; exiting." << endl;
      return 1;
    }
    if (argc == 4) parse_star_file(infile, outfile, specdata, false);
    else           parse_star_file(infile, cout, specdata, true);
    infile.close();
  }

  if (argc == 4) {
    outfile.close();
    cout << "Done." << endl;
  }

  return 0;
}

