// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


// stringops.h
//
// This header includes function declarations for working with strings, plus
//  the StringList class, which provides for automatic memory management of
//  sets of strings.

#ifndef _STRINGOPS_H
#define _STRINGOPS_H

#include <cctype>
#include <cstring>
#include <string>
#include <vector>
#include <cstdio>
#include <cstdlib>

#include "mathops.h" // to make sure that M_PI is defined

#ifdef EXTENDED_ASCII
# define DEGREE_SYMBOL    "\xB0"
# define COPYRIGHT_SYMBOL "\xA9"
#else
# define DEGREE_SYMBOL    "d"
# define COPYRIGHT_SYMBOL "(C)"
#endif

// Old compilers (egcs-2.91, for example) define tolower, toupper as
// macros, which messes everything up.  So fix this:

#ifdef tolower
#undef tolower
#define NO_TOLOWER
namespace std {
  extern char tolower(char);
}
#endif

#ifdef toupper
#undef toupper
#define NO_TOUPPER
namespace std {
  extern char toupper(char);
}
#endif

// Some convenient string functions are ISO C99 or other extensions to the ANSI
//  standard; if they doesn't exist, we have to define them...

// snprintf() seems not to be defined for older libc versions.
// Assume not defined if gcc < 2.95.x, unless specifically told otherwise.
// (I really will get around to making a configure script one year...)
#if (! defined NO_SNPRINTF) && (defined __GNUC__) && (__GNUC__ < 3) \
	&& ((! defined __GNUC_MINOR__) || __GNUC_MINOR__ < 95) && \
	(! defined WITH_SNPRINTF)
#define NO_SNPRINTF
#endif

#ifdef NO_SNPRINTF
#include <cstdarg>
extern int snprintf(char * string, size_t size, const char * format, ...);
#endif // #ifdef NO_SNPRINTF

#ifdef NO_STRCASECMP
extern int strcasecmp(const char * s1, const char * s2);
extern int strncasecmp(const char * s1, const char * s2, size_t n);
#endif // #ifdef NO_STRCASECMP

using namespace std;

// Some useful string functions

extern bool isempty(const char *);
extern char * stripspace(char *);
inline void stripspace(string &s)

#define WHITESPACE " \t\v\n"
{
  s.replace(0, s.find_first_not_of(WHITESPACE), "");
  s.replace(s.find_last_not_of(WHITESPACE) + 1, string::npos, "");
}

inline void uppercase(string &s)
{ for (unsigned int i = 0; i < s.size(); i++) s[i] = std::toupper(s[i]); }

// functions to convert text strings containing RA and Dec in h,m,s and
//  +/-d,m,s format into radians

inline double RAStringsToRadians(const char *hrs, const char *min,
				 const char *sec, bool celestial_coords = true)
{ 
  double value = M_PI_12 * (myatof(hrs) + myatof(min) / 60.0 
			      + myatof(sec) / 3600.0);
  if (!celestial_coords) value /= 15.0;
  return value;
}

inline double DecStringsToRadians(const char *deg, const char *min,
				  const char *sec)
{
  double decmin = myatof(min) + myatof(sec) / 60.0;
  double value;
  
  char temp[10];
  strncpy(temp, deg, 9); temp[9] = 0; stripspace(temp);

  if (temp[0] == '-') // take care of the negative case
    decmin *= -1;
  value = M_PI_180 * (myatof(temp) + decmin / 60.0);
  return value;
}

// functions to do the opposite
extern void RadiansToRAStrings(double phi, char * hrs, char * min, char * sec,
		               bool celestial_coords = true);
extern void RadiansToDecStrings(double theta,
				char * deg, char * min, char * sec);
extern void RadiansToRAString(double phi, char * output,
		              bool celestial_coords = true,
			      bool punctuation = true, char separator = ' ');
extern void RadiansToDecString(double theta, char * output,
			       bool punctuation = true, char separator = ' ');


class StringList {
 private:
  vector<string> stringlist;

 public:
  StringList() { stringlist.reserve(15); }
  StringList(const char *cstring) // Create a StringList with one entry
    { stringlist.reserve(15); stringlist.push_back(cstring); }
  StringList(const StringList &s) { stringlist = s.stringlist; }
  StringList(const char *cstring, char tokenizer);

  StringList & operator = (const StringList &s) 
    { stringlist = s.stringlist; return *this; }
  const char * operator [] (unsigned int posn) const
    { return (posn < stringlist.size()) ? stringlist[posn].c_str() : ""; }

  string flatten(char spacer);
  bool insert(const char *cstring, unsigned int posn);
  bool set(const char *cstring, unsigned int posn);
  bool remove(unsigned int posn);
  void append(const char *cstring)
    { 
      string temp = string(cstring);
      stringlist.push_back(temp);
    }
  void append(const string &s) { append(s.c_str()); }

  bool insert(const string& s, unsigned int posn)
    { return insert(s.c_str(), posn); }
  bool set(const string& s, unsigned int posn)
    { return set(s.c_str(), posn); }
  void append(string s)
    { stringlist.push_back(s); }
  
  StringList stripspace(void);
  StringList touppercase(void);
  StringList tolowercase(void);

  inline unsigned int size() const { return stringlist.size(); }
  inline unsigned int strlen(const unsigned int posn) const
    { 
      return (posn < stringlist.size()) ? 
	std::strlen(stringlist[posn].c_str()) : 0; 
    }
} ;

#endif // #define _STRINGOPS_H
