// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


// specclass.h
//
// The SpecClass class is a representation of stars' spectral type.

#ifndef _SPEC_CLASS_H
#define _SPEC_CLASS_H

#include <cctype>
#include <cstdlib>
#include <cstring>
#include <cmath>
using namespace std;

#include "stringops.h" // for stripspace()

// Colors, in 0xRRGGBB format, in which to plot stars and map decals

const unsigned long RED_MASK = 0xFF0000,
                    GREEN_MASK = 0x00FF00,
                    BLUE_MASK = 0x0000FF;

#define RED(color)   (((color) & RED_MASK) >> 16)
#define GREEN(color) (((color) & GREEN_MASK) >> 8)
#define BLUE(color)  ((color) & BLUE_MASK)
		    
const unsigned long WOLF_RAYET_COLOR = 0x2000A0,
                    O_COLOR = 0x0000FF,
                    B_COLOR = 0x00C0FF,
                    A_COLOR = 0x80E0E0,
                    F_COLOR = 0xF0F0F0,
                    G_COLOR = 0xF0F000,
                    K_COLOR = 0xF0A000,
                    M_COLOR = 0xFF2000,
                    D_COLOR = 0x808080,
                    NON_STELLAR_COLOR = 0xFF00FF,
                    DEFAULT_COLOR = 0x00FF00;

const unsigned long POSITIVE = 0x808080,
                    NEGATIVE = 0x0000FF,
                    BACKGROUND = 0x000000,
                    TEXT_COLOR = 0x00FF00,
                    LABEL_COLOR = 0xC0C0C0,
                    BOLD_COLOR = 0xFFFFFF,
                    ERROR_COLOR = 0xFF4040;


class SpecClass {
 private:
  char * specstring; // i.e. "M3.5 V"
  char   major;  // the "M"
  double minor;  // the "3.5"
  double MKtype; // the "V"

 public:
  SpecClass(const char * spectrum);
  SpecClass() { major = 0; minor = MKtype = 0.0; specstring = 0; }
  SpecClass(const SpecClass &sc);
  SpecClass & operator = (const SpecClass &sc);
  ~SpecClass() { delete [] specstring; }

  void initialize();           // calculate subtype and MK luminosity class
  double temperature() const;          // estimate the star's temperature in K
  double diameter(double mag) const;   // star diameter in light-years
  unsigned long color() const;         // output a nice color in 0xRRGGBB form
  const char * print() const { return specstring; } // print the fields

  inline char getCoarseType() const { return major;  }
  inline double getSubType() const  { return minor;  }
  inline double getMKtype() const   { return MKtype; }
} ;

#endif // #define _SPEC_CLASS_H
