# ifndef _RHEO_CSR_H
# define _RHEO_CSR_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/vec.h"
#include "rheolef/asr.h"
#include "rheolef/vector_of_iterator.h"
#include "rheolef/scatter_message.h"
namespace rheolef {

// -------------------------------------------------------------
// the sequential representation
// -------------------------------------------------------------
template<class T>
class csr_seq_rep : public vector_of_iterator<std::pair<typename std::vector<T>::size_type,T> > {
public:
    typedef typename std::vector<T>::size_type                            size_type;
    typedef T                                                             element_type;
    typedef sequential                                                    memory_type;
    typedef typename std::pair<size_type,T>                               pair_type;
    typedef typename vector_of_iterator<pair_type>::iterator              iterator;
    typedef typename vector_of_iterator<pair_type>::const_iterator        const_iterator;
    typedef typename vector_of_iterator<pair_type>::difference_type       difference_type;
    typedef typename vector_of_iterator<pair_type>::value_type            data_iterator;
    typedef typename vector_of_iterator<pair_type>::const_value_type      const_data_iterator;

    csr_seq_rep (size_type loc_nrow1 = 0, size_type loc_ncol1 = 0, size_type loc_nnz1 = 0);
    void resize (size_type loc_nrow1 = 0, size_type loc_ncol1 = 0, size_type loc_nnz1 = 0);
    csr_seq_rep (const distributor& row_ownership, const distributor& col_ownership, size_type nnz1 = 0);
    void resize (const distributor& row_ownership, const distributor& col_ownership, size_type nnz1 = 0);
    csr_seq_rep (const csr_seq_rep<T>& a);
    explicit csr_seq_rep (const asr_seq_rep<T>& a);
    void to_asr (asr_seq_rep<T>& b) const;

    const distributor& row_ownership() const { return _row_ownership; }
    const distributor& col_ownership() const { return _col_ownership; }
    const_iterator begin() const { return vector_of_iterator<pair_type>::begin(); }
    const_iterator end()   const { return vector_of_iterator<pair_type>::end(); }
          iterator begin()       { return vector_of_iterator<pair_type>::begin(); }
          iterator end()         { return vector_of_iterator<pair_type>::end(); }
    size_type nrow() const       { return vector_of_iterator<pair_type>::size()-1; }
    size_type ncol() const { return _col_ownership.size(); }
    size_type nnz() const { return _data.size(); }
    size_type dis_nrow() const { return nrow(); }
    size_type dis_ncol() const { return ncol(); }
    size_type dis_nnz() const { return nnz(); }
    bool is_symmetric() const { return _is_symmetric; }
    void set_symmetry (bool is_symm) { _is_symmetric = is_symm; }
    size_type pattern_dimension() const { return _pattern_dimension; }
    void set_pattern_dimension(size_type dim) { _pattern_dimension = dim; }
    size_type row_first_index () const { return 0; }
    size_type row_last_index () const { return nrow(); }
    size_type col_first_index () const { return 0; }
    size_type col_last_index () const { return ncol(); }
    size_type ext_nnz() const { return 0; }
    idiststream& get (idiststream&);
    odiststream& put (odiststream&, size_type istart = 0) const;
    void dump (const std::string& name, size_type istart = 0) const;
    void mult       (const vec<T,sequential>& x, vec<T,sequential>& y) const;
    void trans_mult (const vec<T,sequential>& x, vec<T,sequential>& y) const;
    csr_seq_rep<T>& operator*= (const T& lambda);
    template <class BinaryOp>
    void assign_add (const csr_seq_rep<T>& a, const csr_seq_rep<T>& b, BinaryOp binop);
    void build_transpose (csr_seq_rep<T>& b) const;
//protected:
    distributor          		                          _row_ownership;
    distributor          		                          _col_ownership;
    std::vector<std::pair<typename std::vector<T>::size_type,T> > _data;
    bool							  _is_symmetric;
    size_type							  _pattern_dimension; // e.g. FEM 3d-pattern
};
// -------------------------------------------------------------
// the distributed representation
// -------------------------------------------------------------
#ifdef _RHEOLEF_HAVE_MPI
template<class T>
class csr_mpi_rep : public csr_seq_rep<T> {
public:
    typedef typename csr_seq_rep<T>::size_type           size_type;
    typedef typename csr_seq_rep<T>::element_type        element_type;
    typedef distributed                                  memory_type;
    typedef typename csr_seq_rep<T>::iterator            iterator;
    typedef typename csr_seq_rep<T>::const_iterator      const_iterator;
    typedef typename csr_seq_rep<T>::data_iterator       data_iterator;
    typedef typename csr_seq_rep<T>::const_data_iterator const_data_iterator;

    csr_mpi_rep ();
    csr_mpi_rep (const csr_mpi_rep<T>& a);
    explicit csr_mpi_rep (const asr_mpi_rep<T>& a);
    void build_from_asr (const asr_mpi_rep<T>& a);
    void to_asr (asr_mpi_rep<T>& b) const;
    void resize (const distributor& row_ownership, const distributor& col_ownership, size_type nnz1 = 0);

    const distributor& row_ownership() const { return csr_seq_rep<T>::_row_ownership; }
    const distributor& col_ownership() const { return csr_seq_rep<T>::_col_ownership; }
    const communicator& comm() const { return row_ownership().comm(); }
    const_iterator begin() const { return csr_seq_rep<T>::begin(); }
    const_iterator end()   const { return csr_seq_rep<T>::end(); }
          iterator begin()       { return csr_seq_rep<T>::begin(); }
          iterator end()         { return csr_seq_rep<T>::end(); }
    size_type      ext_nnz()  const { return _ext.nnz(); }
    const_iterator ext_begin() const { return _ext.begin(); }
    const_iterator ext_end()   const { return _ext.end(); }
    size_type nrow() const { return csr_seq_rep<T>::nrow(); }
    size_type ncol() const { return csr_seq_rep<T>::ncol(); }
    size_type nnz() const { return csr_seq_rep<T>::nnz(); }
    size_type dis_nrow() const { return row_ownership().dis_size(); }
    size_type dis_ncol() const { return col_ownership().dis_size(); }
    size_type dis_nnz() const { return _dis_nnz; }
    bool is_symmetric() const { return csr_seq_rep<T>::is_symmetric(); }
    void set_symmetry (bool is_symm) { csr_seq_rep<T>::set_symmetry(is_symm); }
    size_type pattern_dimension() const { return csr_seq_rep<T>::pattern_dimension(); }
    void set_pattern_dimension(size_type dim) { csr_seq_rep<T>::set_pattern_dimension(dim); }
    size_type row_first_index () const { return row_ownership().first_index(); }
    size_type row_last_index ()  const { return row_ownership().last_index(); }
    size_type col_first_index () const { return col_ownership().first_index(); }
    size_type col_last_index ()  const { return col_ownership().last_index(); }
    size_type jext2dis_j (size_type jext)  const;
    idiststream& get (idiststream&);
    odiststream& put (odiststream&) const;
    void dump (const std::string& name) const;
    void mult       (const vec<T,distributed>& x, vec<T,distributed>& y) const;
    void trans_mult (const vec<T,distributed>& x, vec<T,distributed>& y) const;
    csr_mpi_rep<T>& operator*= (const T& lambda);
    template <class BinaryOp>
    void assign_add (const csr_mpi_rep<T>& a, const csr_mpi_rep<T>& b, BinaryOp binop);
    void build_transpose (csr_mpi_rep<T>& b) const;
protected:
    // diagonal part is the basic csr_seq_rep type
    // extra-diagonal blocs are sequential csr also:
    csr_seq_rep<T>       		_ext;
    std::vector<size_type>    		_jext2dis_j;
    size_type            		_dis_nnz;
    
    // A*x internal stuff: scatter and buffer:
    mutable scatter_message<std::vector<T> > _from;
    mutable scatter_message<std::vector<T> > _to;
    mutable std::vector<T>    		     _buffer;
};
template<class T>
inline
typename csr_mpi_rep<T>::size_type
csr_mpi_rep<T>::jext2dis_j (size_type jext)  const
{
  check_macro (jext < _jext2dis_j.size(), "jext2dis_j: jext="<<jext<<" is out of range [0:"<<_jext2dis_j.size()<<"[");
  return _jext2dis_j [jext];
}
#endif // _RHEOLEF_HAVE_MPI

// these classes are used for allocator from the std::initializer_list
template <class T, class M> class csr_concat_value;
template <class T, class M> class csr_concat_line;

// -------------------------------------------------------------
// the basic class with a smart pointer to representation
// the user-level class with memory-model parameter
// -------------------------------------------------------------
/*Class:csr
NAME:  @code{csr} - compressed sparse row matrix (@PACKAGE@-@VERSION@)
SYNOPSYS:       
  Distributed compressed sparse matrix container stored row by row.
DESCRIPTION:
  Sparse matrix are compressed by rows. In distributed environment, the
  distribution follows the row distributor (see @ref{distributor class}).
ALGEBRA:
  Adding or substracting two matrices writes @code{a+b} and @code{a-b}, respectively,
  and multiplying a matrix by a scalar writes @code{lambda*x}.
  Thus, any linear combination of sparse matrices is available.
 
  Matrix-vector product writes @code{a*x} where @code{x} is a vector (see @ref{vec class}).
LIMITATIONS:
  Some basic linear algebra is still under development:
  @code{a.trans_mult(x)} matrix transpose vector product,
  @code{trans(a)} matrix transpose,
  @code{a*b} matrix product.
AUTHORS: Pierre.Saramito@imag.fr
DATE:   10 february 1999
METHODS: @csr
End:
*/
template <class T, class M = rheo_default_memory_model>
class csr {
public:
    typedef M memory_type;
};
//<verbatim:
template<class T>
class csr<T,sequential> : public smart_pointer<csr_seq_rep<T> > {
public:

// typedefs:

    typedef csr_seq_rep<T>                    rep;
    typedef smart_pointer<rep>                base;
    typedef typename rep::memory_type         memory_type;
    typedef typename rep::size_type           size_type;
    typedef typename rep::element_type        element_type;
    typedef typename rep::iterator            iterator;
    typedef typename rep::const_iterator      const_iterator;
    typedef typename rep::data_iterator       data_iterator;
    typedef typename rep::const_data_iterator const_data_iterator;

// allocators/deallocators:

    csr() : base(new_macro(rep())) {}
    explicit csr(const asr<T,sequential>& a) : base(new_macro(rep(a.data()))) {}
    void resize (size_type loc_nrow1 = 0, size_type loc_ncol1 = 0, size_type loc_nnz1 = 0)
        { base::data().resize(loc_nrow1, loc_ncol1, loc_nnz1); }
    void resize (const distributor& row_ownership, const distributor& col_ownership, size_type nnz1 = 0)
        { base::data().resize(row_ownership, col_ownership, nnz1); }

// allocators from initializer list (c++ 2011):

#ifdef _RHEOLEF_HAVE_STD_INITIALIZER_LIST
    csr (const std::initializer_list<csr_concat_value<T,sequential> >& init_list);
    csr (const std::initializer_list<csr_concat_line<T,sequential> >&  init_list);
#endif // _RHEOLEF_HAVE_STD_INITIALIZER_LIST

// accessors:

    // global sizes
    const distributor& row_ownership() const { return base::data().row_ownership(); }
    const distributor& col_ownership() const { return base::data().col_ownership(); }
    size_type dis_nrow () const              { return row_ownership().dis_size(); }
    size_type dis_ncol () const              { return col_ownership().dis_size(); }
    size_type dis_nnz () const               { return base::data().nnz(); }
    bool is_symmetric() const                { return base::data().is_symmetric(); }
    void set_symmetry (bool is_symm)         { base::data().set_symmetry(is_symm); }
    size_type pattern_dimension() const      { return base::data().pattern_dimension(); }
    void set_pattern_dimension(size_type dim){ base::data().set_pattern_dimension(dim); }

    // local sizes
    size_type nrow () const		     { return base::data().nrow(); }
    size_type ncol () const		     { return base::data().ncol(); }
    size_type nnz () const		     { return base::data().nnz(); }
    
    // range on local memory
    size_type row_first_index () const       { return base::data().row_first_index(); }
    size_type row_last_index () const        { return base::data().row_last_index(); }
    size_type col_first_index () const       { return base::data().col_first_index(); }
    size_type col_last_index () const        { return base::data().col_last_index(); }

    const_iterator begin() const             { return base::data().begin(); }
    const_iterator end()   const             { return base::data().end(); }
    iterator begin_nonconst()                { return base::data().begin(); }
    iterator end_nonconst()                  { return base::data().end(); }

    // accessors, only for distributed (for interface compatibility)
    size_type ext_nnz() const                { return 0; }
    const_iterator ext_begin() const         { return const_iterator(); }
    const_iterator ext_end()   const         { return const_iterator(); }
    size_type jext2dis_j (size_type jext) const { return 0; }

// algebra:

    // y := a*x
    void mult (const vec<element_type,sequential>& x, vec<element_type,sequential>& y) const {
      base::data().mult (x,y);
    }
    vec<element_type,sequential> operator* (const vec<element_type,sequential>& x) const {
      vec<element_type,sequential> y (row_ownership(), element_type());
      mult (x, y);
      return y;
    }
    void trans_mult (const vec<element_type,sequential>& x, vec<element_type,sequential>& y) const {
      base::data().trans_mult (x,y);
    }
    vec<element_type,sequential> trans_mult (const vec<element_type,sequential>& x) const {
      vec<element_type,sequential> y (col_ownership(), element_type());
      trans_mult (x, y);
      return y;
    }
    // a+b, a-b
    csr<T,sequential> operator+ (const csr<T,sequential>& b) const;
    csr<T,sequential> operator- (const csr<T,sequential>& b) const;

    // lambda*a
    csr<T,sequential>& operator*= (const T& lambda) {
      base::data().operator*= (lambda);
      return *this;
    }
// output:

    void dump (const std::string& name) const { base::data().dump(name); }
};
// lambda*a
template<class T>
inline
csr<T,sequential>
operator* (const T& lambda, const csr<T,sequential>& a)
{
  csr<T,sequential> b = a;
  b.operator*= (lambda);
  return b;
}
// -a
template<class T>
inline
csr<T,sequential>
operator- (const csr<T,sequential>& a)
{
  return T(-1)*a;
}
// trans(a)
template<class T>
inline
csr<T,sequential>
trans (const csr<T,sequential>& a)
{
  csr<T,sequential> b;
  a.data().build_transpose (b.data());
  return b;
}
//>verbatim:

#ifdef _RHEOLEF_HAVE_MPI
//<verbatim:
template<class T>
class csr<T,distributed> : public smart_pointer<csr_mpi_rep<T> > {
public:

// typedefs:

    typedef csr_mpi_rep<T>                    rep;
    typedef smart_pointer<rep>                base;
    typedef typename rep::memory_type         memory_type;
    typedef typename rep::size_type           size_type;
    typedef typename rep::element_type        element_type;
    typedef typename rep::iterator            iterator;
    typedef typename rep::const_iterator      const_iterator;
    typedef typename rep::data_iterator       data_iterator;
    typedef typename rep::const_data_iterator const_data_iterator;

// allocators/deallocators:

    csr() : base(new_macro(rep())) {}
    explicit csr(const asr<T,memory_type>& a) : base(new_macro(rep(a.data()))) {}
    void resize (const distributor& row_ownership, const distributor& col_ownership, size_type nnz1 = 0)
        { base::data().resize(row_ownership, col_ownership, nnz1); }

// allocators from initializer list (c++ 2011):

#ifdef _RHEOLEF_HAVE_STD_INITIALIZER_LIST
    csr (const std::initializer_list<csr_concat_value<T,distributed> >& init_list);
    csr (const std::initializer_list<csr_concat_line<T,distributed> >&  init_list);
#endif // _RHEOLEF_HAVE_STD_INITIALIZER_LIST

// accessors:

    // global sizes
    const distributor& row_ownership() const { return base::data().row_ownership(); }
    const distributor& col_ownership() const { return base::data().col_ownership(); }
    size_type dis_nrow () const              { return row_ownership().dis_size(); }
    size_type dis_ncol () const              { return col_ownership().dis_size(); }
    size_type dis_nnz () const               { return base::data().dis_nnz(); }
    bool is_symmetric() const                { return base::data().is_symmetric(); }
    void set_symmetry (bool is_symm)         { base::data().set_symmetry(is_symm); }
    size_type pattern_dimension() const      { return base::data().pattern_dimension(); }
    void set_pattern_dimension(size_type dim){ base::data().set_pattern_dimension(dim); }

    // local sizes
    size_type nrow () const		     { return base::data().nrow(); }
    size_type ncol () const		     { return base::data().ncol(); }
    size_type nnz () const		     { return base::data().nnz(); }
    
    // range on local memory
    size_type row_first_index () const       { return base::data().row_first_index(); }
    size_type row_last_index () const        { return base::data().row_last_index(); }
    size_type col_first_index () const       { return base::data().col_first_index(); }
    size_type col_last_index () const        { return base::data().col_last_index(); }

    const_iterator begin() const             { return base::data().begin(); }
    const_iterator end()   const             { return base::data().end(); }
    iterator begin_nonconst()                { return base::data().begin(); }
    iterator end_nonconst()                  { return base::data().end(); }

    // accessors, only for distributed
    size_type ext_nnz() const                { return base::data().ext_nnz(); }
    const_iterator ext_begin() const         { return base::data().ext_begin(); }
    const_iterator ext_end()   const         { return base::data().ext_end(); }
    size_type jext2dis_j (size_type jext) const { return base::data().jext2dis_j(jext); }

// algebra:


    // y := a*x
    void mult (const vec<element_type,distributed>& x, vec<element_type,distributed>& y) const {
      base::data().mult (x,y);
    }
    vec<element_type,distributed> operator* (const vec<element_type,distributed>& x) const {
      vec<element_type,distributed> y (row_ownership(), element_type());
      mult (x, y);
      return y;
    }
    void trans_mult (const vec<element_type,distributed>& x, vec<element_type,distributed>& y) const {
      base::data().trans_mult (x,y);
    }
    vec<element_type,distributed> trans_mult (const vec<element_type,distributed>& x) const {
      vec<element_type,distributed> y (col_ownership(), element_type());
      trans_mult (x, y);
      return y;
    }
    // a+b, a-b
    csr<T,distributed> operator+ (const csr<T,distributed>& b) const;
    csr<T,distributed> operator- (const csr<T,distributed>& b) const;

    // lambda*a
    csr<T,distributed>& operator*= (const T& lambda) {
      base::data().operator*= (lambda);
      return *this;
    }
// output:

    void dump (const std::string& name) const { base::data().dump(name); }
};
// lambda*a
template<class T>
inline
csr<T,distributed>
operator* (const T& lambda, const csr<T,distributed>& a)
{
  csr<T,distributed> b = a;
  b.operator*= (lambda);
  return b;
}
// -a
template<class T>
inline
csr<T,distributed>
operator- (const csr<T,distributed>& a)
{
  return T(-1)*a;
}
// trans(a)
template<class T>
inline
csr<T,distributed>
trans (const csr<T,distributed>& a)
{
  csr<T,distributed> b;
  a.data().build_transpose (b.data());
  return b;
}
//>verbatim:
#endif // _RHEOLEF_HAVE_MPI

// ------------------------------
// i/o
// ------------------------------
template <class T, class M>
inline
idiststream&
operator >> (idiststream& s,  csr<T,M>& x)
{ 
    return x.data().get(s); 
}
template <class T, class M> 
inline
odiststream&
operator << (odiststream& s, const csr<T,M>& x)
{
    return x.data().put(s); 
}
// ------------------------------
// a+b, a-b
// ------------------------------
template <class T> 
inline
csr<T,sequential>
csr<T,sequential>::operator+ (const csr<T,sequential>& b) const {
  csr<T,sequential> c;
  c.data().assign_add (this->data(), b.data(), std::plus<T>());
  return c;
}
template <class T> 
inline
csr<T,sequential>
csr<T,sequential>::operator- (const csr<T,sequential>& b) const {
  csr<T,sequential> c;
  c.data().assign_add (this->data(), b.data(), std::minus<T>());
  return c;
}
#ifdef _RHEOLEF_HAVE_MPI
template <class T> 
inline
csr<T,distributed>
csr<T,distributed>::operator+ (const csr<T,distributed>& b) const {
  csr<T,distributed> c;
  c.data().assign_add (this->data(), b.data(), std::plus<T>());
  return c;
}
template <class T> 
inline
csr<T,distributed>
csr<T,distributed>::operator- (const csr<T,distributed>& b) const {
  csr<T,distributed> c;
  c.data().assign_add (this->data(), b.data(), std::minus<T>());
  return c;
}
#endif // _RHEOLEF_HAVE_MPI
  
} // namespace rheolef
# endif // _RHEO_CSR_H
