///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is sequential in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/geo.h"
#include "rheolef/geo_domain.h"
#include "rheolef/rheostream.h"
#include "rheolef/iorheo.h"

#include "geo_header.h"

namespace rheolef {

// =========================================================================
// geo_base_rep members
// =========================================================================
template <class T, class M>
geo_base_rep<T,M>::~geo_base_rep()
{
#ifdef TO_CLEAN
  trace_macro ("dstor geo_base_rep: name="<<name()<<", dim="<<dimension()<<", map_dim="<<map_dimension()<<" size="<<size());
#endif // TO_CLEAN
}
template <class T, class M>
std::string
geo_base_rep<T,M>::name() const
{
  if (_serial_number == 0) return _name;
  return _name + "-" + itos(_serial_number);
}
template <class T, class M>
bool
geo_base_rep<T,M>::have_domain_indirect (const std::string& name) const
{
  for (typename std::vector<domain_indirect_basic<M> >::const_iterator
        iter = _domains.begin(), last = _domains.end(); iter != last; iter++) {
    const domain_indirect_basic<M>& dom = *iter;
    if (name == dom.name()) return true;
  }
  return false;
}
template <class T, class M>
const domain_indirect_basic<M>&
geo_base_rep<T,M>::get_domain_indirect (const std::string& name) const
{
  for (typename std::vector<domain_indirect_basic<M> >::const_iterator
        iter = _domains.begin(), last = _domains.end(); iter != last; iter++) {
    const domain_indirect_basic<M>& dom = *iter;
    if (name == dom.name()) return dom;
  }
  error_macro ("undefined domain \""<<name<<"\" in mesh \"" << _name << "\"");
  return *(_domains.begin()); // not reached
}
template <class T, class M>
void
geo_base_rep<T,M>::insert_domain_indirect (const domain_indirect_basic<M>& dom) const
{
  for (typename std::vector<domain_indirect_basic<M> >::iterator
        iter = _domains.begin(), last = _domains.end(); iter != last; iter++) {
    domain_indirect_basic<M>& curr_dom = *iter;
    if (dom.name() == curr_dom.name()) {
      // a domain with the same name already exists: replace it
      curr_dom = dom;
      return;
    }
  }
  // insert a new domain:
  _domains.push_back (dom);
}
template <class T, class M>
typename geo_base_rep<T,M>::size_type
geo_base_rep<T,M>::size (size_type dim) const
{
  size_type sz = 0;
  for (size_type variant = reference_element::first_variant_by_dimension(dim);
                 variant < reference_element:: last_variant_by_dimension(dim); variant++) {
    sz += _geo_element [variant].size();
  }
  return sz;
}
template <class T, class M>
typename geo_base_rep<T,M>::size_type
geo_base_rep<T,M>::dis_size (size_type dim) const
{
  size_type sz = 0;
  for (size_type variant = reference_element::first_variant_by_dimension(dim);
                 variant < reference_element:: last_variant_by_dimension(dim); variant++) {
    sz += _geo_element [variant].dis_size();
  }
  return sz;
}
template <class T, class M>
void
geo_base_rep<T,M>::compute_bbox()
{
  // first, compute bbox sequentialy:
  for (size_type j = 0; j < _dimension; j++) {
    _xmin[j] =  std::numeric_limits<T>::max();
    _xmax[j] = -std::numeric_limits<T>::max();
  }
  for (size_type j = _dimension+1; j < 3; j++) {
    _xmin [j] = _xmax [j] = 0;
  }
  for (size_type inod = 0, nnod = _node.size(); inod < nnod; inod++) {
    const point_basic<T>& x = _node [inod];
    for (size_type j = 0 ; j < _dimension; j++) {
      _xmin[j] = min(x[j], _xmin[j]);
      _xmax[j] = max(x[j], _xmax[j]);
    }
  }
  // distributed case: min & max are grouped accross procs 
#ifdef _RHEOLEF_HAVE_MPI
  if (is_distributed<M>::value) {
    for (size_type j = 0 ; j < _dimension; j++) {
      _xmin[j] = mpi::all_reduce (comm(), _xmin[j], mpi::minimum<T>());
      _xmax[j] = mpi::all_reduce (comm(), _xmax[j], mpi::maximum<T>());
    }
  }
#endif // _RHEOLEF_HAVE_MPI
}
template <class T, class M>
typename geo_base_rep<T,M>::const_reference
geo_base_rep<T,M>::get_geo_element (size_type dim, size_type ige) const
{
  size_type first_ige = 0, last_ige = 0;
  for (size_type variant = reference_element::first_variant_by_dimension(dim);
                 variant < reference_element:: last_variant_by_dimension(dim); variant++) {
    last_ige += _geo_element [variant].size();
    if (ige < last_ige) return _geo_element [variant] [ige - first_ige];
    first_ige = last_ige;
  }
  error_macro ("geo_element index " << ige << " out of range [0:"<< last_ige << "[");
  return _geo_element [0][0]; // not reached
}
template <class T, class M>
typename geo_base_rep<T,M>::reference
geo_base_rep<T,M>::get_geo_element (size_type dim, size_type ige)
{
  size_type first_ige = 0, last_ige = 0;
  for (size_type variant = reference_element::first_variant_by_dimension(dim);
                 variant < reference_element:: last_variant_by_dimension(dim); variant++) {
    last_ige += _geo_element [variant].size();
    if (ige < last_ige) return _geo_element [variant] [ige - first_ige];
    first_ige = last_ige;
  }
  error_macro ("geo_element index " << ige << " out of range [0:"<< last_ige << "[");
  return _geo_element [0][0]; // not reached
}
template <class T, class M>
typename geo_base_rep<T,M>::node_type
geo_base_rep<T,M>::piola (const geo_element& K, const node_type& hat_x) const
{
  // TODO: use basis_on_lattice [K.variant] instead of values[]
  std::vector<T> values;
  reference_element hat_K (K.variant());
  _numbering.get_basis().eval (hat_K, hat_x, values);

  std::vector<size_type> dis_inod;
  _numbering.dis_idof (_gs, K, dis_inod);
  node_type x;
  for (size_type loc_inod = 0, loc_nnod = values.size(); loc_inod < loc_nnod; loc_inod++) {
    const node_type& xnod = dis_node(dis_inod[loc_inod]);
    const T& coeff = values[loc_inod];
    for (size_type i = 0, d = dimension(); i < d; i++) {
      x[i] += coeff*xnod[i];
    }
  }
  return x;
}
// ----------------------------------------------------------------------------
// set order & resize node array ; internal node are not computed
// since they depend on the curved boundary information (cad data)
// that is not aivailable here
// ----------------------------------------------------------------------------
#define _RHEOLEF_reset_order(M)						\
template <class T>							\
void									\
geo_rep<T,M>::reset_order (size_type new_order)				\
{									\
  if (new_order == base::_numbering.degree()) return;			\
  base::_numbering.set_degree (new_order);				\
  size_type dis_nnod = base::_numbering.dis_ndof (base::_gs, base::_map_dimension); \
  size_type     nnod = base::_numbering.ndof     (base::_gs, base::_map_dimension); \
  base::_gs.node_ownership = distributor (nnod, base::comm(), nnod);	\
  array<point_basic<T>, M> new_node (base::_gs.node_ownership);		\
  for (size_type iv = 0, nv = base::_gs.ownership_by_dimension[0].size(); iv < nv; iv++) { \
    new_node [iv] = base::_node [iv];					\
  }									\
  base::_node = new_node;						\
  build_external_entities ();						\
}
_RHEOLEF_reset_order(sequential)
#ifdef _RHEOLEF_HAVE_MPI
_RHEOLEF_reset_order(distributed)
#endif // _RHEOLEF_HAVE_MPI
#undef _RHEOLEF_reset_order
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class geo_base_rep<Float,sequential>;
template class geo_rep<Float,sequential>;
#ifdef _RHEOLEF_HAVE_MPI
template class geo_base_rep<Float,distributed>;
template class geo_rep<Float,distributed>;
#endif // _RHEOLEF_HAVE_MPI

template geo_basic<Float,sequential> compact (const geo_basic<Float,sequential>&);

} // namespace rheolef
