///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "curl.h"
#include "rheolef/ublas_matrix_range.h"

namespace rheolef {
using namespace std;
using namespace ublas;
// --------
// 2d case
// --------
//
// curl = [ -d_dx1 , d_dx0 ]
//
//  axisymmetric: idem (with r dr dz weight)
// --------
// 3d case
// --------
//
//        [    0     -d_dx2 ,  d_dx1 ]
// curl = [  d_dx2 ,    0,    -d_dx0 ]
//        [ -d_dx1 ,  d_dx0,     0   ]
//
template<class T, class M>
void
curl<T,M>::operator() (const geo_element& K, ublas::matrix<T>& dd) const
{
  check_macro (! base::is_on_band(), "unsupported banded level set extension");

  switch (K.dimension()) {
    case 1:
      base::build_d_dx (K, dd, 0);
      dd *= 2;
      break;
    case 2: {
      matrix<T> dx0, dx1;
      base::build_d_dx (K, dx0, 0);
      base::build_d_dx (K, dx1, 1);
      size_type ni = dx0.size1();
      size_type nj = dx0.size2();
      dd.resize (ni, 2*nj);
      dd.clear();
      // [ -d_dx1 , d_dx0 ]
      mr_set (dd, range(0,ni), range(0,nj),    -dx1);
      mr_set (dd, range(0,ni), range(nj,2*nj),  dx0);
      break;
    }
    case 3: {
      matrix<T> dx0, dx1, dx2;
      base::build_d_dx (K, dx0, 0);
      base::build_d_dx (K, dx1, 1);
      base::build_d_dx (K, dx2, 2);
      size_type ni = dx0.size1();
      size_type nj = dx0.size2();
      dd.resize (3*ni, 3*nj);
      dd.clear();
      //        [   0    -dx2 ,  dx1 ]
      // curl = [  dx2 ,   0,   -dx0 ]
      //        [ -dx1 ,  dx0,    0   ]
      mr_set (dd, range(0,ni),      range(nj,2*nj),   -dx2);
      mr_set (dd, range(0,ni),      range(2*nj,3*nj),  dx1);
      mr_set (dd, range(ni,2*ni),   range(0,nj),       dx2);
      mr_set (dd, range(ni,2*ni),   range(2*nj,3*nj), -dx0);
      mr_set (dd, range(2*ni,3*ni), range(0,nj),      -dx1);
      mr_set (dd, range(2*ni,3*ni), range(nj,2*nj),    dx0);
      break;
    }
    default: {
      fatal_macro ("form `curl' not yet supported in " << K.dimension() << "d geometry");
      break;
    }
  }
}
template<class T, class M>
void
curl<T,M>::initialize () const
{
  base::set_n_derivative(1);

  check_macro (base::get_first_space().get_numbering().is_continuous(),
	"illegal discontinuous `"<<base::get_first_space().get_numbering().name()
	<< "' first space for the `curl' operator");
  size_t d = base::coordinate_dimension();
  check_macro (
	 ((d==1) && space_constant::scalar == base::get_first_space().valued_tag() 
	         && space_constant::scalar == base::get_second_space().valued_tag()   ) || 
	 ((d==2) && space_constant::vector == base::get_first_space().valued_tag() 
	         && space_constant::scalar == base::get_second_space().valued_tag()   ) || 
	 ((d==3) && space_constant::vector == base::get_first_space().valued_tag() 
	         && space_constant::vector == base::get_second_space().valued_tag()   ),
        "illegal "<< base::get_first_space().valued()<<" and "<< base::get_second_space().valued()
	<< " valued first and second spaces for the `curl' form in "<<d<<" dimension");
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class curl<Float,sequential>;

#ifdef _RHEOLEF_HAVE_MPI
template class curl<Float,distributed>;
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
