///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "Pkd_numbering.h"
#include "rheolef/geo.h"
#include "rheolef/rheostream.h"
namespace rheolef {
using namespace std;

template <class T, class M>
numbering_Pkd<T,M>::numbering_Pkd (std::string name)
 : _name(name), _degree()
{
  if (_name == "P0d") _name = "P0"; // guaranties the name unicity: strip "P0d"
  if (_name.length() > 0 && _name[0] == 'P') {
    // TODO: check also that name fits "Pkd" where is an k integer
    _degree = atoi(name.c_str()+1);
  } else if (_name.length() > 0) { // missing 'P' !
    error_macro ("invalid polynomial name `"<<_name<<"' for the Pkd family");
  } else {
    // empty _name : default cstor
    _degree = 0;
  }
  base::_basis = basis ("P" + itos(_degree));
  init_local_ndof_by_variant();
}
template <class T, class M>
void
numbering_Pkd<T,M>::init_local_ndof_by_variant () const
{
  size_type k = _degree;
  _loc_ndof_by_variant [reference_element::p] = 1;
  _loc_ndof_by_variant [reference_element::e] = k+1;
  _loc_ndof_by_variant [reference_element::t] = (k+1)*(k+2)/2;
  _loc_ndof_by_variant [reference_element::q] = (k+1)*(k+1);
  _loc_ndof_by_variant [reference_element::T] = (k+1)*(k+2)*(k+3)/6;
  _loc_ndof_by_variant [reference_element::P] = (k+1)*(k+1)*(k+2)/2;
  _loc_ndof_by_variant [reference_element::H] = (k+1)*(k+1)*(k+1);
}
template <class T, class M>
void
numbering_Pkd<T,M>::set_degree (size_type degree) const
{
  _name   = "P" + itos(degree);
  if (degree != 0) {
    _name = _name + "d";
  }
  _degree = degree;
  base::_basis = basis ("P" + itos(degree));
  init_local_ndof_by_variant();
}
template <class T, class M>
typename numbering_Pkd<T,M>::size_type
numbering_Pkd<T,M>::degree () const
{
  return _degree;
}
template <class T, class M>
std::string
numbering_Pkd<T,M>::name() const
{
  return _name;
}
template <class T, class M>
bool
numbering_Pkd<T,M>::is_continuous () const
{
  return false;
}
template <class T, class M>
typename numbering_Pkd<T,M>::size_type
numbering_Pkd<T,M>::ndof (
        const geo_size& gs, 
        size_type       map_dim) const
{
  size_type ndof = 0;
  for (size_type variant = reference_element::first_variant_by_dimension(map_dim);
                 variant < reference_element:: last_variant_by_dimension(map_dim);
	         variant++) { 
    ndof += gs.ownership_by_variant [variant].size() * _loc_ndof_by_variant [variant];
  }
  return ndof;
}
template <class T, class M>
typename numbering_Pkd<T,M>::size_type
numbering_Pkd<T,M>::dis_ndof (
        const geo_size& gs, 
        size_type       map_dim) const
{
  size_type dis_ndof = 0;
  for (size_type variant = reference_element::first_variant_by_dimension(map_dim);
                 variant < reference_element:: last_variant_by_dimension(map_dim);
	         variant++) { 
    dis_ndof += gs.ownership_by_variant [variant].dis_size() * _loc_ndof_by_variant [variant];
  }
  return dis_ndof;
}
template <class T, class M>
void
numbering_Pkd<T,M>::dis_idof (
        const geo_size&       gs, 
	const geo_element&    K, 
	std::vector<size_type>& dis_idof) const
{
  // get some infos on K:
  size_type variant = K.variant();
  size_type map_dim = K.dimension();
  size_type dis_ige = K.dis_ie();
  size_type iproc   = gs.ownership_by_dimension [map_dim].find_owner (dis_ige);
  if (map_dim == 0) {
    // convert node numbering to vertex numbering, for geo order > 1
    size_type dis_inod = dis_ige;
    size_type first_dis_inod = gs.node_ownership.first_index(iproc);
    size_type first_dis_iv   = gs.ownership_by_variant [variant].first_index(iproc);
    assert_macro (dis_inod >= first_dis_inod, "invalid index");
    size_type inod = dis_inod - first_dis_inod;
    size_type iv = inod;
    size_type dis_iv = first_dis_iv + iv;
    dis_ige = dis_iv;
  }
  // compute dis_igev from dis_ige:
  size_type first_dis_v = 0;
  for (size_type prev_variant = reference_element::first_variant_by_dimension(map_dim);
                 prev_variant < variant;
	         prev_variant++) { 
    size_type shift = gs.ownership_by_variant [prev_variant].last_index(iproc);
    first_dis_v += shift;
  }
  for (size_type next_variant = variant+1;
                 next_variant < reference_element::last_variant_by_dimension(map_dim);
	         next_variant++) { 
    size_type shift = gs.ownership_by_variant [next_variant].first_index(iproc);
    first_dis_v += shift;
  }
  assert_macro (dis_ige >= first_dis_v, "invalid index");
  size_type dis_igev = dis_ige - first_dis_v;

  // compute the first dis_idof on K:
  size_type dis_idof0 = 0;
  for (size_type prev_variant = reference_element::first_variant_by_dimension(map_dim);
                 prev_variant < variant; 
                 prev_variant++) {
    dis_idof0 += gs.ownership_by_variant [prev_variant]. last_index(iproc) * _loc_ndof_by_variant [prev_variant];
  }
  dis_idof0   += dis_igev * _loc_ndof_by_variant [variant];

  for (size_type next_variant = variant+1;
                 next_variant < reference_element::last_variant_by_dimension(map_dim);
                 next_variant++) {
    dis_idof0 += gs.ownership_by_variant [next_variant].first_index(iproc) * _loc_ndof_by_variant [next_variant];
  }
  // others dis_idofs are contiguous:
  dis_idof.resize (reference_element::n_node (variant, _degree));
  for (size_type loc_idof = 0, loc_ndof = dis_idof.size(); loc_idof < loc_ndof; loc_idof++) {
    dis_idof [loc_idof] = dis_idof0 + loc_idof;
  }
}
// -------------------------------------------------------------
// permut to/from ios dof numbering, for distributed environment
// -------------------------------------------------------------
template <class T, class M>
static 
void
Pkd_set_ios_permutations (
    const geo_basic<T,M>& omega,
    size_t			       degree,
    array<distributor::size_type,M>&   idof2ios_dis_idof,
    array<distributor::size_type,M>&   ios_idof2dis_idof,
    const boost::array<distributor::size_type,reference_element::max_variant>& loc_ndof_by_variant)
{
}
#ifdef _RHEOLEF_HAVE_MPI
template <class T>
static 
void
Pkd_set_ios_permutations (
    const geo_basic<T,distributed>&              omega,
    size_t					 degree,
    array<distributor::size_type,distributed>&   idof2ios_dis_idof,
    array<distributor::size_type,distributed>&   ios_idof2dis_idof,
    const boost::array<distributor::size_type,reference_element::max_variant>& loc_ndof_by_variant)
{
  // general method:
  // - first compute idof2ios_dis_idof
  // - then invert permutation into ios_idof2dis_idof
  typedef distributor::size_type size_type;
  // ----------------------------
  // 1) compute idof2ios_dis_idof
  // ----------------------------
  size_type map_dim = omega.map_dimension();
  const geo_size& gs = omega.sizes(); 
  size_type ndof = 0;
  size_type dis_ndof = 0;
  for (size_type variant = reference_element::first_variant_by_dimension(map_dim);
                 variant < reference_element:: last_variant_by_dimension(map_dim);
	         variant++) { 
    ndof     += gs.ownership_by_variant [variant].size()     * loc_ndof_by_variant [variant];
    dis_ndof += gs.ownership_by_variant [variant].dis_size() * loc_ndof_by_variant [variant];
  }
  distributor dof_ownership (dis_ndof, omega.comm(), ndof);
  idof2ios_dis_idof.resize  (dof_ownership);
  switch (map_dim) {
	case 0: 
	case 1: {
	  // non-mixed meshes:
	  size_type loc_ndof_per_elt;
          if (map_dim == 0) {
            loc_ndof_per_elt = loc_ndof_by_variant [reference_element::p];
          } else {
            loc_ndof_per_elt = loc_ndof_by_variant [reference_element::e];
          }
          for (size_type ige = 0, nge = omega.geo_element_ownership(map_dim).size(); ige < nge; ige++) {
	    size_type ios_dis_ige = omega.ige2ios_dis_ige (map_dim, ige);
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_elt; loc_idof++) {
	      size_type idof         = loc_ndof_per_elt*ige         + loc_idof;
	      size_type ios_dis_idof = loc_ndof_per_elt*ios_dis_ige + loc_idof;
	      idof2ios_dis_idof [idof] = ios_dis_idof;
            }
          }
	  break;
	}
	case 2: {
	  // possibly (t,q) mixed meshes:
	  // part 1: triangles
          size_type loc_ndof_per_tri = loc_ndof_by_variant [reference_element::t];
          size_type     ntri = omega.sizes().ownership_by_variant[reference_element::t].size();
          size_type dis_ntri = omega.sizes().ownership_by_variant[reference_element::t].dis_size();
          for (size_type itri = 0; itri < ntri; itri++) {
	    size_type ige = itri;
	    size_type ios_dis_ige  = omega.ige2ios_dis_ige (map_dim, ige);
	    size_type ios_dis_itri = ios_dis_ige;
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_tri; loc_idof++) {
	      size_type idof         = loc_ndof_per_tri*itri + loc_idof;
	      size_type ios_dis_idof = loc_ndof_per_tri*ios_dis_itri + loc_idof;
	      assert_macro (idof < idof2ios_dis_idof.size(), "unexpected idof");
	      assert_macro (ios_dis_idof < dis_ndof, "unexpected ios_dis_idof");
	      idof2ios_dis_idof [idof] = ios_dis_idof;
	    }
	  }
	  // part 2: quadrangle
          size_type loc_ndof_per_qua = loc_ndof_by_variant [reference_element::q];
          size_type nqua = omega.sizes().ownership_by_variant[reference_element::q].size();
          for (size_type iqua = 0; iqua < nqua; iqua++) {
	    size_type ige = ntri + iqua;
	    size_type ios_dis_ige  = omega.ige2ios_dis_ige (map_dim, ige);
	    assert_macro (ios_dis_ige >= dis_ntri, "unexpected numbering for element of variant `q'");
	    size_type ios_dis_iqua = ios_dis_ige - dis_ntri;
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_qua; loc_idof++) {
	      size_type idof = loc_ndof_per_tri*ntri + loc_ndof_per_qua*iqua + loc_idof;
	      size_type ios_dis_idof = loc_ndof_per_tri*dis_ntri + loc_ndof_per_qua*ios_dis_iqua + loc_idof;
	      assert_macro (idof < idof2ios_dis_idof.size(), "unexpected idof");
	      assert_macro (ios_dis_idof < dis_ndof, "unexpected ios_dis_idof");
	      idof2ios_dis_idof [idof] = ios_dis_idof;
	    }
	  }
	  break;
	}
	case 3:
        default: {
	  // possibly (T,P,H) mixed meshes:
	  // part 1: tetra
          size_type loc_ndof_per_tet = loc_ndof_by_variant [reference_element::T];
          size_type     ntet = omega.sizes().ownership_by_variant[reference_element::T].size();
          size_type dis_ntet = omega.sizes().ownership_by_variant[reference_element::T].dis_size();
          for (size_type itet = 0; itet < ntet; itet++) {
	    size_type ige = itet;
	    size_type ios_dis_ige  = omega.ige2ios_dis_ige (map_dim, ige);
	    size_type ios_dis_itet = ios_dis_ige;
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_tet; loc_idof++) {
	      size_type idof         = loc_ndof_per_tet*itet + loc_idof;
	      size_type ios_dis_idof = loc_ndof_per_tet*ios_dis_itet + loc_idof;
	      assert_macro (idof < idof2ios_dis_idof.size(), "unexpected idof");
	      assert_macro (ios_dis_idof < dis_ndof, "unexpected ios_dis_idof");
	      idof2ios_dis_idof [idof] = ios_dis_idof;
	    }
	  }
	  // part 2: prism
          size_type loc_ndof_per_pri = loc_ndof_by_variant [reference_element::P];
          size_type     npri = omega.sizes().ownership_by_variant[reference_element::P].size();
          size_type dis_npri = omega.sizes().ownership_by_variant[reference_element::P].dis_size();
          for (size_type ipri = 0; ipri < npri; ipri++) {
	    size_type ige = ntet + ipri;
	    size_type ios_dis_ige  = omega.ige2ios_dis_ige (map_dim, ige);
	    assert_macro (ios_dis_ige >= dis_ntet, "unexpected numbering for element of variant `P'");
	    size_type ios_dis_ipri = ios_dis_ige - dis_ntet;
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_pri; loc_idof++) {
	      size_type idof = loc_ndof_per_tet*ntet + loc_ndof_per_pri*ipri + loc_idof;
	      size_type ios_dis_idof = loc_ndof_per_tet*dis_ntet + loc_ndof_per_pri*ios_dis_ipri + loc_idof;
	      assert_macro (idof < idof2ios_dis_idof.size(), "unexpected idof");
	      assert_macro (ios_dis_idof < dis_ndof, "unexpected ios_dis_idof");
	      idof2ios_dis_idof [idof] = ios_dis_idof;
	    }
	  }
	  // part 3: hexa
          size_type loc_ndof_per_hex = loc_ndof_by_variant [reference_element::H];
          size_type nhex = omega.sizes().ownership_by_variant[reference_element::H].size();
          for (size_type ihex = 0; ihex < nhex; ihex++) {
	    size_type ige = ntet + npri + ihex;
	    size_type ios_dis_ige  = omega.ige2ios_dis_ige (map_dim, ige);
	    assert_macro (ios_dis_ige >= dis_ntet + dis_npri, "unexpected numbering for element of variant `H'");
	    size_type ios_dis_ihex = ios_dis_ige - dis_ntet - dis_npri;
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_hex; loc_idof++) {
	      size_type idof = loc_ndof_per_tet*ntet + loc_ndof_per_pri*npri + loc_ndof_per_hex*ihex + loc_idof;
	      size_type ios_dis_idof = loc_ndof_per_tet*dis_ntet + loc_ndof_per_pri*dis_npri + loc_ndof_per_hex*ios_dis_ihex + loc_idof;
	      assert_macro (idof < idof2ios_dis_idof.size(), "unexpected idof");
	      assert_macro (ios_dis_idof < dis_ndof, "unexpected ios_dis_idof");
	      idof2ios_dis_idof [idof] = ios_dis_idof;
	    }
	  }
	  break;
	}
  }
  // ----------------------------------------------
  // 2) invert permutation into ios_idof2dis_idof
  // ----------------------------------------------
  distributor ios_dof_ownership (dis_ndof, idof2ios_dis_idof.comm(), distributor::decide);
  ios_idof2dis_idof.resize (ios_dof_ownership, std::numeric_limits<size_type>::max());
  idof2ios_dis_idof.reverse_permutation (ios_idof2dis_idof);
}
#endif // _RHEOLEF_HAVE_MPI
template <class T, class M>
void
numbering_Pkd<T,M>::set_ios_permutations (
    const geo_basic<T,M>& omega,
    array<size_type,M>&   idof2ios_dis_idof,
    array<size_type,M>&   ios_idof2dis_idof) const
{
    Pkd_set_ios_permutations (omega, degree(), idof2ios_dis_idof, ios_idof2dis_idof, _loc_ndof_by_variant);
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------

template class numbering_Pkd<Float,sequential>;

#ifdef _RHEOLEF_HAVE_MPI
template class numbering_Pkd<Float,distributed>;
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
