/*****
*
* Copyright (C) 1998, 1999, 2000, 2002 Yoann Vandoorselaere <yoann@prelude-ids.org>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
#include <ctype.h>
#include <assert.h>
#include <inttypes.h>

#include "packet.h"

#include <libprelude/list.h>
#include <libprelude/timer.h>
#include <libprelude/prelude-log.h>
#include <libprelude/extract.h>
#include <libprelude/plugin-common.h>

#include "pcap.h"
#include "ip-fragment.h"
#include "packet-decode.h"
#include "optparse.h"
#include "plugin-detect.h"
#include "plugin-protocol.h"
#include "nids-alert.h"
#include "tcp-stream.h"

#include "pconfig.h"
#include "rules.h"
#include "rules-default.h"


extern Pconfig_t config;


/*
 * List containing plugins.
 */
extern struct list_head tcp_plugins_list;
extern struct list_head udp_plugins_list;
extern struct list_head ip_plugins_list;
extern struct list_head ipencap_plugins_list;
extern struct list_head data_plugins_list;
extern struct list_head icmp_plugins_list;
extern struct list_head igmp_plugins_list;
extern struct list_head arp_plugins_list;
extern struct list_head rarp_plugins_list;
extern struct list_head ether_plugins_list;
extern struct list_head all_plugins_list;


static nids_alert_t *packet_decode_alert(const char *class, size_t clen);


/*
 * Helper for packet_decode_add_header.
 */
#define packet_add_header(pc, data, dlen, proto) do {                            \
        int ret = packet_decode_add_header(__FUNCTION__, pc, data, dlen, proto); \
        if ( ret < 0 )                                                           \
                return;                                                          \
} while (0);


/*
 *
 */
#define setup_alert(class) packet_decode_alert((class), sizeof((class)))


#define bytes_are_in_frame(proto, packet, wanted_len) (                                   \
        (check_frame_len(packet->pcap_hdr->caplen, wanted_len) < 0 ) ?                    \
        (nids_alert(NULL, packet, setup_alert("truncated " proto " header"),              \
                        "truncated " proto " (header len=%d > caplen=%d)",                \
                        wanted_len, packet->pcap_hdr->caplen), -1) : 0 )


#define bytes_are_in_frame_nopkt(proto, caplen, wanted_len) (                             \
        ( check_frame_len(caplen, wanted_len) < 0 ) ?                                     \
        (nids_alert(NULL, NULL, setup_alert("truncated " proto " header"),                \
                        "truncated " proto " (header len=%d > caplen=%d)",                \
                        wanted_len, caplen), -1) : 0 )



static int check_frame_len(int remaining_len, int wanted_len) 
{
        return (wanted_len <= remaining_len) ? 0 : -1;
}



static nids_alert_t *packet_decode_alert(const char *class, size_t clen) 
{
        static int initialized = 0;
        static nids_alert_t alert;
        static idmef_impact_t impact;

        if ( ! initialized ) {
                initialized = 1;

                nids_alert_init(&alert);
                
                impact.type = dos;
                impact.completion = 0; /* we don't know */
                impact.severity = impact_low;
                alert.impact = &impact;
                alert.confidence = NULL;
                idmef_string_set_constant(&impact.description, "Might result in an operating system crash");
        }
        
        alert.classification.name.len = clen;
        alert.classification.name.string = class;
        
        return &alert;
}





static int packet_decode_add_header(const char *function, packet_container_t *pc,
                                           unsigned char *data, unsigned int dlen, proto_enum_t proto) 
{
        pc->depth++;
        
        if ( pc->depth + 1 < MAX_PKTDEPTH ) {
                pc->packet[pc->depth].len = dlen;
                pc->packet[pc->depth].proto = proto;
                pc->packet[pc->depth].p.ip = (void *) data;
                pc->packet[pc->depth + 1].proto = p_end;
                pc->packet[pc->depth + 1].len = 0; 
        } else {
                pc->packet[pc->depth].len = 0; 
                pc->packet[pc->depth].proto = p_end;
                nids_alert(NULL, (pc),
                           setup_alert("Packet depth is too high"),
                           "%s : Maximum packet depth (%d) reached",
                           function, MAX_PKTDEPTH);
                return -1;
        }

        return 0;
}




inline packet_container_t *packet_new(unsigned char *p) 
{
        packet_container_t *pkt;

        pkt = malloc(sizeof(packet_container_t));
        if ( ! pkt ) {
                log(LOG_ERR, "memory exhausted.\n");
                return NULL;
        }

        pkt->packet[0].proto = p_end;
        pkt->protocol_plugin_data = NULL;
        pkt->protocol_plugin_id = -1;
        pkt->application_layer_depth = -1;
        pkt->transport_layer_depth = -1;
        pkt->network_layer_depth = -1;
        pkt->depth = -1;
        pkt->refcount = 0;
        pkt->allocated_data = NULL;
        pkt->tcp_allocated_data = NULL;
        pkt->captured_data = p;
        pkt->paws_tsval = pkt->paws_tsecr = 0;
        pkt->tcp_sack = 0;
        
        packet_lock(pkt);
        
        return pkt;
}




void capture_data(packet_container_t *packet, unsigned char *p, unsigned int len)
{
        int plen;
        
        /*
         * Don't issue warning if there is no data,
         * it can be absolutly normal. This rule doesn't apply if the
         * allocated_data pointer is set, because we can't check this
         * pointer against snapend.
         */
        if ( ! packet->allocated_data && ! len ) 
                return;

        /*
         * add the data to our packet,
         * so that a protocol plugin emiting an alert will show the data
         * contained in the packet.
         */
        packet_add_header(packet, p, len, p_data);

        /*
         * Run the protocol plugins. If the return value
         * is positive, one of them matched the payload of this packet.
         * The returned value is the len it handled before the real payload.
         */        
        plen = protocol_plugins_run(packet, p, len);
        if ( plen < 0 )
                plen = 0; 
       
        if ( plen > 0 ) {                
                /*
                 * a protocol plugin handled some of the payload.
                 * remove the old payload, and add a new one.
                 */
                packet->depth--;
                packet_add_header(packet, p + plen, len - plen, p_data);
        }

        packet->application_layer_depth = packet->depth;
        
        /*
         * start data detection plugin *only* on the remaining data.
         */
        plen = 0;
        if ( len - plen > 0 ) 
                detect_plugins_run(packet, &data_plugins_list, "data");        

        /*
         * Then, we want the whole packet data (including protocol plugin
         * data) dumped at reporting time. correct the header.
         */
        packet->packet[packet->depth].len = len;
}


static void capture_tcp_options(packet_container_t *packet, unsigned char *p, int len, int *do_reasm) 
{
        int ret;
        
        ret = tcp_optdump(packet, p, len);        
        if ( len > 0 ) {
                
                if ( ret == len )
                        /*
                         * all option decoded successfully.
                         */
                        *do_reasm = 1;
                else
                        *do_reasm = 0;
                
                packet_add_header(packet, p, len, p_tcpopts);
        }
}



static void capture_ip_options(packet_container_t *packet, unsigned char *p, int len, int *do_reasm)
{
        int ret;
        
        ret = ip_optdump(packet, p, len); 
        if ( ret > 0 ) {
                if ( ret == len )
                        /*
                         * all option decoded successfully.
                         */
                        *do_reasm = 1;
                else
                        *do_reasm = 0;
                
                packet_add_header(packet, p, len, p_ipopts);
        }
}



static void capture_tcp(packet_container_t *packet, unsigned char *p)
{
        tcphdr_t *tcp;
        int hlen, len, ret, do_reasm = 1;

        if ( bytes_are_in_frame("TCP", packet, sizeof(tcphdr_t)) < 0 )
                return;
        
        packet_add_header(packet, p, sizeof(tcphdr_t), p_tcp);
        packet->transport_layer_depth = packet->depth;
        tcp = packet->packet[packet->depth].p.tcp;
        
        hlen = TH_OFF(tcp) * 4;
        if ( hlen > packet->pcap_hdr->caplen ) {
                nids_alert(NULL, packet, setup_alert("bad TCP header length"),
                           "bad TCP header length (hlen=%d, caplen=%d)", hlen,
                           packet->pcap_hdr->caplen);
                return;
        }

        packet->pcap_hdr->caplen -= hlen;
        detect_plugins_run(packet, &tcp_plugins_list, "tcp");
        
        len = hlen - sizeof(tcphdr_t);
        if ( len > 0 )
                capture_tcp_options(packet, p + sizeof(tcphdr_t), len, &do_reasm);

        if ( tcp_stream_is_enabled() && do_reasm ) {
                
                packet_add_header(packet, (p + hlen), packet->pcap_hdr->caplen, p_data);
                packet->application_layer_depth = packet->depth;

                ret = tcp_stream_store(packet, packet->packet[packet->network_layer_depth].p.ip, tcp);

                /*
                 * if stream is unknown, and that statefull_only is not set -> analyze data anyway.
                 * if tcp stream return -1 (packet is ignored)              -> analyze data.
                 */
                if ( (ret == tcp_stream_unknown && ! config.statefull_only) || ret == -1 ) {
                        packet->depth--;
                        capture_data(packet, p + hlen, packet->pcap_hdr->caplen);
                        signature_engine_process_packet(signature_engine_get_tcp_root(), packet);
                }
                
        } else {
                capture_data(packet, p + hlen, packet->pcap_hdr->caplen);
                signature_engine_process_packet(signature_engine_get_tcp_root(), packet);
        }
        
}



static void capture_udp(packet_container_t *packet, unsigned char *p)
{
        if ( bytes_are_in_frame("UDP", packet, sizeof(udphdr_t)) < 0 )
                return;

        packet_add_header(packet, p, sizeof(udphdr_t), p_udp);
        packet->pcap_hdr->caplen -= sizeof(udphdr_t);
        packet->transport_layer_depth = packet->depth;
        
        detect_plugins_run(packet, &udp_plugins_list, "udp");
        capture_data(packet, p + sizeof(udphdr_t), packet->pcap_hdr->caplen);
	signature_engine_process_packet(signature_engine_get_udp_root(), packet);
}




static void capture_icmp(packet_container_t *packet, unsigned char *p)
{
        int len;
        uint8_t type;
        static uint8_t type_tbl[] = {
            8,  /* type 0 */
            0,  /* type 1 */
            0,  /* type 2 */
            36, /* type 3 */
            36, /* type 4 */
            36, /* type 5 */
            0,  /* type 6 */
            0,  /* type 7 */
            8,  /* type 8 */
            0,  /* type 9 router advertissement message require extenssible struct */
            8,  /* type 10 */
            36, /* type 11 */
            36, /* type 12 */
            20, /* type 13 */
            20, /* type 14 */
            8,  /* type 15 */
            8,  /* type 16 */
            12, /* type 17 address mask request */
            12, /* type 18 address mask reply   */
        };

        if ( bytes_are_in_frame("ICMP", packet, ICMP_MINLEN) < 0 )
                return;
        
        type = (((const icmphdr_t *)p)->icmp_type);
        if ( type < sizeof(type_tbl) ) 
                len = (type_tbl[type] != 0) ? type_tbl[type] : ICMP_MINLEN;
        else
                len = ICMP_MINLEN;

        if ( len > packet->pcap_hdr->caplen ) {
                nids_alert(NULL, packet, setup_alert("truncated ICMP"),
                           "truncated ICMP (header len=%d > caplen=%d)", len,
                           packet->pcap_hdr->caplen);
                return;
        }
        
        packet_add_header(packet, p, len, p_icmp);
        packet->pcap_hdr->caplen -= len; 

        detect_plugins_run(packet, &icmp_plugins_list, "icmp");
        capture_data(packet, p + len, packet->pcap_hdr->caplen);
	
	signature_engine_process_packet(signature_engine_get_icmp_root(), packet);
}



static void capture_igmp(packet_container_t *packet, unsigned char *p)
{
        if ( bytes_are_in_frame("IGMP", packet, sizeof(igmphdr_t)) < 0 )
                return;
        
        packet_add_header(packet, p, sizeof(igmphdr_t), p_igmp);
        packet->pcap_hdr->caplen -= sizeof(igmphdr_t);
        
	detect_plugins_run(packet, &igmp_plugins_list, "igmp");     
        capture_data(packet, p + sizeof(igmphdr_t), packet->pcap_hdr->caplen);
}




static void capture_arp(packet_container_t *packet, unsigned char *p)
{
        if ( bytes_are_in_frame("ARP", packet, sizeof(etherarphdr_t)) < 0 )
                return;
        
        packet_add_header(packet, p, sizeof(etherarphdr_t), p_arp);
        packet->pcap_hdr->caplen -= sizeof(etherarphdr_t);

	detect_plugins_run(packet, &arp_plugins_list, "arp");
        capture_data(packet, p + sizeof(etherarphdr_t), packet->pcap_hdr->caplen);
}



static void capture_revarp(packet_container_t *packet, unsigned char *p)
{
        if ( bytes_are_in_frame("RARP", packet, sizeof(etherarphdr_t)) < 0 )
                return;

        packet_add_header(packet, p, sizeof(etherarphdr_t), p_rarp);
        packet->pcap_hdr->caplen -= sizeof(etherarphdr_t);

	detect_plugins_run(packet, &rarp_plugins_list, "rarp");
        capture_data(packet, p + sizeof(etherarphdr_t), packet->pcap_hdr->caplen);
}



/*
 * Return -1 on error,
 * 1 if a packet was reassembled.
 * 0 if a fragment was queued.
 */

#define FRAGMENT_QUEUED 0
#define PACKET_REASM 1


static int handle_ip_fragment(packet_container_t *packet, iphdr_t *ip, unsigned char *p, unsigned int hlen, unsigned int len) 
{
        int ret;
                
        if ( ((long) (len - hlen)) <= 0 ) {
                nids_alert(NULL, packet, setup_alert("Invalid fragment data size (too short)"),
                           "Caught a fragment with total len ((len=%u - hlen=%u) = %ld) caplen is %u",
                           len, hlen, ((long)len - hlen), packet->pcap_hdr->caplen);
                return -1;
        }

        assert(((long)(len - hlen)) <= packet->pcap_hdr->caplen);
             
        /*
         * ip_defrag return > 0 if a packet was reassembled,
         *                 == 0 for fragment queued,
         *                  < 0 on error.
         */
        ret = ip_defrag(packet, ip, p + hlen, len - hlen, &packet->allocated_data);
        if ( ret > 0 ) {
                ip = (iphdr_t *) packet->allocated_data;
                packet->pcap_hdr->caplen = ret;
                assert((IP_HL(ip) * 4) < packet->pcap_hdr->caplen);

                return PACKET_REASM;
        }

        return ret;
}



static int basic_checksum(unsigned short *elmt, int nb)
{
        int sum = 0, cnt = nb;
        uint16_t last = 0, *item = elmt;

        while ( cnt > 1 ) {
                sum += align_uint16(item++) & 0xffff;
                cnt -= 2;
        }
        
        if ( cnt == 1 ) {
                memcpy(&last, item, sizeof(char));
                sum += last;
        }
        
        return sum;
}




static inline int cksum_complement(int sum)
{
        int tot;
        uint16_t upper, lower;
        
        upper = sum >> 16;
        lower = sum & 0xffff;
        
        tot = lower + upper;
        tot = tot + (tot >> 16);
        
        return ~tot & 0xffff;
}




static void capture_ip(packet_container_t *packet, unsigned char *p, proto_enum_t ip_kind)
{
        iphdr_t *ip_hdr;
        uint16_t off, len, hlen, tmp;
        int ret = -1, sum = 0, do_reasm = 1;

        if ( bytes_are_in_frame("IP", packet, sizeof(iphdr_t)) < 0 )
                return;

        ip_hdr = (iphdr_t *) p;
        if ( IP_V(ip_hdr) != 4 ) 
                return;
        
        packet_add_header(packet, p, sizeof(iphdr_t), ip_kind);
        packet->network_layer_depth = packet->depth; 

        off = extract_uint16(&ip_hdr->ip_off);
        len = extract_uint16(&ip_hdr->ip_len);
        hlen = IP_HL(ip_hdr) * 4;


        if ( hlen < sizeof(iphdr_t) ) {
                nids_alert(NULL, packet, setup_alert("False IP header len"),
                           "Header len for this IP header is %u when it should be >= %d", hlen, sizeof(iphdr_t));
                return;
        }
        


        if ( len > packet->pcap_hdr->caplen ) {
                nids_alert(NULL, packet, setup_alert("truncated IP"),
                           "truncated IP (len=%u > caplen=%u)", len, packet->pcap_hdr->caplen);
                return;
        }

        /*
         * we don't want low level layer junk at the end of our packet. 
         */
        packet->pcap_hdr->caplen = len;

        if ( hlen > packet->pcap_hdr->caplen ) {
                nids_alert(NULL, packet, setup_alert("bad IP header length"),
                           "bad IP header length (hlen=%u > caplen=%u)", hlen, packet->pcap_hdr->caplen);
                return;
        }

        if ( ! packet->allocated_data ) {
                /*
                 * IP checksum verification - only if not a reassembled datagram.
                 */
                sum = basic_checksum((unsigned short *)p, hlen);
                sum = cksum_complement(sum);
                if ( sum != 0 )
                        return;
        }

        if ( ip_kind == p_ip ) 
                detect_plugins_run(packet, &ip_plugins_list, "ip");
        else
                detect_plugins_run(packet, &ipencap_plugins_list, "ipencap");

        
        if ( hlen - sizeof(iphdr_t) > 0)
                capture_ip_options(packet, p + sizeof(iphdr_t), hlen - sizeof(iphdr_t), &do_reasm);
        
	if ( do_reasm && off & 0x3fff ) {
                ret = handle_ip_fragment(packet, ip_hdr, p, hlen, len);
                if ( ret == PACKET_REASM ) {
                        packet->depth--;
                        capture_ip(packet, packet->allocated_data, ip_kind);
                        return;
                }
                
                else if ( ret < 0 )
                        return;
        }
      
        packet->pcap_hdr->caplen -= hlen;

        if ( ret == FRAGMENT_QUEUED ) {
                capture_data(packet, p + hlen, packet->pcap_hdr->caplen);
                return;
        }
        
        
     	switch (ip_hdr->ip_p) {


	case IPPROTO_IP:
                /* Ip encapsulated in Ip */
                capture_ip(packet, p + hlen, p_ipencap);
		break;
                
	case IPPROTO_TCP:
                tmp = len - hlen;
                
                sum = basic_checksum((unsigned short *)&p[hlen - 8], tmp + 8);
                sum += htons(6 + tmp);
                sum = cksum_complement(sum);
                
                if ( sum != 0 ) 
                        return;
                
                capture_tcp(packet, p + hlen);
		break;
                
	case IPPROTO_UDP: 
                capture_udp(packet, p + hlen);
		break;
                
	case IPPROTO_ICMP:
                capture_icmp(packet, p + hlen);
		break;

	case IPPROTO_IGMP:
                capture_igmp(packet, p + hlen);
		break;
	default:
                capture_data(packet, p + hlen, packet->pcap_hdr->caplen);
		break;
	}

        signature_engine_process_packet(signature_engine_get_ip_root(), packet);
}



__inline__ static void switch_ethertype(packet_container_t *packet, uint16_t ethertype, unsigned char *p) 
{        
        switch (ethertype) {
                
	case ETHERTYPE_IP:
                capture_ip(packet, p, p_ip);
		break;
                
	case ETHERTYPE_ARP:
                capture_arp(packet, p);
		break;
                
	case ETHERTYPE_REVARP:
                capture_revarp(packet, p);
		break;
                
	default:
                capture_data(packet, p, packet->pcap_hdr->caplen);
                break;
	}
}



void capture_ethernet(u_char *user, struct pcap_pkthdr *h, u_char *p)
{
        uint16_t type;
        packet_container_t *packet;
        
        if ( bytes_are_in_frame_nopkt("ETHER", h->caplen, sizeof(etherhdr_t)) < 0 )
                return;

        packet = packet_new(p);
        if ( ! packet )
                return;

        packet->pcap_hdr = h;
        
        packet_add_header(packet, p, sizeof(etherhdr_t), p_ether);
        h->caplen -= sizeof(etherhdr_t);
        
	detect_plugins_run(packet, &ether_plugins_list, "ether");

        type = extract_uint16(&packet->packet[packet->depth].p.ether_hdr->ether_type);
        switch_ethertype(packet, type, p + sizeof(etherhdr_t));
        
        detect_plugins_run(packet, &all_plugins_list, "all");
        packet_release(packet);
}




void capture_null(unsigned char *user, struct pcap_pkthdr *h, unsigned char *p)
{
        packet_container_t *packet;

        if ( bytes_are_in_frame_nopkt("NULL", h->caplen, 4) < 0 )
                return;
        
        packet = packet_new(p);
        if ( ! packet )
                return;

        h->caplen -= 4;
        packet->pcap_hdr = h;
        
        packet_add_header(packet, p, sizeof(uint32_t), p_null);
        capture_ip(packet, p + 4, p_ip);
        
        detect_plugins_run(packet, &all_plugins_list, "all");
        packet_release(packet);
}



#define PPP_ADDRESS     0xff    /* The address byte value */
#define PPP_CONTROL     0x03    /* The control byte value */
#define PPP_IP          0x0021  /* Raw IP */

void capture_ppp(unsigned char *user, struct pcap_pkthdr *h, unsigned char *p)
{
        uint16_t proto;
        packet_container_t *packet;
        
        if ( bytes_are_in_frame_nopkt("PPP", h->caplen, 4) < 0 )
                return;

        if ( *p == PPP_ADDRESS && *(p + 1) == PPP_CONTROL ) {
                p += 2;                 /* ACFC not used */
                h->caplen -= 2;
        }
        
        if ( *p % 2 ) {
                proto = *p++;             /* PFC is used */
                h->caplen--;
        } else {
                proto = extract_uint16(p);
                p += 2;
                h->caplen -= 2;
        }

        if ( proto != ETHERTYPE_IP && proto != PPP_IP )
                return;
        
        packet = packet_new(p);
        if ( ! packet )
                return;

        packet->pcap_hdr = h;
        capture_ip(packet, p, p_ip);
        
        detect_plugins_run(packet, &all_plugins_list, "all");
        packet_release(packet);
}




void capture_slip(unsigned char *user, struct pcap_pkthdr *h, unsigned char *p)
{
	packet_container_t *packet;
	
	if ( bytes_are_in_frame_nopkt("SLIP", h->caplen, 16) < 0)
		return ;

	packet = packet_new(p);
	if ( ! packet )
		return;
	
	h->caplen -= 16;
	packet->pcap_hdr = h;
	capture_ip(packet, p + 16, p_ip);
	
	detect_plugins_run(packet, &all_plugins_list, "all");
	packet_release(packet);
}



void capture_fddi(unsigned char *user, struct pcap_pkthdr *h, unsigned char *p)
{
        packet_container_t *packet;

        if ( bytes_are_in_frame_nopkt("FDDI", h->caplen, FDDI_HDRLEN) < 0 )
                return;
        
        packet = packet_new(p);
        if ( ! packet )
                return;

        packet->pcap_hdr = h;
        h->caplen -= FDDI_HDRLEN;
        capture_ip(packet, p + FDDI_HDRLEN, p_ip);

        detect_plugins_run(packet, &all_plugins_list, "all");
        packet_release(packet);
}




void capture_raw(unsigned char *user, struct pcap_pkthdr *h, unsigned char *p)
{
        const iphdr_t *ip;
        packet_container_t *packet;

        if ( bytes_are_in_frame_nopkt("RAW", h->caplen, 20) < 0 )
                return;
        
        ip = (const iphdr_t *) p;
        if ( IP_V(ip) != 4 )
                /*
                 * we don't support IPv6 yet.
                 */
                return;
        
        packet = packet_new(p);
        if ( ! packet )
                return;
        
        packet->pcap_hdr = h;
        capture_ip(packet, p, p_ip);

        detect_plugins_run(packet, &all_plugins_list, "all");
        packet_release(packet);
}




void capture_clip(unsigned char *user, struct pcap_pkthdr *h, unsigned char *p)
{
        packet_container_t *packet;
        
        packet = packet_new(p);
        if ( ! packet )
                return;

        packet->pcap_hdr = h;
        capture_ip(packet, p, p_ip);

        detect_plugins_run(packet, &all_plugins_list, "all");
        packet_release(packet);
}




void capture_sll(unsigned char *user, struct pcap_pkthdr *h, unsigned char *p) 
{
        packet_container_t *packet;
        
        if ( bytes_are_in_frame_nopkt("SLL", h->caplen, 16) < 0 )
                return;

	packet = packet_new(p);
        if ( ! packet )
                return;
        
	h->caplen -= 16;
        packet->pcap_hdr = h;
        switch_ethertype(packet, extract_uint16(p + 14), p + 16);
        
        detect_plugins_run(packet, &all_plugins_list, "all");
        packet_release(packet);
}




void packet_release(packet_container_t *packet)
{
        assert( packet->refcount > 0 );
        
        if ( --packet->refcount == 0 ) {

                if ( packet->allocated_data )
                        free(packet->allocated_data);

                if ( packet->tcp_allocated_data )
                        free(packet->tcp_allocated_data);

                if ( packet->captured_data )
                        pcap_header_free(packet->pcap_hdr);
                
                free(packet);
                
        }
}



void packet_lock(packet_container_t *packet) 
{
        packet->refcount++;
}







