/*****
*
* Copyright (C) 1999,2000, 2002, 2003 Yoann Vandoorselaere <yoann@prelude-ids.org>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <inttypes.h>

#include "packet.h"

#include <libprelude/list.h>
#include <libprelude/prelude-log.h>
#include <libprelude/plugin-common.h>
#include <libprelude/timer.h>
#include <libprelude/extract.h>

#include "optparse.h"
#include "nids-alert.h"

#define PROTO_IP  0
#define PROTO_TCP 1


static idmef_string_t class;
static packet_container_t *pkt;


static nids_alert_t *setup_opt_alert(void) 
{
        static nids_alert_t alert;
        static int initialized = 0;
        static idmef_impact_t impact;
        
        if ( ! initialized ) {
                initialized = 1;
                nids_alert_init(&alert);
                
                alert.impact = &impact;
                impact.type = dos;
                impact.completion = 0; /* we don't know */
                impact.severity = impact_medium;
                idmef_string_set_constant(&impact.description, "Might result in an operating system crash");
        }
        
        alert.classification.name.len = class.len;
        alert.classification.name.string = class.string;
        
        return &alert;
}




/*
 * From rfc793 :
 * Options may occupy space at the end of the TCP header and are a
 * multiple of 8 bits in length.  All options are included in the
 * checksum.  An option may begin on any octet boundary.  There are two
 * cases for the format of an option:
 *
 *  Case 1:  A single octet of option-kind.
 *
 *  Case 2:  An octet of option-kind, an octet of option-length, and
 *           the actual option-data octets.
 *
 * The option-length counts the two octets of option-kind and
 * option-length as well as the option-data octets.
 *
 * Note that the list of options may be shorter than the data offset
 * field might imply.  The content of the header beyond the
 * End-of-Option option must be header padding (i.e., zero).
 *
 * A TCP must implement all options.
 */



/*
 * Verify if the option 'opt' is one of
 * the 1 byte only option (nop || eol).
 */
static int is_1byte_option(int opt) 
{
        if ( opt == TCPOPT_NOP ) {
                return 0;
        }

        else if (opt == TCPOPT_EOL) {
                return 0;
        }

        return -1;
}




static int get_tcp_timestamp(int optlen, const unsigned char *optbuf, const char *proto) 
{
        if ( optlen != TCPOLEN_TIMESTAMP ) {
                nids_alert(NULL, pkt, setup_opt_alert(),
                           "%s TIMESTAMP options not valid : option len (%d) != %d",
                           proto, optlen, TCPOLEN_TIMESTAMP);
                return -1;
        }

        /*
         * size of the buffer is already tested in is_option_valid().
         */
        pkt->paws_tsval = extract_uint32(optbuf);
        pkt->paws_tsecr = extract_uint32(optbuf + 4);

        return 0;
}





static int remember_needed_option(uint8_t proto, int opt, int optlen,
                                  const unsigned char *optbuf, int totlen, const char *proto_str) 
{
        int ret = 0;
        
        if ( proto != PROTO_TCP )
                return 0;
        
        switch (opt) {

        case TCPOPT_TIMESTAMP:
                ret = get_tcp_timestamp(optlen, optbuf, proto_str);
                break;

        case TCPOPT_SACK:                
        case TCPOPT_SACK_PERMITTED:
                pkt->tcp_sack = 1;
                break;
                
        default:
                break;
        }

        return ret;
}




/*
 * Verify that an option is valid :
 * - verify that this option len is > 2 bytes.
 * - verify that this option len is < than our total option len.
 * - do some bound check on our option buffer, to avoid going out of bound.
 */
static int is_option_valid(const unsigned char *optbuf, int optlen, int totlen, const char *proto) 
{        
        if ( optlen < 2 ) {
                nids_alert(NULL, pkt, setup_opt_alert(),
                           "%s options not valid : options is not \"nop\" or \"eol\" so option len (%d) should be >= 2",
                           proto, optlen);
                return -1;
        }
        
        if ( optlen > totlen ) {
                nids_alert(NULL, pkt, setup_opt_alert(),
                           "%s options not valid : option len (%d) is > remaining total options len (%d)",
                           proto, optlen, totlen);
                return -1;
        }

        /*
         * This check should never be reached because
         * of the optlen > totlen test... use an assert.
         */
        assert( (optbuf + (optlen - 2)) <= (optbuf + (totlen - 2)) );

        return 0;
}



/*
 * Verify that our total options len is big enough
 * to contain a len byte, which mean totlen must be
 * >= 2 (1 byte for optkind, and 1 for optlen).
 */
static int is_len_byte_ok(int totlen, const char *proto) 
{
        if ( totlen < 2 ) {
                nids_alert(NULL, pkt, setup_opt_alert(),
                           "%s options not valid : options is not \"nop\" or \"eol\" "
                           "so total options len (%d) should be >= 2 (1 byte kind, 1 "
                           "byte len)", proto, totlen);
                return -1;
        }
        
        return 0;
}


/*
 * Walk options in 'optbuf' of total len 'totlen'.
 * Return the number of byte parsed before error.
 */
static int walk_options(uint8_t proto, const unsigned char *optbuf, int totlen, const char *proto_str) 
{
        int origlen = totlen;
        int opt, optlen, ret;
        
        do {
                opt = *optbuf++;

                if ( is_1byte_option(opt) == 0 )
                        totlen -= 1;
                else {
                        if ( is_len_byte_ok(totlen, proto_str) < 0 )
                                return origlen - totlen;

                        optlen = *optbuf++;
                        
                        ret = is_option_valid(optbuf, optlen, totlen, proto_str);
                        if ( ret < 0 )
                                return origlen - (totlen - 2);

                        
                        ret = remember_needed_option(proto, opt, optlen, optbuf, totlen, proto_str);
                        if ( ret < 0 )
                                return origlen - (totlen - 2);

                        totlen -= optlen;
                        optbuf += optlen - 2;
                }
                
                assert(totlen >= 0);
                
        } while ( totlen != 0 );

        return origlen - totlen;
}



/*
 *
 */
int tcp_optdump(packet_container_t *packet, const unsigned char *optbuf, size_t totlen) 
{
        pkt = packet;
        idmef_string_set_constant(&class, "Invalid TCP Option");
        
        if ( totlen > MAX_OPTS_LEN ) {
                nids_alert(NULL, packet, setup_opt_alert(),
                           "TCP options not valid : total option len (%d) > maximum "
                           "option len (%d)", totlen, MAX_OPTS_LEN);
                return -1;
        }
        
        return walk_options(PROTO_TCP, optbuf, totlen, "TCP");
}



/*
 *
 */
int ip_optdump(packet_container_t *packet, const unsigned char *optbuf, size_t totlen) 
{
        pkt = packet;
        idmef_string_set_constant(&class, "Invalid IP Option");
        
        if ( totlen > MAX_OPTS_LEN ) {
                nids_alert(NULL, packet, setup_opt_alert(),
                           "IP options not valid, total option len (%d) is bigger than maximum option len (%d)",
                           totlen, MAX_OPTS_LEN);
                
                return -1;
        }
        
        return walk_options(PROTO_IP, optbuf, totlen, "IP");
}



int option_is_set(int wanted_option, const unsigned char *optbuf, size_t totlen) 
{
        int opt, optlen;
        
        do {
                opt = *optbuf++;
                if ( opt == wanted_option )
                        return 0;
                
                if ( is_1byte_option(opt) == 0 )
                        totlen -= 1;
                else {
                        if ( totlen < 2 )
                                return -1;

                        optlen = *optbuf++;

                        if ( optlen < 2 || optlen > totlen )
                                return -1;
                        
                        totlen -= optlen;
                        optbuf += optlen - 2;

                }
                
                assert(totlen >= 0);
        } while ( totlen != 0 );

        return -1;
}
























