/*****
*
* Copyright (C) 1998, 1999, 2000, 2002 Yoann Vandoorselaere <yoann@prelude-ids.org>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <dirent.h>
#include <errno.h>
#include <limits.h> /* for NAME_MAX */
#include <sys/time.h>

#include "packet.h"

#include <libprelude/list.h> 
#include <libprelude/prelude-log.h>
#include <libprelude/plugin-common.h>
#include <libprelude/plugin-common-prv.h>

#include "plugin-detect.h"
#include "plugin-protocol.h"


LIST_HEAD(tcp_plugins_list);
LIST_HEAD(udp_plugins_list);
LIST_HEAD(ip_plugins_list);
LIST_HEAD(ipencap_plugins_list);
LIST_HEAD(data_plugins_list);
LIST_HEAD(icmp_plugins_list);
LIST_HEAD(igmp_plugins_list);
LIST_HEAD(arp_plugins_list);
LIST_HEAD(rarp_plugins_list);
LIST_HEAD(ether_plugins_list);
LIST_HEAD(all_plugins_list);



static const char *register_to_plugin_provided_protocol(plugin_container_t *pc, const char *proto) 
{
        plugin_protocol_t *p;

        p = protocol_plugins_search(proto);
        if ( ! p )
                return NULL;

        plugin_add(pc, &p->detect_plugin_list, proto);

        return proto;
}




/*
 * Try to register plugin 'plugin' to protocol 'wanted_proto',
 * return -1 on error, 0 on success.
 */
static const char *register_to_internal_protocol(plugin_container_t *pc, int wanted_proto)
{
        int i;
        static struct {
                const char *proto;
                proto_enum_t type;
                struct list_head *plugin_list;
        } tbl[] = {
                { "IP", p_ip, &ip_plugins_list                },
                { "IPENCAP", p_ipencap, &ipencap_plugins_list },
                { "TCP", p_tcp, &tcp_plugins_list             },
                { "UDP", p_udp, &udp_plugins_list             },
                { "ICMP", p_icmp, &icmp_plugins_list          },
                { "IGMP", p_igmp, &igmp_plugins_list          },
                { "ARP", p_arp, &arp_plugins_list             },
                { "RARP", p_rarp, &rarp_plugins_list          },
                { "ETHER", p_ether, &ether_plugins_list       },
                { "DATA", p_data, &data_plugins_list          },
                { "ALL", p_all, &all_plugins_list             },
                { NULL, 0, NULL                               }
        };
        
        for ( i = 0; tbl[i].proto != NULL; i++ )
                if ( tbl[i].type == wanted_proto ) {
                        
                        plugin_add(pc, tbl[i].plugin_list, tbl[i].proto);
                        return tbl[i].proto;
                }
        
        return NULL;
}



/*
 * Subscribe plugins to it's specified protocol,
 * if the plugin need to subscribe to > 1 protocol :
 * we need to copy it.
 */
static int subscribe(plugin_container_t *pc)
{
	int i, len = 0;
        char buf[8192];
        const char *proto;
        plugin_detect_t *plugin = (plugin_detect_t *) pc->plugin;

        /*
         * set the first byte to zero, so we don't have problem
         * when printing the buffer, if there was no subscribtion.
         */
        buf[0] = '\0'; 
        
	for ( i = 0; plugin->subscribtion[i].type != p_end; i++ ) {
                if ( i > 0 )
                        len += snprintf(buf + len, sizeof(buf) - len, ",");

                if ( plugin->subscribtion[i].type != p_external ) 
                        proto = register_to_internal_protocol(pc, plugin->subscribtion[i].type);
                else
                        proto = register_to_plugin_provided_protocol(pc, plugin->subscribtion[i].filter);
                
                if ( proto )
                        len += snprintf(buf + len, sizeof(buf) - len, "%s", proto);
                else
                        len += snprintf(buf + len, sizeof(buf) - len, "unknow proto %d", plugin->subscribtion[i].type);
        }
        
        log(LOG_INFO, "- %s subscribed to : \"[%s]\".\n", plugin->name, buf);

        return 0;
}



static void unsubscribe(plugin_container_t *pc) 
{
        log(LOG_INFO, "- %s unsubscribed from %s.\n", pc->plugin->name, pc->infos);
        plugin_del(pc);
}





/*
 * Start all plugins of kind 'list'.
 */
void detect_plugins_run(packet_container_t *packet, struct list_head *list, char *proto)
{
        struct list_head *tmp;
        plugin_container_t *pc;

        list_for_each(tmp, list) {
                pc = list_entry(tmp, plugin_container_t, ext_list);
                plugin_run(pc, plugin_detect_t, run, packet);
        }
}



/*
 * Open the plugin directory (dirname),
 * and try to load all plugins located int it.
 */
int detect_plugins_init(const char *dirname, int argc, char **argv)
{
        int ret;

	ret = access(dirname, F_OK);
	if ( ret < 0 ) {
		if ( errno == ENOENT )
			return 0;
		log(LOG_ERR, "can't access %s.\n", dirname);
		return -1;
	}

        ret = plugin_load_from_dir(dirname, argc, argv, subscribe, unsubscribe);
        if ( ret < 0 ) {
                log(LOG_ERR, "couldn't load plugin subsystem.\n");
                return -1;
        }

        return ret;
}


