/*****
*
* Copyright (C) 2001, 2002 Yoann Vandoorselaere <yoann@prelude-ids.org>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
* Written by Yoann Vandoorselaere <yoann@prelude-ids.org>
*
*****/


#include <stdio.h>
#include <assert.h>

#include "protocol.h"


/*
 * define the telnet negotiation character that we're interested in
 */

#define INTERPRET_AS_COMMAND 0xff
#define SUBOPTION_BEGIN      0xfa
#define SUBOPTION_END        0xf0
#define OPTION_NOP           0xf1

#define OPTION_NOP_LENGTH       1
#define OPTION_STANDARD_LENGTH  2



static int is_enabled = 0;
static port_list_t *port_list;



static int decode_telnet(unsigned char *ptr, unsigned char *end) 
{
        unsigned char *write_ptr;
        int iac_met = 0, skip = 0;
        
        write_ptr = ptr;
        
        while ( ptr < end ) {
                
                if ( *ptr == INTERPRET_AS_COMMAND ) {
                        if ( iac_met ) {
                                /*
                                 * if we meet IAC two time, it is escaped
                                 */
                                iac_met = 0;
                                *write_ptr++ = *ptr++;
                                
                        } else {
                                iac_met = 1; 
                                ptr += 1;
                                skip++;
                        }
                }

                else if ( iac_met && *ptr == SUBOPTION_BEGIN ) {
                        iac_met = 0;

                        while ( ptr < end && *ptr != SUBOPTION_END ) {
                                skip++;
                                ptr++;
                        }
                        
                        skip++;
                        ptr++;
                }

                else if ( iac_met ) {                        
                        if ( *ptr == OPTION_NOP ) {
                                ptr += OPTION_NOP_LENGTH;
                                skip += OPTION_NOP_LENGTH;
                        } else {
                                ptr += OPTION_STANDARD_LENGTH;
                                skip += OPTION_STANDARD_LENGTH;
                        }
                        
                        iac_met = 0;
                }

                else
                        *write_ptr++ = *ptr++;
        }
        
        return skip;
}




static int decode_packet(packet_container_t *p, unsigned char *data, int len)
{
        int ret;
        uint16_t dport;
        int depth = p->transport_layer_depth;
        
        if ( depth < 0 || p->packet[depth].proto != p_tcp )
                return -1;

        dport = extract_uint16(&p->packet[depth].p.tcp->th_dport);
                
        ret = protocol_plugin_is_port_ok(port_list, dport);
        if ( ret < 0 )
                return -1;

        /*
         * negotiation strings are at least 2 bytes long
         */
        if ( len < 2 )
                return -1;
        
        ret = decode_telnet(data, data + len);
        if ( ! ret )
                return -1;
        
        return 0;
}



static plugin_protocol_t plugin;



static int set_port_list(prelude_option_t *opt, const char *optarg) 
{
        protocol_plugin_add_string_port_to_list(port_list, optarg);
        return prelude_option_success;
}




static void setup_default_port(void) 
{
        protocol_plugin_add_port_to_list(port_list, 21);
	protocol_plugin_add_port_to_list(port_list, 23);
}



static int set_telnet_state(prelude_option_t *opt, const char *optarg) 
{
        int ret;
        
        if ( is_enabled == 1 ) {
                ret = plugin_unsubscribe((plugin_generic_t *) &plugin);
                if ( ret < 0 )
                        return prelude_option_error;        
                is_enabled = 0;
        }

        else {
                ret = plugin_subscribe((plugin_generic_t *) &plugin);
                if ( ret < 0 )
                        return prelude_option_error;
                is_enabled = 1;
        }
        
        return prelude_option_success;
}




static int get_telnet_state(char *buf, size_t size) 
{
        snprintf(buf, size, "%s", (is_enabled == 1) ? "enabled" : "disabled");
        return prelude_option_success;
}



plugin_generic_t *plugin_init(int argc, char **argv)
{
        prelude_option_t *opt;

        opt = prelude_option_add(NULL, CLI_HOOK|CFG_HOOK|WIDE_HOOK, 0, "telnetmod",
                                 "Set TelnetMod plugin option", no_argument,
                                 set_telnet_state, get_telnet_state);

        prelude_option_add(opt, CLI_HOOK|CFG_HOOK|WIDE_HOOK, 'p', "port-list",
                           "List of port this plugin should look at", required_argument,
                           set_port_list, NULL);
        
        port_list = protocol_plugin_port_list_new();
        setup_default_port();
        
        plugin_set_name(&plugin, "TelnetMod");        
        plugin_set_protocol(&plugin, "telnet");
        plugin_set_author(&plugin, "Yoann Vandoorselaere");
        plugin_set_contact(&plugin, "yoann@prelude-ids.org");
        plugin_set_desc(&plugin, "Decode the Telnet negotiation characters.");
        plugin_set_running_func(&plugin, decode_packet);
        
        return (plugin_generic_t *) &plugin;
}

