/*****
*
* Copyright (C) 2002 Krzysztof Zaraska <kzaraska@student.uci.agh.edu.pl>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
* Written by Krzysztof Zaraska <kzaraska@student.uci.agh.edu.pl>
*
*****/


#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <sys/types.h>
#include <ctype.h>
#include <string.h>

#include <libprelude/prelude-io.h>
#include <libprelude/prelude-message.h>
#include <libprelude/prelude-log.h>
#include <libprelude/prelude-getopt.h>

#include "unicode-to-ascii.h"


static char codes[65536];


/*
 * default to US codepage.
 */
static char *cp_file = NULL;
static int requested_cp = 437;



/*
 * statistics
 */
static int line = 0;
static int uchars = 0;
static int ignored = 0;
static int errors = 0;



static void get_code_for_codepage(uint16_t unichar, const char *curr)
{
	int code, cp, ret;
	
	while ( (curr = strchr(curr, ' ')) ) {
                        
		while ( ! isalnum(*curr) && *curr != '\0' )
			curr++;

	        if ( *curr == '\0' )
 	               break;
                        
		ret = sscanf(curr, "%d:%2x", &cp, &code);
		if ( ret != 2 ) {
			log(LOG_ERR, "line %d: warning: no codepage/character pair\n", line);
			errors++;
			break;
		} 
			
		if ( cp == requested_cp ) 
			codes[unichar] = code;
	}	
}




int unicode_load_table(void)
{
	FILE *fd;
        int i, ret;
	char buf[1024];
	uint16_t unichar;
	
	if ( ! cp_file ) {
		log(LOG_ERR, "codepage file not specified.\n");
		return -1;
	}
        
	fd = fopen(cp_file, "r");
	if ( ! fd ) {
		log(LOG_ERR, "could not open codepage table: file %s\n", cp_file);
		return -1;
	}	

        free(cp_file);
        memset(codes, 0, sizeof(codes));
        
	while ( fgets(buf, sizeof(buf), fd) ) {
		line++;
		
		/*
                 * skip the comments
                 */
		for ( i = 0; buf[i] == ' '; i++ );

                if ( (buf[i] == '#') || (buf[i] == '\n') || (buf[i] == '\r' ) )
                        continue;
		
		uchars++;
		
		ret = sscanf(buf, "%hx", &unichar);
		if (ret != 1) {
			log(LOG_ERR, "line %d: warning: no Unicode character specified\n", line);
			ignored++;
			continue;
		}
		
		get_code_for_codepage(unichar, buf);
	}
        
	fclose(fd);
	log(LOG_INFO, "- Done loading Unicode table (%d Unichars, %d ignored, %d with errors)\n", uchars, ignored, errors);

        for ( i = 0; i < 128; i++ )
                codes[i] = i;
	
	return 0;
}




int unicode_to_ascii(unsigned char *dst, uint16_t unichar) 
{
        if ( unichar > sizeof(codes) ) {
                log(LOG_ERR, "Invalid UTF-8 character (value=%d).\n", unichar);
                return -1;
        }
        
        *dst = codes[unichar];

        return 0;
}



int unicode_set_codepage(prelude_option_t *opt, const char *optarg)
{
	int ret;
	
	ret = sscanf(optarg, "%d", &requested_cp);	
	if ( ret == 1 )
                return prelude_option_success;
	else
                return prelude_option_error;
}



int unicode_set_table_file(prelude_option_t *opt, const char *optarg)
{
	cp_file = strdup(optarg);
	return prelude_option_success;
}
