/*****
*
* Copyright (C) 2001, 2002, 2003 Jeremie Brebec <flagg@ifrance.com>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
* Written by Jeremie Brebec <flagg@ifrance.com>
* Written by Yoann Vandoorselaere <yoann@prelude-ids.org>
*
*****/

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <errno.h>

#include "detect.h"

#include <libprelude/list.h>
#include <libprelude/prelude-log.h>
#include <libprelude/idmef-tree.h>

#include "rules.h"
#include "rules-operations.h"
#include "rules-parsing.h"
#include "rules-default.h"
#include "snort-keys.h"
#include "tcp-stream.h"

#define RULE_BYTE 3


/* BASE */
#define PRIORITY_IP_DST         3
#define PRIORITY_IP_SRC         4

#define PRIORITY_TCP_FLOW       0
#define PRIORITY_PORT_DST       1
#define PRIORITY_PORT_SRC       2

/* IP */
#define PRIORITY_IPPROTO       10
#define PRIORITY_ID            11
#define PRIORITY_TOS           12
#define PRIORITY_TTL           13
#define PRIORITY_FRAGOFFSET    8
#define PRIORITY_FRAGBITS      9

#define PRIORITY_IPOPTIONS     24
#define PRIORITY_DSIZE         25

/* ICMP */
#define PRIORITY_ITYPE         20
#define PRIORITY_ICODE         21
#define PRIORITY_ICMP_ID       27
#define PRIORITY_ICMP_SEQ      28

/* TCP */
#define PRIORITY_TCP_FLAGS     10

#define PRIORITY_WIN           25
#define PRIORITY_SEQ           27
#define PRIORITY_ACK           28


/*
 *
 */
#define PRIORITY_CONTENT_LIST 29


typedef enum {
        mode_all,
        mode_any,
        mode_not,
        mode_exact
} flags_mode_t;


typedef struct {
        idmef_classification_origin_t origin;
        const char *name;
        const char *url;
        struct list_head list;
} reference_t;



int data_msg_id;
int data_sid_id;
int data_reference_id;
int data_classtype_id;
int data_revision_id;


static int id_tcp_flow;
static int id_content_list;
static int id_ip_src, id_ip_dst;
static int id_port_src, id_port_dst;
static int id_tcp_flags, id_ttl, id_tos, id_id, id_dsize, id_win;
static int id_seq, id_ack, id_ipproto;
static int id_itype, id_icode, id_icmp_id, id_icmp_seq, id_fragbits, id_fragoffset;


static LIST_HEAD(reference_list);
static LIST_HEAD(classtype_list);


#define BYTE_OPERATOR_SMALLER   0x01
#define BYTE_OPERATOR_BIGGER    0x02
#define BYTE_OPERATOR_EQUAL     0x04
#define BYTE_OPERATOR_NOT_EQUAL 0x08
#define BYTE_OPERATOR_AND       0x10
#define BYTE_OPERATOR_OR        0x20

#define BYTE_STRING             0x01
#define BYTE_RELATIVE           0x02
#define BYTE_ALIGN              0x04


typedef struct {
        int bytes_offset;
        int bytes_to_convert;
        int value_to_compare;
        int8_t operator;
        int8_t flags;
        
        int base;
} byte_test_t;




static int byte_compare(int value, int operator, int value_to_compare) 
{
        int ret = -1;
        
        if ( operator & BYTE_OPERATOR_EQUAL ) 
                ret = (value == value_to_compare) ? 0 : -1;
        
        else if ( operator & BYTE_OPERATOR_NOT_EQUAL ) 
                ret = (value != value_to_compare) ? 0 : -1;
        
        else if ( operator & BYTE_OPERATOR_BIGGER )
                ret = (value > value_to_compare) ? 0 : -1;
        
        else if ( operator & BYTE_OPERATOR_SMALLER )
                ret = (value < value_to_compare) ? 0 : -1;
        
        else if ( operator & BYTE_OPERATOR_AND )
                ret = (value & value_to_compare) ? 0 : -1;
        
        else if ( operator & BYTE_OPERATOR_OR ) 
                ret = (value ^ value_to_compare) ? 0 : -1;
        
        return ret;
}




static int match_byte_generic(packet_container_t *packet, byte_test_t *test, uint32_t *value) 
{
        uint16_t dlen;
        unsigned char *data;
        int last_matched_offset;
        int depth = packet->application_layer_depth;
        
        if ( depth < 0 )
                return -1;

        assert(packet->packet[depth].proto == p_data);
        
        dlen = packet->packet[depth].len;
        data = packet->packet[depth].p.data;
        
        if ( test->flags & BYTE_RELATIVE ) {
                last_matched_offset = signature_engine_get_last_matched_offset();
                dlen -= last_matched_offset;
                data += last_matched_offset;
        }
                
        if ( (test->bytes_offset + test->bytes_to_convert) > dlen )
                return -1;

        data += test->bytes_offset;
        dlen -= test->bytes_offset;
        
        if ( test->flags & BYTE_STRING ) {
                char buf[test->bytes_to_convert + 1];

                memcpy(buf, data, test->bytes_to_convert);
                buf[test->bytes_to_convert] = '\0';
                
                *value = htonl(strtoul(buf, NULL, test->base));
        }
        
        else if ( test->bytes_to_convert == 1 )
                *value = *data;
        
        else if ( test->bytes_to_convert == 2 )
                *value = align_uint16(data);

        else if ( test->bytes_to_convert == 4 )
                *value = align_uint32(data);

        return 0;
}



static int match_byte_test(packet_container_t *packet, void *ptr) 
{
        int ret;
        uint32_t value;
        byte_test_t *test = ptr;
        
        ret = match_byte_generic(packet, test, &value);
        if ( ret < 0 )
                return -1;
        
        return byte_compare(value, test->flags, test->value_to_compare);
}




static int match_byte_jump(packet_container_t *packet, void *ptr) 
{
        int ret;
        uint32_t value, dlen;
        byte_test_t *test = ptr;
        uint32_t last_matched_offset = 0;
        
        dlen = packet->packet[packet->application_layer_depth].len;
        
        ret = match_byte_generic(packet, test, &value);
        if ( ret < 0 )
                return -1;

        if ( test->flags & BYTE_ALIGN && (value % 4) != 0 )
                value += 4 - (value % 4);

        if ( test->flags & BYTE_RELATIVE )
                last_matched_offset = signature_engine_get_last_matched_offset();
        
        value += last_matched_offset + test->bytes_offset + test->bytes_to_convert;
        if ( value >= dlen )
                return -1;

        signature_engine_set_last_matched_offset(value);
                
        return 0;
}



static int get_byte_operator(const char *str) 
{
        int ret = 0;
        
        switch (*str) {
                
            case '<':
                    ret = BYTE_OPERATOR_SMALLER;
                    break;
                    
            case '>':
                    ret = BYTE_OPERATOR_BIGGER;
                    break;
                    
            case '=':
                    ret = BYTE_OPERATOR_EQUAL;
                    break;
                    
            case '!':
                    ret = BYTE_OPERATOR_NOT_EQUAL;
                    break;
                    
            case '&':
                    ret = BYTE_OPERATOR_AND;
                    break;
                    
            case '^':
                    ret = BYTE_OPERATOR_OR;
                    break;
                    
            default:
                    signature_parser_set_error("Unknow byte_test operator %s.", str);
                    return -1;
        }
        
        return ret;
}




static int get_byte_key(const char *str, byte_test_t *test) 
{
        if ( strcmp(str, "relative") == 0 )
                test->flags |= BYTE_RELATIVE;
        
        else if ( strcmp(str, "big") == 0 )
                /* default is big endian and we already converted, do nothing */ ;
        
        else if ( strcmp(str, "little") == 0 ) {
                test->value_to_compare  = ( ( *(uint8_t *) &test->value_to_compare) & 0xff) << 24;
                test->value_to_compare |= ( ( *(uint8_t *) &test->value_to_compare) & 0xff) << 16;
                test->value_to_compare |= ( ( *(uint8_t *) &test->value_to_compare) & 0xff) <<  8;
                test->value_to_compare |= ( ( *(uint8_t *) &test->value_to_compare) & 0xff);
        }

        else if ( strcmp(str, "string") == 0 )
                test->flags |= BYTE_STRING;
                
        else if ( strcmp(str, "oct") == 0 )
                test->base = 8;
        
        else if ( strcmp(str, "dec") == 0 )
                test->base = 10;

        else if ( strcmp(str, "hex") == 0 )
                test->base = 16;
        
        else if ( strcmp(str, "align") == 0 )
                test->flags |= BYTE_ALIGN;

        else
                return -1;
        
        return 0;
}



static int parse_byte_test(parameters_t *param, rules_t **rules) 
{
        int ret;
        rule_t *rule;
        byte_test_t *new;

        new = calloc(1, sizeof(*new));
        if ( ! new ) {
                log(LOG_ERR, "memory exhausted\n");
                return -1;
        }

        new->bytes_to_convert = atoi(param->str);
        
        param = param->next;
        if ( ! param->next )
                return -1;
        
        new->operator = get_byte_operator(param->str);
        if ( new->operator < 0 )
                return -1;
        
        param = param->next;
        if ( ! param )
                return -1;

        new->value_to_compare = htonl(atoi(param->str));        

        param = param->next;
        if ( ! param )
                return -1;
        
        new->bytes_offset = atoi(param->str);

        param = param->next;
        while ( param ) {
                ret = get_byte_key(param->str, new);
                param = param->next;
        }

        rule = make_new_rule(0, NULL);
	add_rule_leaf_match(rule, RULE_BYTE, new, &match_byte_test);
	*rules = make_new_rules(rule, NULL);

        return 0; 
}



static int parse_byte_jump(parameters_t *param, rules_t **rules) 
{
        int ret;
        rule_t *rule;
        byte_test_t *new;

        new = calloc(1, sizeof(*new));
        if ( ! new ) {
                log(LOG_ERR, "memory exhausted\n");
                return -1;
        }

        new->bytes_to_convert = atoi(param->str);
        
        param = param->next;
        if ( ! param->next )
                return -1;
        
        new->bytes_offset = atoi(param->str);

        param = param->next;
        while ( param ) {
                ret = get_byte_key(param->str, new);
                param = param->next;
        }

        rule = make_new_rule(0, NULL);
	add_rule_leaf_match(rule, RULE_BYTE, new, &match_byte_jump);
	*rules = make_new_rules(rule, NULL);
        
        return 0; 
}



static int parse_sid(char *str, rules_t **rules) 
{
        uint32_t *sid;
        idmef_additional_data_t *data;
        
        sid = malloc(sizeof(*sid));
        if ( ! sid ) {
                log(LOG_ERR, "memory exhausted.\n");
                return -1;
        }
        
        data = malloc(sizeof(*data));
        if ( ! data ) {
                log(LOG_ERR, "memory exhausted.\n");
                return -1;
        }

        *sid = htonl(atoi(str));
        idmef_string_set_constant(&data->meaning, "Snort rule ID");
        idmef_additional_data_set_data(data, integer, sid, sizeof(*sid));
                
        *rules = make_new_rules(make_data_rule(data_sid_id, data), NULL);
        if ( ! *rules )
                return -1;
        
        return 0; 
}




static int parse_revision(char *str, rules_t **rules) 
{
        uint32_t *rev;
        idmef_additional_data_t *data;

        rev = malloc(sizeof(*rev));
        if ( ! rev ) {
                log(LOG_ERR, "memory exhausted.\n");
                return -1;
        }
        
        data = malloc(sizeof(*data));
        if ( ! data ) {
                log(LOG_ERR, "memory exhausted.\n");
                return -1;
        }

        *rev = htonl(atoi(str));
        idmef_string_set_constant(&data->meaning, "Snort rule revision");
        idmef_additional_data_set_data(data, integer, rev, sizeof(*rev));
                
        *rules = make_new_rules(make_data_rule(data_revision_id, data), NULL);
        if ( ! *rules )
                return -1;
        
        return 0;
}



static int parse_classtype(char *str, rules_t **rules) 
{
        classtype_t *class;
        struct list_head *tmp;
        
        list_for_each(tmp, &classtype_list) {
                class = list_entry(tmp, classtype_t, list);
                
                if ( strcmp(str, class->shortname) != 0 )
                        continue;
                
                *rules = make_new_rules(make_data_rule(data_classtype_id, class), NULL);
                if ( ! *rules )
                        return -1;

                return 0;
        }

        signature_parser_set_error("Unknow classtype %s.", str);

        return -1;
}




static int parse_msg(char *str, rules_t **rules) 
{
	data_msg_t *msg;
        
        msg = malloc(sizeof(data_msg_t));
	if ( ! msg ) {
                log(LOG_ERR, "memory exhausted.\n");
		return -1;
	}
        
	msg->msg = strdup(str);
	msg->len = strlen(str) + 1;

	*rules = make_new_rules(make_data_rule(data_msg_id, msg), NULL);
        
        return 0;

}



static int parse_reference(parameters_t *parameters, rules_t **rules) 
{
        char buf[256];
        reference_t *ref;
        const char *orig;
        data_reference_t *new;
        struct list_head *tmp;

        orig = parameters->str;
        parameters = parameters->next;
        
        list_for_each(tmp, &reference_list) {
                ref = list_entry(tmp, reference_t, list);
                
                if ( strcasecmp(orig, ref->name) != 0 )
                        continue;

                new = malloc(sizeof(*new));
                if ( ! new ) {
                        log(LOG_ERR, "memory exhausted.\n");
                        return -1;
                }
                
                snprintf(buf, sizeof(buf), "%s%s", ref->url, parameters->str);
                               
                new->url = strdup(buf);
                new->url_len = strlen(new->url) + 1;
                new->origin = ref->origin;
                
                *rules = make_new_rules(make_data_rule(data_reference_id, new), NULL);
                if ( ! *rules )
                        return -1;

                return 0;
        }

        signature_parser_set_error("Unknow reference %s.", parameters->str);

        return -1;
}




static int parse_port_type(char *str, segment_t **port) 
{
	char *s, *cstr;

        *port = NULL;
        
	if ( strncasecmp(str, "any", 3) == 0 ) 		
		return 0;

	*port = malloc(sizeof(segment_t));
	if ( ! port ) {
                log(LOG_ERR, "memory exhausted.\n");
		return -1;
	}

	cstr = strdup(str);
        
	s = strtok(cstr, ":");
	if ( *s != '\0' )
		(*port)->lo = atoi(s);
	else 
		(*port)->lo = 0;

	s = strtok(NULL, "");
        if ( ! s )
                (*port)->hi = (*port)->lo;
        else {
                if ( *s == '\0' )
                        (*port)->hi = 65535;
                else
                        (*port)->hi = atoi(s);
        }
        
	if ( (*port)->lo < 0 || (*port)->lo > 65535 || (*port)->hi < 0 || (*port)->hi > 65535 ) {
		signature_parser_set_error("Invalid port [%d:%d]", (*port)->lo, (*port)->hi);
		free(*port);
		free(cstr);
		return -1;
	}

	free(cstr);
        
	return 0;
}




static int parse_portsrc(char *str, rules_t **rules) 
{
        int ret;
        segment_t *port;
        
        ret = parse_port_type(str, &port);
        if ( ret < 0 ) {
                *rules = NULL;
                return -1;
        }

        if ( ! port ) { /* any rule */
                delete_rules(*rules);
                *rules = NULL;
                return 0;
        }

        *rules = make_new_rules(make_new_rule(id_port_src, port), *rules);
        
	return 0;
}



static int parse_portdst(char *str, rules_t **rules) 
{
        int ret;
        segment_t *port;
        
        ret = parse_port_type(str, &port);
        if ( ret < 0 ) {
                *rules = NULL;
                return -1;
        }
                
        if ( ! port ) { /* "any" rule */
                *rules = NULL;
                return 0;
        }
                                     
        *rules = make_new_rules(make_new_rule(id_port_dst, port), *rules);
        
	return 0;
}




static int parse_ip_type(int type_id, char *str, rules_t **rules)
{
	ip_t *ip;
        
        if ( strcasecmp(str, "any") == 0 ) {
                *rules = NULL;
                return 0;
        }
        
        ip = parse_ip(str);
        if ( ! ip ) 
                return -1;

        *rules = make_new_rules(make_new_rule(type_id, ip), *rules);
                
	return 0;
}



static int parse_ip_src(char *str, rules_t **rules) 
{        
        return parse_ip_type(id_ip_src, str, rules);
}




static int parse_ip_dst(char *str, rules_t **rules) 
{
        return parse_ip_type(id_ip_dst, str, rules);
}



static int parse_ipopts(char *str, rules_t **rules) 
{
	rule_t *rule;
        flags_t *flag;
        int i = 0, r = 1;
	static struct {
		const char *name;
		int value;
	} ipopts_tbl[] = {
		{ "rr" ,   IPOPT_RR }, 
		{ "eol" ,  IPOPT_EOL },
		{ "nop",   IPOPT_NOP },
		{ "ts",    IPOPT_TIMESTAMP },
		{ "sec",   IPOPT_SECURITY },
		{ "lsrr",  IPOPT_LSRR },
		{ "lsrre", /*IPOPT_LSRRE*/ 0x84 },
		{ "ssrr",  IPOPT_SSRR },
		{ "satid", IPOPT_SATID },
		{ "ra",    IPOPT_RA },
		{ NULL,    0 }
	};
        
	while( ipopts_tbl[i].name != NULL && strcasecmp(ipopts_tbl[i].name, str) ) {
		i++;
		r = r << 1;
	}

	if ( ! ipopts_tbl[i].name ) {
		signature_parser_set_error("Unknow ipopt '%s'", str);
		return -1;
	}

	flag = malloc(sizeof(flags_t));
	if ( ! flag ) {
                log(LOG_ERR, "memory exhausted.\n");
		return -1;
	}

	flag->mask = r;
	flag->flags = ipopts_tbl[i].value;
        
	rule = make_new_rule(0, NULL);
	add_rule_leaf_match(rule, RULE_IPOPTS, flag, &signature_match_ipopts);
	*rules = make_new_rules(rule, NULL);

	return 0;
}




static rules_t *create_any_flags_rules(int flags, int mask, int test_id) 
{
        int i;
        rule_t *rule;
        flags_t *flag;
        rules_t *new_rules;
        
        new_rules = make_new_rules(NULL, NULL);
        
        for ( i = 1; i < 256; i = i << 1 ) {
                
                if ( ! (flags & i) )
                        continue;
                                        
                flag = malloc(sizeof(*flag));
                if ( ! flag ) {
                        log(LOG_ERR, "memory exhausted.\n");
                        return NULL;
                }
                        
                flag->mask = i & mask;
                flag->flags = i;
                        
                rule = make_new_rule(test_id, flag);
                new_rules = rule_and(new_rules, rule);
        }

        return new_rules;
}




static rules_t *create_not_flags_rules(int flags, int mask, int test_id) 
{
        flags_t *flag;
        
        flag = malloc(sizeof(flags_t));
        if ( ! flag ) {
                log(LOG_ERR, "memory exhausted.\n");
                return NULL;
        }
        
        flag->mask = flags & mask;
        flag->flags = 0;
        
        return make_new_rules(make_new_rule(test_id, flag), NULL);
}




static rules_t *create_all_flags_rules(int flags, int mask, int test_id) 
{
        flags_t *flag;
        
        flag = malloc(sizeof(flags_t));
        if ( ! flag ) {
                log(LOG_ERR, "memory exhausted.\n");
                return NULL;
        }
        
        flag->mask = flags & mask;
        flag->flags = flags;
        
        return make_new_rules(make_new_rule(test_id, flag), NULL);
}



static rules_t *create_exact_flags_rules(int flags, int mask, int test_id) 
{
        flags_t *flag;
        
        flag = malloc(sizeof(flags_t));
        if ( ! flag ) {
                log(LOG_ERR, "memory exhausted.\n");
                return NULL;
        }
        
        flag->mask = ~0 & mask;
        flag->flags = flags;
        
        return make_new_rules(make_new_rule(test_id, flag), NULL);
}




static int parse_flags_mode(char c, flags_mode_t *mode) 
{        
        switch (c) {

        case '+':
                *mode = mode_all;
                break;
                
        case '*': 
                *mode = mode_any;
                break;
                
        case '!':
                *mode = mode_not;
                break;
        }

        return 0;
}




static int parse_ip_flags(char *str, flags_mode_t *mode, int *out) 
{
        int ret;
        
        if ( strlen(str) == 0 ) 
		return -1;
        
	while ( toupper(*str) ) {

                switch (*str) {
                
		case 'R': 
			*out |= IP_RF;
			break;
                
		case 'D':
			*out |= IP_DF;
			break;
                
		case 'M':
			*out |= IP_MF;
			break;
                        
		case '0':
			*out = 0; /* snort compatibility ? */
			break;
                        
                case '!':
                case '+':
                case '*':
                        ret = parse_flags_mode(*str, mode);
                        if ( ret < 0 )
                                return -1;
                        break;
                        
		default: 
			signature_parser_set_error("Unknow fragbits flag %c", *str);
			return -1;
		}
		str++;
	}

        return  0;
}



/*
 * parse fragbits
 *
 * R - Reserved bit
 * D - DF bits
 * M - MF bits
 */
static int parse_fragbits(parameters_t *param, rules_t **rules) 
{
        int ret;
        int flags = 0, mask = 0;
        flags_mode_t mode = mode_exact;
        
        *rules = NULL;
        
	if ( strlen(param->str) == 0 ) 
		return -1;
        
        ret = parse_ip_flags(param->str, &mode, &flags);
        if ( ret < 0 )
                return -1;
        
        param = param->next;

        if ( param ) {
                
                ret = parse_ip_flags(param->str, &mode, &mask);
                if ( ret < 0 )
                        return -1;
        }

        mask = (mask) ? ~mask : ~0;
        
        switch (mode) {
                
        case mode_any:
                *rules = create_any_flags_rules(flags, mask, id_fragbits);
                break;

        case mode_all:
                *rules = create_all_flags_rules(flags, mask, id_fragbits);
                break;

        case mode_not:
                *rules = create_not_flags_rules(flags, mask, id_fragbits);
                break;
                
        case mode_exact:
                *rules = create_exact_flags_rules(flags, mask, id_fragbits);
                break;
        }
        
        return 0;
}



static int parse_fragoffset(char *str, rules_t **rules) 
{
        integer_t *integer;

        integer = malloc(sizeof(*integer));
        if ( ! integer ) {
                log(LOG_ERR, "memory exhausted.\n");
                return -1;
        }

        integer->num = atoi(str);
        *rules = make_new_rules(make_new_rule(id_fragoffset, integer), NULL);

        return 0;
}




/*
 * parse tcp flags:
 *
 * F - FIN (LSB in TCP Flags byte)
 * S - SYN
 * R - RST
 * P - PSH
 * A - ACK
 * U - URG
 * 2 - Reserved bit 2
 * 1 - Reserved bit 1 (MSB in TCP Flags byte)
 
 * There are also logical operators that can be used to specify matching criteria for the
 * indicated flags:
 * + - ALL flag, match on all specified flags plus any others
 * * - ANY flag, match on any of the specified flags
 *
 */

#ifndef TH_RES1
#define TH_RES2         0x40
#define TH_RES1         0x80                                                                            
#endif



static int do_parse_tcp_flags(char *str, flags_mode_t *mode, int *out) 
{
        int ret;
        
        if ( strlen(str) == 0 ) 
		return -1;
        
	while ( *str ) {
		switch (toupper(*str) ) {

                case 'F': 
			*out |= TH_FIN;
			break;
                        
		case 'S':
			*out |= TH_SYN;
			break;
                        
		case 'R':
			*out |= TH_RST;
			break;
                        
		case 'P':
			*out |= TH_PSH;
			break;
                        
		case 'A':
			*out |= TH_ACK;
			break;
                        
		case 'U':
			*out |= TH_URG;
			break;
                        
		case '1':
			*out |= TH_RES1;
			break;
                        
		case '2':
			*out |= TH_RES2;
			break;
                        
		case '0':
			*out = 0; /* snort compatibility ? */
			break;

                case '!':
                case '+':
                case '*':
                        ret = parse_flags_mode(*str, mode);
                        if ( ret < 0 )
                                return -1;
                        break;

                default:
			signature_parser_set_error("Unknown tcp flag %c", *str);
			return -1;
		}
		str++;
	}

        return 0;
}



static int parse_tcp_flags(parameters_t *param, rules_t **rules) 
{
        int ret;
	int flags = 0, mask = 0;
        flags_mode_t mode = mode_exact;

        ret = do_parse_tcp_flags(param->str, &mode, &flags);
        if ( ret < 0 )
                return -1;

        param = param->next;

        if ( param ) {
                ret = do_parse_tcp_flags(param->str, &mode, &mask);
                if ( ret < 0 )
                        return -1;
        }
        
        
        mask = (mask) ? ~mask : ~0;
        
        switch (mode) {
                         
        case mode_any:
                *rules = create_any_flags_rules(flags, mask, id_tcp_flags);
                break;

        case mode_all:
                *rules = create_all_flags_rules(flags, mask, id_tcp_flags);
                break;

        case mode_not:
                *rules = create_not_flags_rules(flags, mask, id_tcp_flags);
                break;
                
        case mode_exact:
                *rules = create_exact_flags_rules(flags, mask, id_tcp_flags);
                break;
        }
        
	return 0;
}




static int parse_content_list(char *filename, rules_t **rules) 
{
        FILE *fd;
        rule_t *rule;
        char buf[1024];

        fd = fopen(filename, "r");
        if ( ! fd ) {
                signature_parser_set_error("couldn't open file %s (%s).\n", filename, strerror(errno));
                return -1;
        }
        
        while ( fgets(buf, sizeof(buf), fd) ) {
                buf[strlen(buf) - 1] = '\0';

                rule = signature_get_content_rule(buf, NULL);
                if ( ! rule )
                        return -1;

                *rules = make_new_rules(rule, *rules);
        }

        fclose(fd);
        
        return 0;
}




static int match_tcp_flow(packet_container_t *pc, flags_t *flag) 
{
        int ret;

        if ( pc->network_layer_depth < 0 || pc->transport_layer_depth < 0 )
                return -1;
        
        if ( pc->packet[pc->network_layer_depth].proto != p_ip ||
             pc->packet[pc->transport_layer_depth].proto != p_tcp )
                return -1;
        
        ret = tcp_stream_get_state(pc);
        
        return ((ret & flag->mask) == flag->flags ) ? 0 : -1;
}




static int get_flow_num(const char *str, int *flow) 
{
        if ( strcasecmp(str, "to_client") == 0 )
                *flow |= STREAM_FROM_SERVER;

        else if ( strcasecmp(str, "to_server") == 0 )
                *flow |= STREAM_FROM_CLIENT;

        else if ( strcasecmp(str, "from_client") == 0 )
                *flow |= STREAM_FROM_CLIENT;

        else if ( strcasecmp(str, "from_server") == 0 )
                *flow |= STREAM_FROM_SERVER;

        else if ( strcasecmp(str, "established") == 0 )
                *flow |= STREAM_ESTABLISHED;

        else if ( strcasecmp(str, "stateless") == 0 )
                *flow |= STREAM_STATELESS;

        else if ( strcasecmp(str, "no_stream") == 0 )
                *flow |= STREAM_PACKET_NOT_REASSEMBLED;
                
        else if ( strcasecmp(str, "only_stream") == 0 ) 
                *flow |= STREAM_PACKET_REASSEMBLED;
        
        else {
                signature_parser_set_error("Invalid flow value (%s)", str);
                return -1;
        }
        
        return 0;
}




static int parse_tcp_flow(parameters_t *param, rules_t **rules) 
{
        flags_t *flag;
        int flow = 0, ret;

        if ( ! tcp_stream_is_enabled() ) {                
                *rules = NULL;
                return 0;
        }
        
        while ( param ) {
            
                ret = get_flow_num(param->str, &flow);
                if ( ret < 0 ) 
                        return -1;
                
                param = param->next;
        }
        
        flag = malloc(sizeof(*flag));
        if ( ! flag ) {
                log(LOG_ERR, "memory exhausted.\n");
                return -1;
        }

        flag->mask = flow;
        flag->flags = flow;
        
        *rules = make_new_rules(make_new_rule(id_tcp_flow, flag), NULL);

        return 0;
}




#define DO_PARSE_VALUE(Name, Type)                                                   \
                                                                                     \
int signature_parse_##Name(char *str, rules_t **rules) {                             \
       Type##_t *type_parsed = parse_##Type(str);                                    \
       if ( type_parsed ) {                                                          \
               *rules = make_new_rules(make_new_rule(id_##Name, type_parsed), NULL); \
               return 0;                                                             \
       } else {                                                                      \
               signature_parser_set_error("Invalid " #Name " value (%s)", str);      \
               *rules = NULL;                                                        \
               return -1;                                                            \
       }                                                                             \
}



DO_PARSE_VALUE(ttl,      integer);
DO_PARSE_VALUE(tos,      integer);
DO_PARSE_VALUE(id,       integer);
DO_PARSE_VALUE(dsize,    segment);
DO_PARSE_VALUE(win,      integer);
DO_PARSE_VALUE(seq,      integer);
DO_PARSE_VALUE(ack,      integer);
DO_PARSE_VALUE(itype,    integer);
DO_PARSE_VALUE(icode,    integer);
DO_PARSE_VALUE(icmp_id,  integer);
DO_PARSE_VALUE(icmp_seq, integer);
DO_PARSE_VALUE(ipproto,  integer);



void init_key_parser(void) 
{
        data_msg_id = signature_engine_get_new_data_id();
        data_classtype_id = signature_engine_get_new_data_id();
        data_reference_id = signature_engine_get_new_data_id();
        data_sid_id = signature_engine_get_new_data_id();
        data_revision_id = signature_engine_get_new_data_id();
        
        /*
         * register default type
         */
	id_ip_src = REGISTER_TYPE(PRIORITY_IP_SRC, ip, signature_match_ipsrc);
	id_ip_dst = REGISTER_TYPE(PRIORITY_IP_DST, ip, signature_match_ipdst);
	id_port_src = REGISTER_TYPE(PRIORITY_PORT_SRC, segment, signature_match_portsrc);
	id_port_dst = REGISTER_TYPE(PRIORITY_PORT_DST, segment, signature_match_portdst);
	id_tcp_flags = REGISTER_TYPE(PRIORITY_TCP_FLAGS, flags, signature_match_tcpflags);
	id_ttl = REGISTER_TYPE(PRIORITY_TTL, integer, signature_match_ttl);
	id_tos = REGISTER_TYPE(PRIORITY_TOS, integer, signature_match_tos);
	id_id = REGISTER_TYPE(PRIORITY_ID, integer, signature_match_id);
	id_dsize = REGISTER_TYPE(PRIORITY_DSIZE, segment, signature_match_datasize); 
	id_win = REGISTER_TYPE(PRIORITY_WIN, integer, signature_match_win);
	id_seq = REGISTER_TYPE(PRIORITY_SEQ, integer, signature_match_seq);
	id_ack = REGISTER_TYPE(PRIORITY_ACK, integer, signature_match_ack);
	id_itype = REGISTER_TYPE(PRIORITY_ITYPE, integer, signature_match_icmptype);
	id_icode = REGISTER_TYPE(PRIORITY_ICODE, integer, signature_match_icmpcode);
	id_icmp_id = REGISTER_TYPE(PRIORITY_ICMP_ID, integer, signature_match_icmpid);
	id_icmp_seq = REGISTER_TYPE(PRIORITY_ICMP_SEQ, integer, signature_match_icmpseq);
	id_fragbits = REGISTER_TYPE(PRIORITY_FRAGBITS, flags, signature_match_fragbits);
        id_fragoffset = REGISTER_TYPE(PRIORITY_FRAGOFFSET, integer, signature_match_fragoffset);
        id_ipproto = REGISTER_TYPE(PRIORITY_IPPROTO, integer, signature_match_ipproto);
        id_tcp_flow = REGISTER_TYPE(PRIORITY_TCP_FLOW, flags, match_tcp_flow);
        
        id_content_list = signature_engine_register_type(PRIORITY_CONTENT_LIST, NULL, NULL, NULL,
                                                         NULL, NULL, NULL, NULL,
                                                         signature_match_packet_content);

        
        /*
         * Theses generic parsing function are provided by Prelude.
         */
        signature_parser_add_one_arg_key("ip_proto", &signature_parse_ipproto);
        signature_parser_add_one_arg_key("ttl", &signature_parse_ttl);
        signature_parser_add_one_arg_key("tos", &signature_parse_tos);
        signature_parser_add_one_arg_key("id", &signature_parse_id);
        signature_parser_add_one_arg_key("dsize", &signature_parse_dsize);
	signature_parser_add_one_arg_key("window", &signature_parse_win);
        signature_parser_add_one_arg_key("seq", &signature_parse_seq);
	signature_parser_add_one_arg_key("ack", &signature_parse_ack);
        signature_parser_add_one_arg_key("itype", &signature_parse_itype);
        signature_parser_add_one_arg_key("icode", &signature_parse_icode);
        signature_parser_add_one_arg_key("icmp_id", &signature_parse_icmp_id);
        signature_parser_add_one_arg_key("icmp_seq", &signature_parse_icmp_seq);
        signature_parser_add_multiple_args_key("sameip", &signature_parse_sameip);
        signature_parser_add_one_arg_key("content", &signature_parse_content);
        signature_parser_add_one_arg_key("offset", &signature_parse_offset);
        signature_parser_add_one_arg_key("depth", &signature_parse_depth);
        signature_parser_add_one_arg_key("within", &signature_parse_within);
        signature_parser_add_one_arg_key("distance", &signature_parse_distance);
        signature_parser_add_multiple_args_key("flow", &parse_tcp_flow);
        
        signature_parser_add_multiple_args_key("nocase", &signature_parse_nocase);
        
        /*
         * Theses are our.
         */
        signature_parser_add_one_arg_key("ipopts", &parse_ipopts);
        signature_parser_add_one_arg_key("fragoffset", &parse_fragoffset);        
        signature_parser_add_one_arg_key("msg", &parse_msg);
        signature_parser_add_one_arg_key("classtype", &parse_classtype);
        signature_parser_add_one_arg_key("content-list", &parse_content_list);
        signature_parser_add_one_arg_key("sid", &parse_sid);
        signature_parser_add_one_arg_key("rev", &parse_revision);
        signature_parser_add_one_arg_key("portsrc", &parse_portsrc);
        signature_parser_add_one_arg_key("portdst", &parse_portdst);
        signature_parser_add_one_arg_key("ipsrc", &parse_ip_src);
	signature_parser_add_one_arg_key("ipdst", &parse_ip_dst);
        
        /*
         * Register key that don't take any argument.
         */
        signature_parser_add_multiple_args_key("reference", &parse_reference);
        
        /*
         * Key to be ignored.
         */
        signature_parser_add_multiple_args_key("react", &signature_parse_nothing);
        signature_parser_add_multiple_args_key("resp", &signature_parse_nothing);
        signature_parser_add_multiple_args_key("logto", &signature_parse_nothing);
        signature_parser_add_multiple_args_key("flags", &parse_tcp_flags);
        signature_parser_add_multiple_args_key("fragbits", &parse_fragbits);

        signature_parser_add_multiple_args_key("byte_test", &parse_byte_test);
        signature_parser_add_multiple_args_key("byte_jump", &parse_byte_jump);
}



struct tbl {
        const char *name;
        int num;
};


int add_classtype(const char *shortname, const char *desc,
                  const char *priority, const char *type, const char *completion)
{
        int i;
        classtype_t *class;
        struct tbl severity_tbl[] = {
                { "high", high     },
                { "medium", medium },
                { "low", low       },
                { NULL,            },
        };

        struct tbl completion_tbl[] = {
                { "failed", failed       },
                { "succeeded", succeeded },
                { NULL,                  },
        };

        struct tbl type_tbl[] = { 
                { "other", other },
                { "admin", admin },
                { "dos", dos     },
                { "file", file   },
                { "recon", recon },
                { "user", user   },
                { NULL,          },
        };
        
        class = calloc(1, sizeof(*class));
        if ( ! class ) {
                log(LOG_ERR, "memory exhausted.\n");
                return -1;
        }

        for ( i = 0; severity_tbl[i].name; i++ ) {
                if ( strcmp(severity_tbl[i].name, priority) == 0 ) {
                        class->severity = severity_tbl[i].num;
                        break;
                }
        }
        
        for ( i = 0; type_tbl[i].name; i++ ) {
                if ( strcmp(type_tbl[i].name, type) == 0 ) {
                        class->type = type_tbl[i].num;
                        break;
                }
        }

        for ( i = 0; completion_tbl[i].name; i++ ) {
                if ( strcmp(completion_tbl[i].name, completion) == 0 ) {
                        class->completion = completion_tbl[i].num;
                        break;
                }
        }
        
        class->shortname = strdup(shortname);
        class->desc = strdup(desc);
        class->desclen = strlen(desc) + 1;

        list_add_tail(&class->list, &classtype_list);
        
        return 0;
}




int add_reference(const char *name, const char *url) 
{
        int i;
        reference_t *ref;
        struct tbl ref_tbl[] = {
                { "bugtraq", bugtraqid               },
                { "cve",     cve                     },
                { "vendor-specific", vendor_specific },
                { NULL, 0 },
        };
        
        ref = calloc(1, sizeof(*ref));
        if ( ! ref ) {
                log(LOG_ERR, "memory exhausted.\n");
                return -1;
        }
        
        if ( url )
                ref->url = strdup(url);
        
        if ( name )
                ref->name = strdup(name);

        ref->origin = origin_unknown;
        
        for ( i = 0; ref_tbl[i].name; i++ ) {

                if ( strcmp(name, ref_tbl[i].name) == 0 ) {
                        ref->origin = ref_tbl[i].num;
                        break;
                }
        }
        
        list_add_tail(&ref->list, &reference_list);

        return 0;
}
