// See www.openfst.org for extensive documentation on this weighted
// finite-state transducer library.
//
// Determinizes an FST.

#include <memory>

#include <fst/script/determinize.h>

DEFINE_double(delta, fst::kDelta, "Comparison/quantization delta");
DEFINE_string(weight, "", "Weight threshold");
DEFINE_int64(nstate, fst::kNoStateId, "State number threshold");
DEFINE_int64(subsequential_label, 0,
             "Input label of arc corresponding to residual final output when"
             " producing a subsequential transducer");
DEFINE_string(det_type, "functional",
              "Type of determinization: \"functional\", "
              "\"nonfunctional\", \"disambiguate\"");
DEFINE_bool(increment_subsequential_label, false,
            "Increment subsequential_label to obtain distinct labels for "
            " subsequential arcs at a given state");

int main(int argc, char **argv) {
  namespace s = fst::script;
  using fst::script::FstClass;
  using fst::script::MutableFstClass;
  using fst::script::VectorFstClass;
  using fst::script::WeightClass;

  string usage = "Determinizes an FST.\n\n  Usage: ";
  usage += argv[0];
  usage += " [in.fst [out.fst]]\n";

  std::set_new_handler(FailedNewHandler);
  SET_FLAGS(usage.c_str(), &argc, &argv, true);
  if (argc > 3) {
    ShowUsage();
    return 1;
  }

  fst::DeterminizeType det_type;
  if (FLAGS_det_type.empty() || FLAGS_det_type[0] == 'f') {
    det_type = fst::DETERMINIZE_FUNCTIONAL;
  } else if (FLAGS_det_type[0] == 'n') {
    det_type = fst::DETERMINIZE_NONFUNCTIONAL;
  } else if (FLAGS_det_type[0] == 'd') {
    det_type = fst::DETERMINIZE_DISAMBIGUATE;
  } else {
    LOG(ERROR) << argv[0] << ": Unknown determinize type: " << FLAGS_det_type;
    return 1;
  }

  string in_name = (argc > 1 && strcmp(argv[1], "-") != 0) ? argv[1] : "";
  string out_name = argc > 2 ? argv[2] : "";

  std::unique_ptr<FstClass> ifst(FstClass::Read(in_name));
  if (!ifst) return 1;

  VectorFstClass ofst(ifst->ArcType());

  const WeightClass weight_threshold =
      FLAGS_weight.empty() ? WeightClass::Zero(ifst->WeightType())
                           : WeightClass(ifst->WeightType(), FLAGS_weight);

  s::DeterminizeOptions opts(FLAGS_delta, weight_threshold, FLAGS_nstate,
                             FLAGS_subsequential_label, det_type,
                             FLAGS_increment_subsequential_label);

  s::Determinize(*ifst, &ofst, opts);

  ofst.Write(out_name);

  return 0;
}
