/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */

#include <stdio.h>
#include <string.h>
#include "RCX_Cmd.h"


// handy macros
#define LO_BYTE(s)	((UByte)((s) & 0xff))
#define HI_BYTE(s)	((UByte)(((s) >> 8) & 0xff))


RCX_Cmd::~RCX_Cmd()
{
	if (fLength > kRCX_Cmd_MaxShortLength)
		delete [] fBody.fPtr;
}


void RCX_Cmd::SetLength(int length)
{
	if (fLength == length) return;

	if (fLength > kRCX_Cmd_MaxShortLength)
		delete [] fBody.fPtr;

	fLength = length;

	if (fLength > kRCX_Cmd_MaxShortLength)
		fBody.fPtr = new UByte[fLength];	
}


int RCX_Cmd::CopyOut(UByte *dst)
{
	memcpy(dst, GetBody(), (ULong)fLength);
	return fLength;
}


RCX_Cmd* RCX_Cmd::Set(const UByte *data, int length)
{
	SetLength(length);
	memcpy(GetBody(), data, (size_t)length);
	return this;
}


RCX_Cmd* RCX_Cmd::Set(UByte d0)
{
	SetLength(1);
	fBody.fData[0] = d0;
	return this;
}


RCX_Cmd* RCX_Cmd::Set(UByte d0, UByte d1)
{
	SetLength(2);
	fBody.fData[0] = d0;
	fBody.fData[1] = d1;
	return this;
}


RCX_Cmd* RCX_Cmd::Set(UByte d0, UByte d1, UByte d2)
{
	SetLength(3);
	fBody.fData[0] = d0;
	fBody.fData[1] = d1;
	fBody.fData[2] = d2;
	return this;
}



RCX_Cmd* RCX_Cmd::Set(UByte d0, UByte d1, UByte d2, UByte d3)
{
	SetLength(4);
	fBody.fData[0] = d0;
	fBody.fData[1] = d1;
	fBody.fData[2] = d2;
	fBody.fData[3] = d3;
	return this;
}


RCX_Cmd* RCX_Cmd::Set(UByte d0, UByte d1, UByte d2, UByte d3, UByte d4)
{
	SetLength(5);
	fBody.fData[0] = d0;
	fBody.fData[1] = d1;
	fBody.fData[2] = d2;
	fBody.fData[3] = d3;
	fBody.fData[4] = d4;
	return this;
}


RCX_Cmd* RCX_Cmd::Set(UByte d0, UByte d1, UByte d2, UByte d3, UByte d4, UByte d5)
{
	SetLength(6);
	fBody.fData[0] = d0;
	fBody.fData[1] = d1;
	fBody.fData[2] = d2;
	fBody.fData[3] = d3;
	fBody.fData[4] = d4;
	fBody.fData[5] = d5;
	return this;
}


RCX_Cmd* RCX_Cmd::MakeUnlock()
{
	return Set(kRCX_UnlockOp, 1, 3, 5, 7, 0xb);
}

RCX_Cmd* RCX_Cmd::MakePlayTone(UShort freq, UByte duration)
{
	return Set(kRCX_PlayToneOp, LO_BYTE(freq), HI_BYTE(freq), duration);
}


RCX_Cmd* RCX_Cmd::MakeBegin(RCX_FragmentType type, UByte taskNumber, UShort length)
{
	UByte op = (UByte)((type == kRCX_TaskFragment) ? kRCX_BeginTaskOp : kRCX_BeginSubOp);
	return Set(op, 0, taskNumber, 0, LO_BYTE(length), HI_BYTE(length));
}


RCX_Cmd* RCX_Cmd::MakeOutputPower(UByte outputs, RCX_Value value)
{
	return Set(kRCX_OutputPowerOp, outputs, RCX_VALUE_TYPE(value), (UByte)RCX_VALUE_DATA(value));
}


RCX_Cmd* RCX_Cmd::MakeValue16(UByte op, RCX_Value value)
{
	int data = RCX_VALUE_DATA(value);
	
	return Set(op, RCX_VALUE_TYPE(value), LO_BYTE(data), HI_BYTE(data));
}

RCX_Cmd* RCX_Cmd::MakeValue8(UByte op, RCX_Value value)
{	
	return Set(op, RCX_VALUE_TYPE(value), (UByte)RCX_VALUE_DATA(value));
}

RCX_Cmd* RCX_Cmd::MakeSet(RCX_Value dst, RCX_Value src)
{
	SetLength(6);
	UByte *ptr = GetBody();
	
	*ptr++ = 0x05;
	*ptr++ = RCX_VALUE_TYPE(dst);
	*ptr++ = LO_BYTE(RCX_VALUE_DATA(dst));
	*ptr++ = RCX_VALUE_TYPE(src);
	*ptr++ = LO_BYTE(RCX_VALUE_DATA(src));
	*ptr++ = HI_BYTE(RCX_VALUE_DATA(src));
	
	return this;
}


RCX_Cmd* RCX_Cmd::MakeTest(RCX_Value v1, RCX_Relation rel, RCX_Value v2, short offset)
{
	int d1 = RCX_VALUE_DATA(v1);
	UByte *ptr;
	SetLength(8);
	
	ptr = GetBody();
	
	*ptr++ = kRCX_TestOp;
	*ptr++ = (UByte)((rel << 6) | RCX_VALUE_TYPE(v1));
	*ptr++ = RCX_VALUE_TYPE(v2);
	*ptr++ = LO_BYTE(d1);
	*ptr++ = HI_BYTE(d1);
	*ptr++ = (UByte)RCX_VALUE_DATA(v2);
	SetOffset(offset);
	return this;
}


RCX_Cmd* RCX_Cmd::MakeCheckLoop(short offset)
{
	Set(kRCX_CheckLoopOp, 0, 0);
	SetOffset(offset);
	return this;
}


RCX_Cmd* RCX_Cmd::MakeJump(short offset)
{
	SetLength(3);
	
	GetBody()[0] = kRCX_JumpOp;
	SetOffset(offset);
	return this;
}


void RCX_Cmd::SetOffset(int offset)
{
	UByte *ptr;
	UByte neg = 0;
		
	offset -= fLength - 2;
	ptr = GetBody() + fLength - 2;
	
	if (GetBody()[0] != kRCX_JumpOp)
	{
		*ptr++ = (UByte)(offset & 0xff);
		*ptr++ = (UByte)((offset >> 8) & 0xff);
	}
	else
	{
		if (offset < 0)
		{
			neg = 0x80;
			offset = -offset;
		}

		*ptr++ = (UByte)(neg | (offset & 0x7f));
		*ptr++ = (UByte)((offset >> 7) & 0xff);
	}
}


RCX_Cmd* RCX_Cmd::MakeDownload(UShort seq, const UByte *data, UShort length)
{
	UByte *ptr;
	UByte checksum;
	
	SetLength(6 + length);
	
	ptr = GetBody();
	*ptr++ = kRCX_DownloadOp;
	*ptr++ = LO_BYTE(seq);
	*ptr++ = HI_BYTE(seq);
	*ptr++ = LO_BYTE(length);
	*ptr++ = HI_BYTE(length);
	
	checksum = 0;
	while(length--)
	{
		UByte b = *data++;
		checksum += b;
		*ptr++ = b;
	}
	
	*ptr++ = checksum;
	
	return this;
}


RCX_Cmd* RCX_Cmd::MakeUploadDatalog(UShort start, UShort count)
{
	return Set(kRCX_UploadDatalogOp,
		LO_BYTE(start), HI_BYTE(start),
		LO_BYTE(count), HI_BYTE(count));
}


RCX_Cmd* RCX_Cmd::MakeVar(RCX_VarCode code, UByte var, RCX_Value value)
{
	int data = RCX_VALUE_DATA(value);
	return Set((UByte)kRCX_VarOp(code), var, RCX_VALUE_TYPE(value), LO_BYTE(data), HI_BYTE(data));
}


RCX_Cmd* RCX_Cmd::MakeBoot()
{

	Set(kRCX_BootOp, 0x4c, 0x45, 0x47, 0x4f, 0xae);
	return this;
};


RCX_Cmd* RCX_Cmd::MakeUnlockCM()
{
	static const UByte unlockMsg[] = {kRCX_BootOp,
	  'D','o',' ','y','o','u',' ','b','y','t','e',',',' ',
	  'w','h','e','n',' ','I',' ','k','n','o','c','k','?' };
	 Set(unlockMsg, sizeof(unlockMsg));
	 
	 return this;
}


void RCX_Cmd::Print()
{
	UByte *ptr = GetBody();

	for(int i=0; i<fLength; i++)
		printf("%02x ", ptr[i]);
	printf("\n");
}
