/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1999 David Baum.
 * All Rights Reserved.
 */


#include "CallStmt.h"
#include "Bytecode.h"
#include "Program.h"
#include "Symbol.h"
#include "Fragment.h"
#include "GosubStmt.h"
#include "InlineStmt.h"
#include "FunctionDef.h"
#include "Mapping.h"
#include "AtomExpr.h"
#include "DeclareStmt.h"
#include "Error.h"
#include "BlockStmt.h"
#include "ScopeStmt.h"


CallStmt::CallStmt()
{
	fName = 0;
}


CallStmt::~CallStmt()
{
	for(size_t i=0; i<fParams.size(); i++)
		delete fParams[i];

}


void CallStmt::Expand(Fragment *fragment)
{
	if (Fragment *sub = gProgram->GetSub(fName))
	{
		// sub call
		if (fragment->GetType() == kRCX_SubFragment)
		{
			Error(kErr_NestedSubs).Raise(&fLocation);
		}
		else if (fParams.size() != 0)
		{
			Error(kErr_ParamCount).Raise(&fLocation);
		}
		else
		{
			fBody = new GosubStmt(sub, fLocation);
			sub->AssignTaskID(fragment->GetTaskID());
		}
	}
	else if (FunctionDef *func = gProgram->GetFunction(fName))
	{
		if (func->IsExpanded())
		{
			Error(kErr_RecursiveCall, fName->GetKey()).Raise(&fLocation);
		}
		else
		{
			func->SetExpanded(true);
			ExpandFunction(func, fragment);
			func->SetExpanded(false);
		}
	}
	else
	{
		Error(kErr_UndefinedFunction, fName->GetKey()).Raise(&fLocation);
	}
}


void CallStmt::EmitActual(Bytecode &b)
{
	if (fBody)
		fBody->Emit(b);
}

Stmt *CallStmt::Clone(Mapping *b) const
{
	CallStmt *c = new CallStmt;
	
	c->fName = fName;
	c->fLocation = fLocation;
	c->fBody = fBody ? fBody->Clone(b) : 0;
	
	for(size_t i=0; i<fParams.size(); i++)
		c->fParams.push_back(fParams[i]->Clone(b));

	return c;
}


void CallStmt::ExpandFunction(FunctionDef *func, Fragment *fragment)
{
	size_t argCount = func->GetArgCount();
	
	// check number of parameters
	if (argCount != fParams.size())
	{
		Error(kErr_ParamCount).Raise(&fLocation);
		return;
	}

	/* statment should look like this:
	 *
	 * CallStmt
	 *   |
	 * InlineStmt
	 *   |
	 * ScopeStmt
	 *   |
	 * BlockStmt
	 *   /        |       \
	 * DeclareStmt...   body of function
	 */
	BlockStmt *block = new BlockStmt();
	fBody = new InlineStmt(new ScopeStmt(block));

	Mapping mapping;
		
	for(size_t i=0; i<argCount; i++)
	{
		const Expr* arg = fParams[i];
		int var = func->GetArgVar(i);
		int val;
		
		switch(func->GetArgType(i))
		{
			case FunctionDef::kConstantArg:
				if (!arg->Evaluate(val))
				{
					Error(kErr_ParamType, "constant").Raise(&arg->GetLoc());
					return;
				}
				mapping.Add(var, new AtomExpr(kRCX_ConstantType, val, fLocation));
				break;
			case FunctionDef::kIntegerArg:
				val = gProgram->NextVirtualVar();
				mapping.Add(var, new AtomExpr(kRCX_VariableType, val, fLocation));
				{
					DeclareStmt *ds = new DeclareStmt(val, fLocation, 1);
					ds->SetInitialValue(arg->Clone(0));
					block->Add(ds);
				}
				break;
			case FunctionDef::kReferenceArg:
				val = arg->GetLValue();
				if (val == kIllegalVar)
				{
					Error(kErr_ParamType, "variable").Raise(&arg->GetLoc());
					return;
				}
				mapping.Add(var, new AtomExpr(kRCX_VariableType, val, fLocation));
				break;
			case FunctionDef::kConstRefArg:
				mapping.Add(var, arg->Clone(0));
				break;
			case FunctionDef::kSensorArg:
				if (RCX_VALUE_TYPE(arg->GetStaticEA()) != kRCX_InputValueType)
				{
					Error(kErr_ParamType, "sensor").Raise(&arg->GetLoc());
					return;
				}
				mapping.Add(var, arg->Clone(0));
				break;
			default:
				Error(kErr_ParamType, "???").Raise(&fParams[i]->GetLoc());
				return;
		}
	}

	
	// add body of inline and then expand
	block->Add(func->GetBody()->Clone(&mapping));

	Expander e(fragment);
	Apply(fBody, e);
}


void CallStmt::GetExprs(vector<Expr *> &v) const
{
	int n = fParams.size();
	
	for(int i=0; i<n; ++i)
		v.push_back(fParams[i]);
}



bool CallStmt::Expander::operator()(Stmt *s)
{
	CallStmt *cs;
	
	if ((cs=dynamic_cast<CallStmt*>(s)) != 0)
	{
		cs->Expand(fFragment);
		return false;
	}

	return true;
}

