/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */

#ifndef __Bytecode_h
#define __Bytecode_h

#ifndef __PListS_h
#include "PListS.h"
#endif

#ifndef __PTypes_h
#include "PTypes.h"
#endif

#ifndef __JumpStmt_h
#include "JumpStmt.h"
#endif

#ifndef __RCX_Constants_h
#include "RCX_Constants.h"
#endif

#ifndef __VarAllocator_h
#include "VarAllocator.h"
#endif

#include <vector.h>

class RCX_Cmd;

// Fixup types
#define kPatch_Normal	0
#define kPatch_SignBit	1

#define kCurrentPosition	-1
#define kIllegalLabel		-1

class VarPool;
class RCX_Target;

class Bytecode
{
public:	
	enum HandlerCode
	{
		// codes for Handler contexts
		kEventHandler = 0,
		kAcquireHandler,
		kHandlerCount	// not an actual handler - must be the last code
	};
	
	enum FlowCode
	{
		// codes for Flow contexts
		kContinueFlow = 0,
		kBreakFlow,
		kReturnFlow,
		kFlowCount	// not an actual flow - must be the last code
	};
	
				Bytecode(VarAllocator &varAllocator, const RCX_Target *target);
				~Bytecode();

	int				GetLength()	const	{ return fData.size(); }
	const UByte*	GetData() const		{ return &fData[0]; }
	
	void		Add(const RCX_Cmd &cmd);
	void		AddJump(int label);
	void		AddTest(RCX_Value v1, RCX_Relation rel, RCX_Value v2, int label);
	void		AddMove(int dst, RCX_Value ea);

	bool		AddFlowJump(FlowCode code);
	
	int			NewLabel();
	void		SetLabel(int label, int target = kCurrentPosition);
	void		AddFixup(int type, UShort location, int label);
	void		ApplyFixups();

	int			PushFlow(FlowCode code, bool legal=true);
	void		PopFlow(FlowCode code)		{ fFlowContexts[code].pop_back(); }

	bool		BeginHandler(int type);
	void		EndHandler(int type);

	int			GetTempVar(bool canUseLocal);
	void		ReleaseTempEA(RCX_Value ea);
	bool		IsTempEA(RCX_Value ea) const;
	bool		IsLocalEA(RCX_Value ea) const;
	
	VarAllocator&	GetVarAllocator()	{ return fVarAllocator; }
		
	const RCX_Target*	GetTarget() const	{ return fTarget; }
	
private:
	int			GetFlowLevel(FlowCode code)	{ return fFlowContexts[code].size(); }
	int			GetFlowLabel(FlowCode code);
	void		Add(const UByte *data, int count);
	void		AddHandlerExit(int i);
	
	class Fixup : public PLinkS<Fixup>
	{
	public:
		int 	fType;
		UShort	fLocation;
		int		fLabel;
	};
		
	struct HandlerContext
	{
				HandlerContext() : fInUse(false)	{ }

		int		fFlowLevels[kFlowCount];
		bool	fInUse;
	};


	vector<UByte>		fData;

	PListSS<Fixup>		fFixups;
	vector<UShort>		fLabels;

	vector<int>			fFlowContexts[kFlowCount];	
	HandlerContext		fHandlerContexts[kHandlerCount];

	VarAllocator&		fVarAllocator;
	const RCX_Target*	fTarget;
};

#endif
