/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */

#include <string.h>
#include "Bytecode.h"
#include "RCX_Cmd.h"
#include "RCX_Target.h"

Bytecode::Bytecode(VarAllocator &varAllocator, const RCX_Target *target) :
	fVarAllocator(varAllocator),
	fTarget(target)
{
	fData.reserve(256);
	fLabels.reserve(50);
	
	for(int i=0; i<kFlowCount; ++i)
		fFlowContexts[i].reserve(10);

	PushFlow(kContinueFlow, false);
	PushFlow(kBreakFlow, false);
	PushFlow(kReturnFlow, false);
}


Bytecode::~Bytecode()
{
	Fixup *f;

	while((f=fFixups.RemoveHead()) != nil)
		delete f;
}


void Bytecode::Add(const UByte *data, int count)
{
	int n = fData.size();
	
	fData.resize(n + count);
	memcpy(&fData[n], data, (size_t) count);
}


void Bytecode::Add(const RCX_Cmd &cmd)
{
	Add(cmd.GetBody(), cmd.GetLength());
}


void Bytecode::AddJump(int label)
{
	RCX_Cmd cmd;
	
	cmd.MakeJump(0);
	Add(cmd);
	AddFixup(kPatch_SignBit, fData.size() - 2, label);
}


bool Bytecode::AddFlowJump(FlowCode code)
{
	int label = GetFlowLabel(code);
	if (label == kIllegalLabel) return false;
	
	for(int i=0; i<kHandlerCount; ++i)
	{
		HandlerContext &h = fHandlerContexts[i];
		if (h.fInUse &&
			GetFlowLevel(code) == h.fFlowLevels[code])
		{
			AddHandlerExit(i);
		}
	}
	
	AddJump(label);
	
	return true;
}


void Bytecode::AddHandlerExit(int i)
{
	RCX_Cmd cmd;

	if (i==kEventHandler)
		cmd.Set(0xb0);
	else
		cmd.Set(0xa0);
	Add(cmd);
}


void Bytecode::AddTest(RCX_Value v1, RCX_Relation rel, RCX_Value v2, int label)
{
	RCX_Cmd cmd;
	
	cmd.MakeTest(v1, rel, v2, 0);
	Add(cmd);
	AddFixup(kPatch_Normal, GetLength() - 2, label); 
}


void Bytecode::AddMove(int dst, RCX_Value ea)
{
	if (ea == RCX_VALUE(kRCX_VariableType, dst)) return;
	
	RCX_Cmd cmd;
	cmd.MakeVar(kRCX_SetVar, (UByte)dst, ea);
	Add(cmd);
}


int Bytecode::NewLabel()
{
	int label = fLabels.size();
	
	fLabels.push_back(GetLength());
	
	return label;
}


void Bytecode::SetLabel(int label, int target)
{
	// don't do anything if the label is illegal
	if (label == kIllegalLabel) return;
	
	if (target == kCurrentPosition)
		fLabels[label] = GetLength();
	else
		fLabels[label] = (UShort)target;
}


void Bytecode::AddFixup(int type, UShort location, int label)
{
	Fixup *f;
	
	f = new Fixup();
	
	f->fType = type;
	f->fLocation = location;
	f->fLabel = label;
	
	fFixups.InsertHead(f);
}


void Bytecode::ApplyFixups()
{
	Fixup *f;
	
	while((f = fFixups.RemoveHead()) != nil)
	{
		int offset = fLabels[f->fLabel] - f->fLocation;

		if (f->fType == kPatch_SignBit)
		{
			UByte neg = 0;

			if (offset < 0)
			{
				neg = 0x80;
				offset = -offset;
			}

			fData[f->fLocation] = (UByte)(neg | (offset & 0x7f));
			fData[f->fLocation+1] = (UByte)((offset >> 7) & 0xff);
		}
		else
		{
			fData[f->fLocation] = (UByte)(offset & 0xff);
			fData[f->fLocation+1] = (UByte)((offset >> 8) & 0xff);
		}
		
		delete f;
	}
}


int Bytecode::PushFlow(FlowCode code, bool legal)
{
	int label = legal ? NewLabel() : kIllegalLabel;
	fFlowContexts[code].push_back(label);
	return label;
}


int Bytecode::GetFlowLabel(FlowCode code)
{
	vector<int> &v = fFlowContexts[code];
	return v[v.size() - 1];
}



/*
void Bytecode::PushLoopContext(bool continueAllowed)
{
	fLoopContexts.push_back(LoopContext(
		continueAllowed ? NewLabel() : kIllegalLabel,
		NewLabel()));
}


void Bytecode::PopLoopContext(int continueTarget, int breakTarget)
{
	SetLabel(GetLoopLabel(kContinueLabel), continueTarget);
	SetLabel(GetLoopLabel(kBreakLabel), breakTarget);

	fLoopContexts.pop_back();
}


int Bytecode::GetLoopLabel(int type) const
{
	return fLoopContexts[GetLoopLevel() - 1].fLabels[type];
}


void Bytecode::PushReturnLabel()
{
	fReturnLabels.push_back(NewLabel());
	fLoopContexts.push_back(LoopContext(kIllegalLabel, kIllegalLabel));
}


void Bytecode::PopReturnLabel()
{
	fReturnLabels.pop_back();
	fLoopContexts.pop_back();
}
*/


bool Bytecode::BeginHandler(int type)
{
	HandlerContext &h = fHandlerContexts[type];
	
	if (h.fInUse) return false;

	for(int i=0; i<kFlowCount; ++i)
	{
		h.fFlowLevels[i] = GetFlowLevel(static_cast<FlowCode>(i));
	}
	h.fInUse = true;
	
	return true;
}

void Bytecode::EndHandler(int type)
{
	fHandlerContexts[type].fInUse = false;
}


int Bytecode::GetTempVar(bool canUseLocals)
{
	return fVarAllocator.Allocate(true, canUseLocals, 1);
}


void Bytecode::ReleaseTempEA(RCX_Value ea)
{
	if (RCX_VALUE_TYPE(ea) == kRCX_VariableType ||
		ea & kRCX_ValueUsesTemp)
	{
		fVarAllocator.ReleaseTemp(RCX_VALUE_DATA(ea));
	}
}


bool Bytecode::IsTempEA(RCX_Value ea) const
{
	if (RCX_VALUE_TYPE(ea) != kRCX_VariableType) return false;
	int v = RCX_VALUE_DATA(ea);
	return fVarAllocator.IsTemp(v);
}


bool Bytecode::IsLocalEA(RCX_Value ea) const
{
	if (RCX_VALUE_TYPE(ea) != kRCX_VariableType) return false;
	return RCX_VALUE_DATA(ea) >= fTarget->fMaxGlobalVars;
}
