/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */

#include <stdio.h>
#include "RCX_Link.h"
#include "RCX_Disasm.h"

RCX_Link gLink;	// must be global so that destructor is called at exit


class Reader
{
public:
			Reader(RCX_Link &link);
			~Reader();
			
	void	poll();
	void	beginDownload();
	void	continueDownload();
	
private:
	RCX_Link&	link;
	UByte*	code;
	ULong	codeLength;
	ULong	codePos;
	ULong	lastBlock;
};


int main(int, char **)
{
	if (RCX_ERROR(gLink.Open("B")))
	{
		printf("Error opening RCX link\n");
		return -1;
	}
	
	printf("RCX Link open\n");

	Reader reader(gLink);
	
	while(true) {
		printf(" \b");
		reader.poll();
	}
}


Reader::Reader(RCX_Link &l)
	: link(l), code(0)
{
}	

Reader::~Reader()
{
	delete [] code;
}


void Reader::poll()
{
	RCX_Result r = link.Receive(true);
	if (r <= 0) return;
	
	UByte cmd = link.GetReplyByte(0);
	UByte data[9];
	UByte *ptr = data;
	
	*ptr++ = ~cmd;

	cmd &= 0xf7;
	
	switch(cmd)
	{
		case 0x15:	// firmware version
			*ptr++ = 0;
			*ptr++ = 3;
			*ptr++ = 0;
			*ptr++ = 1;
			*ptr++ = 0;
			*ptr++ = 3;
			*ptr++ = 3;
			*ptr++ = 0;
			break;
		case 0x25:	// begin download
			beginDownload();
			*ptr++ = 0;
			break;
		case 0x45:	// continue download
			continueDownload();
			*ptr++ = 0; 
			break;
		case 0x30:	// battery level
			*ptr++ = 35;
			*ptr++ = 40;
			break;
		default:
			break;
	}

	link.Send1(data, ptr-data);
	printf("%02x\n", cmd);
}


void Reader::beginDownload()
{
	UByte number = link.GetReplyByte(2);	
	codeLength = link.GetReplyByte(4) + ((ULong)link.GetReplyByte(5) << 8);
	
	delete [] code;
	
	code = new UByte[codeLength];
	codePos = 0;
	lastBlock = -1;
}


void Reader::continueDownload()
{
	ULong block = link.GetReplyByte(1) + ((ULong)link.GetReplyByte(2) << 8);
	ULong length = link.GetReplyByte(3) + ((ULong)link.GetReplyByte(4) << 8);
	
	if (block == lastBlock) return;
	
	for(int i=0; i<length && codePos < codeLength; ++i)
		code[codePos++] = link.GetReplyByte(5 + i);

	if (block == 0)
	{
		RCX_StdioPrinter dst(stdout);
		gRCX_Disasm.Print(&dst, code, codeLength);
	}
}
