/* Copyright (C) 2006 Andi Kleen, SuSE Labs.
   Use SMBIOS/DMI to map address to DIMM description.
   For reference see the SMBIOS specification 2.4

   dmi is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; version
   2.

   dmi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should find a copy of v2 of the GNU General Public License somewhere
   on your Linux system; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/* Notebook
   could check first if all dimms cover all memory and not dump then
   (aka the Sun/HP disease)
   add an option to dump existing errors in SMBIOS?
   implement code to look up PCI resources too.
 */
#define _GNU_SOURCE 1
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <stddef.h>
#include <sys/mman.h>

#ifdef STANDALONE
#define NELE(x) (sizeof(x)/sizeof(*(x)))
#define Wprintf printf
#else
#include "mcelog.h"
#include "dmi.h"
#endif

int verbose = 0;

struct anchor { 
	char str[4];	/* _SM_ */
	char csum;
	char entry_length;
	char major;
	char minor;
	short maxlength;
	char rev;
	char fmt[5];
	char str2[5]; /* _DMI_ */
	char csum2;
	short length;
	unsigned table;
	unsigned short numentries;
	char bcdrev;
} __attribute__((packed));

struct dmi_entry { 
	unsigned char type;
	unsigned char length;
	unsigned short handle;
};

enum { 
	DMI_MEMORY_MAPPED_ADDR = 20
};

struct dmi_memdev_addr {
	struct dmi_entry header;
	unsigned start_addr;
	unsigned end_addr;
	unsigned short dev_handle;
	unsigned short memarray_handle;
	unsigned char row;	
	unsigned char interleave_pos;
	unsigned char interleave_depth;
} __attribute__((packed)); 

struct dmi_memdev {
	struct dmi_entry header;
	unsigned short array_handle;
	unsigned short memerr_handle;
	unsigned short total_width;
	unsigned short data_width;
	unsigned short size;
	unsigned char form_factor;
	unsigned char device_set;
	unsigned char device_locator;
	unsigned char bank_locator;
	unsigned char memory_type;
	unsigned short type_details;
	unsigned short speed;
	unsigned char manufacturer;
	unsigned char serial_number;
	unsigned char asset_tag;
	unsigned char part_number;	
} __attribute__((packed));

static struct dmi_entry *entries;
static int numentries;
static int dmi_length;
static struct dmi_entry *handle_to_entry[0xffff];

static unsigned checksum(unsigned char *s, int len)
{
	unsigned char csum = 0;
	int i;
	for (i = 0; i < len; i++)
		csum += s[i];
	return csum;
}

/* Check if entry is valid */
static int check_entry(struct dmi_entry *e, struct dmi_entry **next) 
{
	char *end = (char *)entries + dmi_length;
	char *s = (char *)e + e->length;
	if (!e)
		return 0;
	if (verbose >= 2)
		printf("entry %lx length %d handle %x\n", (char*)e - (char*)entries, 
			e->length, e->handle);
	do {
		if (verbose >= 2) 
			printf("string %s\n", s);
		while (s < end-1 && *s)
			s++;
		if (s >= end-1)
			return 0;
		s++;
	} while (*s);
	if (s >= end) 
		*next = NULL;
	else
		*next = (struct dmi_entry *)(s + 1);
	return 1;
}

/* Relies on sanity checks in check_entry */
static char *getstring(struct dmi_entry *e, unsigned number)
{
	char *s = (char *)e + e->length;
	if (number == 0) 
		return "<unknown>";
	do { 
		if (--number == 0) 
			return s;
		while (*s)
			s++;
		s++;
	} while (*s);
	return NULL;
}

static void fill_handles(void)
{
	int i;
	struct dmi_entry *e, *next;
	e = entries;
	for (i = 0; i < numentries; i++, e = next) { 
		if (!check_entry(e, &next))
			break;
		handle_to_entry[e->handle] = e; 
	}
}

#define round_up(x,y) (((x) + (y) - 1) & ~((y)-1))
#define round_down(x,y) ((x) & ~((y)-1))

int opendmi(void)
{
	int pagesize = getpagesize();
	int memfd; 
	memfd = open("/dev/mem", O_RDONLY);
	if (memfd < 0) { 
		fprintf(stderr, "Cannot open /dev/mem for DMI decoding: %s\n",
			strerror(errno));
		return -1;
	}	

	struct anchor *a, *abase;
	abase = mmap(NULL, 0xffff, PROT_READ, MAP_SHARED, memfd, 0xf0000); 
	if (abase == (struct anchor *)-1) {
		fprintf(stderr, "Cannot mmap 0xf0000: %s\n", strerror(errno));
		return -1;
	}   

	a = abase;
	for (a = abase;;a += 4) { 
		a = memmem(a, 0xffff, "_SM_", 4);
		if (!a) {
			fprintf(stderr, "Cannot find SMBIOS DMI tables\n");
			return -1;
		}
		if (checksum((unsigned char *)a, a->entry_length) == 0) 
			break;
	}
	if (verbose) 
		printf("DMI tables at %x, %u bytes, %u entries\n", 
			a->table, a->length, a->numentries);
	unsigned corr = a->table - round_down(a->table, pagesize); 
 	entries = mmap(NULL, round_up(a->length + pagesize, pagesize), 
		       	PROT_READ, MAP_SHARED, memfd, 
			round_down(a->table, pagesize));
	if (entries == (struct dmi_entry *)-1) { 
		fprintf(stderr, "Cannot mmap SMBIOS tables at %x\n", a->table);
		return -1;
	}
	entries = (struct dmi_entry *)(((char *)entries) + corr);
	numentries = a->numentries;
	dmi_length = a->length;
	fill_handles();
	return 0;	
}

static unsigned dimm_size(unsigned short size, char *unit)
{
	unsigned mbflag = !(size & (1<<15));
	size &= ~(1<<15);
	strcpy(unit, "KB");
	if (mbflag) {
		unit[0] = 'M';
		if (size >= 1024) {
			unit[0] = 'G';
			size /= 1024;
		}
	}
	return size;
}

static char *form_factors[] = { 
	"?",
	"Other", "Unknown", "SIMM", "SIP", "Chip", "DIP", "ZIP", 
	"Proprietary Card", "DIMM", "TSOP", "Row of chips", "RIMM",
	"SODIMM", "SRIMM"
};
static char *memory_types[] = {
	"?",
	"Other", "Unknown", "DRAM", "EDRAM", "VRAM", "SRAM", "RAM",
	"ROM", "FLASH", "EEPROM", "FEPROM", "EPROM", "CDRAM", "3DRAM",
	"SDRAM", "SGRAM", "RDRAM", "DDR", "DDR2"
};

#define LOOKUP(array, val, buf) ((val) >= NELE(array) ? (sprintf(buf,"<%u>",(val)), (buf)) : (array)[val])

static char *type_details[16] = {
	"Reserved", "Other", "Unknown", "Fast-paged", "Static Column",
	"Pseudo static", "RAMBUS", "Synchronous", "CMOS", "EDO",
	"Window DRAM", "Cache DRAM", "Non-volatile", "Res13", "Res14", "Res15"
}; 

static void dump_type_details(unsigned short td)
{
	int i;
	if (!td)
		return;
	for (i = 0; i < 16; i++) 
		if (td & (1<<i))
			Wprintf("%s ", type_details[i]);
}

void dump_memdev(unsigned handle, unsigned long addr)
{
	char tmp[20];
	struct dmi_memdev *md = (struct dmi_memdev *)handle_to_entry[handle];
	if (!md) { 
		Wprintf("Cannot resolve memory device %x for %lx\n", 
			handle, addr);
		return;
	}
	if (md->header.length < 
			offsetof(struct dmi_memdev, manufacturer)) { 
		Wprintf("Memory device %x for address %lx too short %hu expected %lu\n",
			handle, addr, md->header.length, 
			sizeof(struct dmi_memdev));
		return;
	}	
	char unit[10];
	Wprintf("%s ", LOOKUP(memory_types, md->memory_type, tmp));
	if (md->form_factor >= 3) 
		Wprintf("%s ", LOOKUP(form_factors, md->form_factor, tmp));
	if (md->speed != 0)
		Wprintf("%hu Mhz ", md->speed);
	dump_type_details(md->type_details);
	Wprintf("Width %hu Data Width %hu Size %u %s\n",
		md->total_width, md->data_width, 
		dimm_size(md->size, unit), unit);

	char *s;
#define DUMPSTR(n,x) \
	if (md->x) { \
		s = getstring(&md->header, md->x);	\
		if (s && strcmp(s,"None"))		\
			Wprintf(n ": %s\n", s);		\
	}
	DUMPSTR("Device Locator", device_locator);
	DUMPSTR("Bank Locator", bank_locator);
	if (md->header.length < offsetof(struct dmi_memdev, manufacturer))
		return;
	DUMPSTR("Manufacturer", manufacturer);
	DUMPSTR("Serial Number", serial_number);
	DUMPSTR("Asset Tag", asset_tag);
	DUMPSTR("Part Number", part_number);
}

static void warnuser(void)
{
	static int warned; 
	if (warned) 
		return;
	warned = 1;
	Wprintf("WARNING: SMBIOS data is often unreliable. Take with a grain of salt!\n");
}

int dmi_decodeaddr(unsigned long addr) 
{ 
	int i, found, matching;
	struct dmi_entry *e, *next;
	e = entries;
	found = 0;
	matching = 0;
	Wprintf("Resolving address %lx using SMBIOS\n", addr);
	for (i = 0; i < numentries; i++, e = next) {
		if (!check_entry(e, &next))
			break; 
		if (verbose)
			printf("handle %x type %u length %u\n", 
			       e->handle, e->type, e->length);
		if (e->type != DMI_MEMORY_MAPPED_ADDR)
			continue;
		if (e->length < sizeof(struct dmi_memdev_addr)) {
			printf("too short %hd vs %lu\n", e->length,
				sizeof(struct dmi_memdev_addr));
			continue;
		}
		found++;
		struct dmi_memdev_addr *da = (struct dmi_memdev_addr *)e;
		if (verbose)
			printf("--- %Lx-%Lx\n", 
				((long long)da->start_addr)*1024, 
				((long long)da->end_addr) * 1024);
		if (addr < ((unsigned long long)da->start_addr)*1024 ||
		    addr >= ((unsigned long long)da->end_addr)*1024)
			continue;
		warnuser(); 
		matching++; 
		dump_memdev(da->dev_handle, addr);
		if (0)
		Wprintf("Row %u Interleave Position %u Interleave Depth %u\n",
			da->row, da->interleave_pos, da->interleave_depth);	
	}
	if (!found) 
		Wprintf("No DIMMs found in SMBIOS tables\n");
	else if (!matching) 
		Wprintf(
	     "No matching memory address found for %lx in SMBIOS\n",
	     addr);
	return matching > 0;
} 

#ifdef STANDALONE
int main(int ac, char **av)
{
	if (!av[1]) { 
		printf("%s physaddr [verboselevel]\n", av[0]);
		exit(1);
	}
	if (av[2]) 
		verbose = atoi(av[2]);
	if (opendmi() < 0) 
		exit(1);
	dmi_decodeaddr(strtoul(av[1], NULL, 0));
	return 0;
}
#endif
