/*
madman - a music manager
Copyright (C) 2003  Andreas Kloeckner <ak@ixion.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/




#include <qsettings.h>

#include <stdexcept>

#include "utility/player.h"
#include "utility/plugin.h"
#include "utility/prefs.h"
#include "database/song.h"




namespace
{
  void addAutoDJScoringRule(const QString &comment, const QString &criterion, int score, tAutoDJPreferences &prefs)
  {
    tAutoDJScoringRule adj;
    adj.Comment = comment;
    adj.Criterion = criterion;
    adj.ScoreAdjustment = score;
    prefs.ScoringRuleList.push_back(adj);
  }
}




// tAutoDJPreferences ---------------------------------------------------------
void tAutoDJPreferences::resetToDefault()
{
  ScoringRuleList.clear();
  addAutoDJScoringRule("Baseline score", "~all", 5, *this);
  addAutoDJScoringRule("Unrated", "~unrated", 5, *this);
  addAutoDJScoringRule("Rated '*****'", "~rating(5)", 15, *this);
  addAutoDJScoringRule("Rated '****'", "~rating(4)", 12, *this);
  addAutoDJScoringRule("Rated '***'", "~rating(3)", 8, *this);
  addAutoDJScoringRule("Rated '**'", "~rating(2)", 5, *this);
  addAutoDJScoringRule("Rated '*'", "~rating(1)", 3, *this);
  addAutoDJScoringRule("Rated '-'", "~rating(0)", -10, *this);
  addAutoDJScoringRule("Unplayed", "~play_count(0)", 4, *this);
  addAutoDJScoringRule("Played within an hour", "~last_played_n_days_ago(<=0.04166)", -10, *this);
  addAutoDJScoringRule("Played within 24 hours", "~last_played_n_days_ago(<=1)&~last_played_n_days_ago(>0.04166)", -5, *this);
  addAutoDJScoringRule("Played within 3 days", "~last_played_n_days_ago(<=3)&~last_played_n_days_ago(>1)", -3, *this);
  addAutoDJScoringRule("Played within a week", "~last_played_n_days_ago(<=7)&~last_played_n_days_ago(>3)", -1, *this);
  addAutoDJScoringRule("Not played for a month", "~last_played_n_days_ago(>=30)&~last_played_n_days_ago(<60)", 3, *this);
  addAutoDJScoringRule("Not played for two months", "!~last_played_n_days_ago(>60)", 6, *this);
  addAutoDJScoringRule("Often played to the end", "~play_count(>=4)&~full_play_ratio(>0.5)", 5, *this);
  addAutoDJScoringRule("Rarely played to the end", "~play_count(>=4)&~full_play_ratio(<0.3)", -5, *this);
  addAutoDJScoringRule("Played more than three times", "~play_count(>=3)", 3, *this);
  addAutoDJScoringRule("Less than a week old", "~existed_for_days(<= 7)", 10, *this);
  addAutoDJScoringRule("Less than two weeks old", "~existed_for_days(<= 14)", 5, *this);
  addAutoDJScoringRule("Less than three weeks old", "~existed_for_days(<= 21)", 3, *this);
}




// tPreferences ---------------------------------------------------------------
void tPreferences::save(QSettings &settings)
{
  settings.writeEntry("/madman/player", Player.name());

#define WRITE_BOOL(SETTING, VARIABLE) \
  settings.writeEntry("/madman/" SETTING, VARIABLE ? 1 : 0);

  QStringList plugin_dirs = stringvector2QStringList(PluginDirectories);
  settings.writeEntry("/madman/plugin_directories", plugin_dirs);
  WRITE_BOOL("remember_geometry", RememberGeometry);
  WRITE_BOOL("scan_at_startup", ScanAtStartup);
  WRITE_BOOL("collect_history", CollectHistory);
  WRITE_BOOL("enable_tray_icon", EnableSystemTrayIcon);
  WRITE_BOOL("enable_system_tray_icon", EnableSystemTrayIcon);
  WRITE_BOOL("minimize_to_system_tray", MinimizeToSystemTray);
  WRITE_BOOL("enable_passive_popup_song_announcements", EnablePassivePopupSongAnnouncements);

  settings.writeEntry("/madman/number_of_backups_kept", BackupCount);

  WRITE_BOOL("http_daemon_enabled", HttpDaemonEnabled);
  settings.writeEntry("/madman/http_server_port", HttpDaemonPort);
  WRITE_BOOL("http_downloads_enabled", HttpDownloadsEnabled);
  WRITE_BOOL("http_scripting_enabled", HttpScriptingEnabled);
  WRITE_BOOL("http_write_scripting_enabled", HttpWriteScriptingEnabled);
  WRITE_BOOL("http_local_play_enabled", HttpLocalPlayEnabled);
  WRITE_BOOL("http_browsing_enabled", HttpBrowsingEnabled);
  WRITE_BOOL("http_restrict_to_localhost", HttpRestrictToLocalhost);

  settings.writeEntry("/madman/tray_tooltip_format", TrayTooltipFormat );
  settings.writeEntry("/madman/passive_popup_format", PassivePopupFormat);

  unsigned i = 0;
  settings.writeEntry("/madman/auto_dj/score_adjustment_count", (int) AutoDJPreferences.ScoringRuleList.size());
  FOREACH_CONST(first, AutoDJPreferences.ScoringRuleList, tAutoDJScoringRuleList)
  {
    const tAutoDJScoringRule &adj(*first);
    settings.writeEntry(QString("/madman/auto_dj/score_adjustment.%1.comment").arg(i), 
	adj.Comment);
    settings.writeEntry(QString("/madman/auto_dj/score_adjustment.%1.criterion").arg(i), 
	adj.Criterion);
    settings.writeEntry(QString("/madman/auto_dj/score_adjustment.%1.score").arg(i), 
	adj.ScoreAdjustment);
    i++;
  }
}




void tPreferences::load(QSettings &settings)
{
  QString player_name = settings.readEntry("/madman/player");
  
  tPlayer * new_player = createPlayer(player_name);
  if (!new_player)
  {
    vector<QString> players;
    listPlayers(players);
    if (players.size())
      new_player = createPlayer(players[0]);
    if (!new_player)
      throw runtime_error("can't instantiate any player, giving up.");
  }
  setPlayer(new_player);

  bool ok;
  PluginDirectories = QStringList2stringvector(settings.readListEntry("/madman/plugin_directories", &ok));
  if (!ok)
  {
    PluginDirectories.clear();
    PluginDirectories.push_back(STRINGIFY(MADMAN_LIBDIR) "/madman/plugins");
  }
#define READ_BOOL(SETTING, VARIABLE, DEFAULT) \
  VARIABLE = settings.readNumEntry("/madman/" SETTING, DEFAULT);

  READ_BOOL("remember_geometry", RememberGeometry, 1);
  READ_BOOL("scan_at_startup", ScanAtStartup, 0);
  READ_BOOL("collect_history", CollectHistory, 1);
  READ_BOOL("enable_system_tray_icon", EnableSystemTrayIcon, 1);
  READ_BOOL("minimize_to_system_tray", MinimizeToSystemTray, 0);
  READ_BOOL("enable_passive_popup_song_announcements", EnablePassivePopupSongAnnouncements, 1);

  BackupCount = settings.readNumEntry("/madman/number_of_backups_kept", 4);

  READ_BOOL("http_daemon_enabled", HttpDaemonEnabled, 0);
  HttpDaemonPort = settings.readNumEntry("/madman/http_server_port", 51533);
  READ_BOOL("http_downloads_enabled", HttpDownloadsEnabled, 1);
  READ_BOOL("http_scripting_enabled", HttpScriptingEnabled, 1);
  READ_BOOL("http_write_scripting_enabled", HttpWriteScriptingEnabled, 1);
  READ_BOOL("http_local_play_enabled", HttpLocalPlayEnabled, 1);
  READ_BOOL("http_browsing_enabled", HttpBrowsingEnabled, 1);
  READ_BOOL("http_restrict_to_localhost", HttpRestrictToLocalhost, 1);

  TrayTooltipFormat = settings.readEntry("/madman/tray_tooltip_format", "madman: %artist% - %title%");
  PassivePopupFormat = settings.readEntry("/madman/passive_popup_format", "%artist%%newline%%title%");

  unsigned criteria_count = 
    settings.readNumEntry("/madman/auto_dj/score_adjustment_count", 0);
  if (criteria_count == 0)
    AutoDJPreferences.resetToDefault();
  else
  {
    for (unsigned i = 0; i < criteria_count; i++)
    {
      tAutoDJScoringRule adj;
      adj.Comment = settings.readEntry(
	  QString("/madman/auto_dj/score_adjustment.%1.comment").arg(i), "");
      adj.Criterion = settings.readEntry(
	  QString("/madman/auto_dj/score_adjustment.%1.criterion").arg(i), "");
      adj.ScoreAdjustment = settings.readNumEntry(
	  QString("/madman/auto_dj/score_adjustment.%1.score").arg(i), 0);

      if (adj.ScoreAdjustment < adj.ScorePlayNever)
	adj.ScoreAdjustment = adj.ScorePlayNever;
      if (adj.ScoreAdjustment > adj.ScoreMaximum)
	adj.ScoreAdjustment = adj.ScoreMaximum;
      AutoDJPreferences.ScoringRuleList.push_back(adj);
    }
  }
}




void tPreferences::setPlayer(tPlayer *newplayer)
{
  Player.setBackend(newplayer);
}
