/*
madman - a music manager
Copyright (C) 2003  Andreas Kloeckner <ak@ixion.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/




#include "utility/player.h"
#include "database/database.h"
#include "database/song_list_tools.h"
#include "ui/overview.h"
#include "ui/mainwin.h"
#include <qpopupmenu.h>
#include <qlistview.h>
#include <algorithm>




// private helpers ------------------------------------------------------------
namespace
{
  QString normalize(const QString &victim)
  {
    return victim.lower();
  }
}




// tOverviewItem --------------------------------------------------------------
tOverviewItem::tOverviewItem(QListView *view,  QListViewItem *after, const QString &label, const QString &criterion)
: QListViewItem(view, after, label), Criterion(criterion)
{ 
}




tOverviewItem::tOverviewItem(QListViewItem *parent, QListViewItem *after, const QString &label, const QString &criterion)
: QListViewItem(parent, after, label), Criterion(criterion)
{
}





// tExpandableOverviewItem --------------------------------------------------------------
tExpandableOverviewItem::tExpandableOverviewItem(QListView *view,  QListViewItem *after, const QString &label, const QString &criterion)
: tOverviewItem(view, after, label, criterion)
{ 
  setExpandable(true);
  HasExpanded = false;
}




tExpandableOverviewItem::tExpandableOverviewItem(QListViewItem *parent, QListViewItem *after, const QString &label, const QString &criterion)
: tOverviewItem(parent, after, label, criterion)
{
  setExpandable(true);
  HasExpanded = false;
}




void tExpandableOverviewItem::setOpen(bool o)
{
  if (!HasExpanded && o)
  {
    listView()->setUpdatesEnabled(false);
    addExpansion();
    listView()->setUpdatesEnabled(true);
    HasExpanded = true;
  }
  tOverviewItem::setOpen(o);
}




void tExpandableOverviewItem::addExpansion()
{
}




// tool routine ---------------------------------------------------------------
namespace
{
  int buildOverview(tSongList &songs, tSongField field1, tSongField field2, QListViewItem *parent, const QString & critprefix = "")
  {
    sort(songs, field1);

    QListViewItem *last_item = NULL;
    QString last_entry = QString::null;

    int distinct_count = 0;

    tSongList::iterator first = songs.begin(), last = songs.end();
    while (first != last)
    {
      QString this_nonnormalized_entry = (*first)->fieldText(field1);
      QString this_entry = normalize(this_nonnormalized_entry);
      if (this_nonnormalized_entry == "")
	this_nonnormalized_entry = "<none>";

      tSongList this_songs;
      while (first != last && this_entry == normalize((*first)->fieldText(field1)))
	this_songs.push_back(*first++);

      QString criterion = QString("%1~%2(complete:%3)")
	.arg(critprefix)
	.arg(getFieldIdentifier (field1)) 
	.arg(quoteString(this_entry));

      QListViewItem *entry_item = new tOverviewItem(
	  parent, last_item, 
	  QString("%1 (%2)")
	    .arg(this_nonnormalized_entry)
	    .arg(this_songs.size()), 
	    criterion);

      if (field2 != FIELD_COUNT)
	buildOverview(this_songs, field2, FIELD_COUNT, entry_item, criterion+"&");

      last_item = entry_item;
      distinct_count++;
    }
    return distinct_count;
  }
}




// tArtistsOverviewItem -------------------------------------------------------
tArtistsOverviewItem::tArtistsOverviewItem(tDatabase &database, QListView *view,  QListViewItem *after, const QString &label, const QString &criterion)
: tExpandableOverviewItem(view, after, label, criterion), 
  Database(database)
{ 
}




tArtistsOverviewItem::tArtistsOverviewItem(tDatabase &database, QListViewItem *parent, QListViewItem *after, const QString &label, const QString &criterion)
: tExpandableOverviewItem(parent, after, label, criterion),
  Database(database)
{ 
}




void tArtistsOverviewItem::addExpansion()
{
  tSongList songs;
  FOREACH_CONST(first, Database.SongCollection, tSongCollection)
    songs.push_back(*first);
  buildOverview(songs, FIELD_ARTIST, FIELD_ALBUM, this);
}




// tAlbumOverviewItem -------------------------------------------------------
tAlbumOverviewItem::tAlbumOverviewItem(tDatabase &database, QListView *view,  QListViewItem *after, const QString &label, const QString &criterion)
: tExpandableOverviewItem(view, after, label, criterion), 
  Database(database)
{ 
}




tAlbumOverviewItem::tAlbumOverviewItem(tDatabase &database, QListViewItem *parent, QListViewItem *after, const QString &label, const QString &criterion)
: tExpandableOverviewItem(parent, after, label, criterion),
  Database(database)
{ 
}




void tAlbumOverviewItem::addExpansion()
{
  tSongList songs;
  FOREACH_CONST(first, Database.SongCollection, tSongCollection)
    songs.push_back(*first);
  buildOverview(songs, FIELD_ALBUM, FIELD_COUNT, this);
}





// tGenreOverviewItem -------------------------------------------------------
tGenreOverviewItem::tGenreOverviewItem(tDatabase &database, QListView *view,  QListViewItem *after, const QString &label, const QString &criterion)
: tExpandableOverviewItem(view, after, label, criterion), 
  Database(database)
{ 
}




tGenreOverviewItem::tGenreOverviewItem(tDatabase &database, QListViewItem *parent, QListViewItem *after, const QString &label, const QString &criterion)
: tExpandableOverviewItem(parent, after, label, criterion),
  Database(database)
{ 
}




void tGenreOverviewItem::addExpansion()
{
  tSongList songs;
  FOREACH_CONST(first, Database.SongCollection, tSongCollection)
    songs.push_back(*first);
  buildOverview(songs, FIELD_GENRE, FIELD_COUNT, this);
}




// tOverviewManager -----------------------------------------------------------
tOverviewManager::tOverviewManager(QListView &listview, tMainWindow *mw)
  : ListView(listview), MainWindow(mw)
{
  connect(&ListView, SIGNAL(contextMenuRequested(QListViewItem *, const QPoint &, int)),
           this, SLOT(contextMenuRequested(QListViewItem *, const QPoint &, int)));
  Menu = buildPopupMenu();
}




QPopupMenu *tOverviewManager::buildPopupMenu()
{
  QPopupMenu *menu = new QPopupMenu();

  QPopupMenu *play_menu = new QPopupMenu(&ListView);
  play_menu->insertItem(tr("&Play now"), this, SLOT(slotPlayNow()));
  play_menu->insertItem(tr("Play &next"), this, SLOT(slotPlayNext()));
  play_menu->insertItem(tr("Play &evenutally"), this, SLOT(slotPlayEventually()));
  menu->insertItem(tr("&Play"), play_menu);

  return menu;
}




void tOverviewManager::contextMenuRequested (QListViewItem * item, const QPoint & pos, int col)
{
  Menu->popup(pos);
}




void tOverviewManager::slotPlayNow()
{
  tOverviewItem *my_item = dynamic_cast<tOverviewItem *>(ListView.selectedItem());
  if (my_item)
  {
    tPlaylist set;
    set.setSongCollection(&MainWindow->database().SongCollection);
    set.setCriterion(my_item->criterion());
    set.reevaluateCriterion();

    tSongList list;
    set.render(list);

    MainWindow->preferences().Player.playNow(list);
  }
}




void tOverviewManager::slotPlayNext()
{
  tOverviewItem *my_item = dynamic_cast<tOverviewItem *>(ListView.selectedItem());
  if (my_item)
  {
    tPlaylist set;
    set.setSongCollection(&MainWindow->database().SongCollection);
    set.setCriterion(my_item->criterion());
    set.reevaluateCriterion();

    tSongList list;
    set.render(list);

    MainWindow->preferences().Player.playNext(list);
  }
}




void tOverviewManager::slotPlayEventually()
{
  tOverviewItem *my_item = dynamic_cast<tOverviewItem *>(ListView.selectedItem());
  if (my_item)
  {
    tPlaylist set;
    set.setSongCollection(&MainWindow->database().SongCollection);
    set.setCriterion(my_item->criterion());
    set.reevaluateCriterion();

    tSongList list;
    set.render(list);

    MainWindow->preferences().Player.playEventually(list);
  }
}




