// -*- C++ -*-
/**
 * \file lyxfunc.h
 * This file is part of LyX, the document processor.
 * Licence details can be found in the file COPYING.
 *
 * \author Lars Gullik Bjnnes
 * \author Jean-Marc Lasgouttes
 * \author John Levon
 * \author Andr Pnitz
 *
 * Full author contact details are available in file CREDITS.
 */

#ifndef LYXFUNC_H
#define LYXFUNC_H

#include "kbsequence.h"
#include "lfuns.h"

#include <boost/shared_ptr.hpp>
#include <boost/signals/trackable.hpp>


class BufferView;
class FuncRequest;
class FuncStatus;
class LyXKeySym;
class LyXText;
class LyXView;


/** This class encapsulates all the LyX command operations.
    This is the class of the LyX's "high level event handler".
    Every user command is processed here, either invocated from
    keyboard or from the GUI. All GUI objects, including buttons and
    menus should use this class and never call kernel functions directly.
*/
class LyXFunc : public boost::signals::trackable {
public:
	///
	explicit LyXFunc(LyXView *);

	/// LyX dispatcher, executes lyx actions.
	void dispatch(FuncRequest const &);

	/// return the status bar state string
	std::string const viewStatusMessage();

	///
	typedef boost::shared_ptr<LyXKeySym> LyXKeySymPtr;
	///
	void processKeySym(LyXKeySymPtr key, key_modifier::state state);

	///
	FuncStatus getStatus(FuncRequest const & action) const;

	/// The last key was meta
	bool wasMetaKey() const;

	/// True if lyxfunc reports an error
	bool errorStat() const { return errorstat; }
	/// Buffer to store result messages
	void setMessage(std::string const & m) const;
	/// Buffer to store result messages
	void setErrorMessage(std::string const &) const;
	/// Buffer to store result messages
	std::string const getMessage() const { return dispatch_buffer; }
	/// Handle a accented char key sequence
	void handleKeyFunc(kb_action action);

private:
	///
	BufferView * view() const;

	///
	LyXView * owner;

	/// the last character added to the key sequence, in ISO encoded form
	char encoded_last_key;

	///
	kb_sequence keyseq;
	///
	kb_sequence cancel_meta_seq;
	///
	key_modifier::state meta_fake_bit;
	///
	void setupLocalKeymap();
	/// Error status, only Dispatch can change this flag
	mutable bool errorstat;

	/** Buffer to store messages and result data. Is there a
	    good reason to have this one as static in Dispatch? (Ale)
	*/
	mutable std::string dispatch_buffer;

	/// send a post-dispatch status message
	void sendDispatchMessage(std::string const & msg,
		FuncRequest const & ev);

	// I think the following should be moved to BufferView. (Asger)
	///
	void menuNew(std::string const & argument, bool fromTemplate);
	///
	void open(std::string const &);
	///
	void doImport(std::string const &);
	///
	void closeBuffer();
};

#endif
